
;************************************************************************
;*									*
;*									*
;*	EMM/EMS driver program for SUNTAC-62 chip set version B1	*
;*									*
;*	               sample program - 10/20/1987	rev. 1.02	*
;*									*
;************************************************************************

MAX_HNDL	EQU	255	;max handle size
PAGE_MAX	EQU	216	;max logical page size
BACK_MAX	EQU	40	;max mapping data backup size
EMSIOA		EQU	0E8H	;EMS i/o port address A
EMSIOB		EQU	98H	;EMS i/o port address B
CHKIOA		EQU	9DH	;EMS i/o port A check data
CHKIOB		EQU	93H	;EMS i/o port B check data
I8042		EQU	60H	;i8042 i/o port address
STATSEG		EQU	0C000H	;defult start phygical page segment address
ROMSEG		EQU	0E000H	;option ROM area start segment address
ENDSEG		EQU	0F000H	;
NOT_USE		EQU	0AA55H	;not used logical page number
CR		EQU	0DH	;CR code
LF		EQU	0AH	;LF code
TAB		EQU	09H	;TAB code
MEMSIZE		EQU	413H	;memory size area address (word) KB
PAGESIZE	EQU	6	;1 page kanri data size
TBLSIZE		EQU	7	;1 table size
BACKSIZE	EQU	TBLSIZE * 4 + 2	;mapping backup area 1 data size
DIS_EMS		EQU	0	;phygical page disable data

;-- EMS REGISTERS ------------------------------------------------------:
;									:
; EMS_R0 : 76543210 (Read/Write)					:
;          |+-----+--- EMS i/o port address A9 - A3 (defult 13H or 1DH)	:
;          +-(Write)-- [98/E8] port read flag (0:not read, 1:read)	:
;          +-(Read)--- Register data transmit flag (0:end, 1:not end)	:
;									:
; EMS_R1 : 76543210 (Reserved)						:
;									:
; EMS_R2 : 76543210 (Write only)					:
;          |    |++--- EMS flag (00:EMS, 11:Non EMS)			:
;          +----+----- system memory size A19 - A14 (A0H = 640KB)	: 
;									:
; EMS_R3 : 76543210 (Write only)					:
;          |    |++--- bank size (00:16KB)				:
;          +----+----- EMS segment address A19 - A14 (C0H = C000H)	:
;									:
; EMS_R4 : 76543210 (Write only)					:
;          +------+--- bank 0 page number (1 - 255)			:
;		       * data 0 = page 0 disable			:
;									:
; EMS_R5 : 76543210 (Write only)					:
;          +------+--- bank 1 page number (1 - 255)			:
;		       * data 0 = page 1 disable			:
;									:
; EMS_R6 : 76543210 (Write only)					:
;          +------+--- bank 2 page number (1 - 255)			:
;		       * data 0 = page 2 disable			:
;									:
; EMS_R7 : 76543210 (Write only)					:
;          +------+--- bank 3 page number (1 - 255)			:
;		       * data 0 = page 3 disable			:
;									:
;-----------------------------------------------------------------------:


code	segment
	assume	cs:code,ds:code
	public	emmdrv,emmstat,emmint,parofs,parseg,int67,rpage
	public	maptbl,bkmap,emminit

	org	0000h

	; DEVICE header block
emmdrv	DW	-1,-1		;Link to next device
	DW	8000H		;
	DW	offset emmstat	;
	DW	offset emmint	;
	DB	'EMMXXXX0'	;

ptrsav	label	dword
parofs	DW	0		;
parseg	DW	0		;

emmseg	DW	0C000h		;phygical page segment address
total	DW	PAGE_MAX	;total logical page count
unalloc	DW	PAGE_MAX	;unallocate logical page count
hndlcnt	DW	0		;EMS handle used count
emmsts	DB	0		;EMS driver install status
emmver	DW	32h		;EMS version 3.2
dssav	DW	?		;DS save area
fjmp	DW	?		;function near jump address

;
;	Define offsets for io data packet
;
iodat	struc
	cmdlen	DB	?	;LENGTH OF THIS COMMAND
	unit	DB	?	;SUB UNIT SPECIFIER
	cmd	DB	?	;COMMAND CODE
	status	DW	?	;STATUS
		DB	8 dup (?)
	media	DB	?	;MEDIA DESCRIPTOR
	trans	DD	?	;TRANSFER ADDRESS
	count	DW	?	;COUNT OF BLOCKS OR CHARACTERS
	start	DW	?	;FIRST BLOCK TO TRANSFER
iodat	ends

;
;	Define offsets for io data packet 2
;
iodat2	struc
		DB	13 dup (?)
		DB	?
	brkoff	DW	?	;BREAK ADDRESS (OFFSET)
	brkseg	DW	?	;BREAK ADDRESS (SEGMENT)
iodat2	ends


;
; Simplistic Strategy routine for non-multi-Tasking system.
;
;	Currently just saves I/O packet pointers in PTRSAV for
;	later processing by the individual interrupt routines.
;
emmstat proc	far
	mov	cs:parofs,bx	;
	mov	cs:parseg,es	;
	ret			;
emmstat endp


;
; Common program for handling the simplistic I/O packet
;	processing scheme in MSDOS 2.0
;
emmint	proc	far
	push	si
	push	ax		;Save all nessacary registers.
	push	cx
	push	dx
	push	di
	push	bp
	push	ds
	push	es
	push	bx
	lds	bx,cs:[ptrsav]	;Retrieve pointer to I/O Packet.
	mov	cx,[bx.count]	;cx = Contains byte/sector count.
	mov	AL,[bx.cmd]	;Retrieve Command type. (1 => 11)
	cbw			;
	mov	si,offset emstbl;
	add	si,ax		;Compute entry pointer in dispatch table.
	add	si,ax		;
	cmp	al,11		;Verify that not more than 11 commands.
	ja	cmderr		;Ah, well, error out.
	les	di,[bx.trans]	;di contains addess of Transfer address.
	push	cs
	pop	ds		;Data segment same as Code segment.
	jmp	[si]		;Perform I/O packet command.

bus_exit:			;Device busy exit.
	mov	ah,00000011b	;Set busy and done bits.
	jmp	short exit1

cmderr:	mov	al,3		;Set unknown command error #.

;
;	Common error processing routine.
;	AL contains actual error code.
;
;	Error # 0 = Write Protect violation.
;		1 = Unkown unit.
;		2 = Drive not ready.
;		3 = Unknown command in I/O packet.
;		4 = CRC error.
;		5 = Bad drive request structure length.
;		6 = Seek error.
;		7 = Unknown media discovered.
;		8 = Sector not found.
;		9 = Printer out of paper.
;		10 = Write fault.
;		11 = Read fault.
;		12 = General failure.
;

err_exit:
	mov	ah,10000001b	;Set error and done bits.
	stc			;Set carry bit also.
	jmp	short exit1	;Quick way out.

exit:	mov	ah,00000001b	;Set done bit for MSDOS.
exit1:	lds	bx,cs:[ptrsav]
	mov	[bx.status],ax	;Save operation compete and status.
	pop	bx		;Restore registers.
	pop	es
	pop	ds
	pop	bp
	pop	di
	pop	dx
	pop	cx
	pop	ax
	pop	si
	ret
emmint	endp

;
;
;
emstbl:	DW	emminit	;0  - Init. (Not used)
	DW	emmchk	;1  - Media check (Not used)
	DW	emmchk	;2  - Get Bios Parameter Block (Not used)
	DW	emmchk	;3  - Reserved. (Currently returns error)
	DW	exit	;4  - Character read. (Destructive)
	DW	exit	;5  - Character read. (Non-destructive)
	DW	exit	;6  - Return status. (Not used)
	DW	exit	;7  - Flush Input buffer.
	DW	exit	;8  - Character write.
	DW	exit	;9  - Character write with Verify.
	DW	exit	;10 - Character write status.
	DW	exit	;11 - Flush output buffer. (Not used.)
	DW	exit	;12 - IO Control.

;
;	EMS driver install check routine
;
emmchk:
	cmp	[emmsts],1	;
	jnz	err_exit	;
	jmp	exit		;

;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

;
;	int 67h EMS driver main routine
;
int67	proc	far
	push	ds
	push	es
	push	bp
	push	di
	push	si
	push	dx
	push	cx
	push	bx
	push	ax
	mov	bp,sp

	cld

	push	ax		;Save DS register
	mov	ax,ds		;
	mov	cs:[dssav],ax	;
	pop	ax		;

	push	cs
	pop	ds

	assume	ds:code

	cmp	ah,40h		;--- function code check ---
	jb	err84		;
	cmp	ah,50h		;
	jnb	err84		;---------------------------
	push	ax		;
	xchg	ah,al		;
	xor	ah,ah		;
	sub	ax,40h		;
	shl	ax,1		;
	mov	bx,offset table	;
	add	bx,ax		;
	pop	ax		;
	jmp	[bx]		;---- jump functions -----

noerr:				;normal return
	pop	ax		;
	xor	ah,ah		;
ngexit:				;
	pop	bx		;
	pop	cx
	pop	dx
	pop	si
	pop	di
	pop	bp
	pop	es
	pop	ds
	iret
int67	endp

;
;	register back area struct
;
retreg	struc
	axsv	DW	?	;AX save area
	bxsv	DW	?	;BX save area
	cxsv	DW	?	;CX save area
	dxsv	DW	?	;DX save area
retreg	ends

;
;
;	EMS driver command function jump table
;		(40H - 4FH)
;
table	label	word
	DW	offset noerr	;function 1
	DW	offset func2	;function 2
	DW	offset func3	;function 3
	DW	offset func4	;function 4
	DW	offset func5	;function 5
	DW	offset func6	;function 6
	DW	offset func7	;function 7
	DW	offset func8	;function 8
	DW	offset func9	;function 9
	DW	offset func10	;function 10
	DW	offset func11	;function 11
	DW	offset func12	;function 12
	DW	offset func13	;function 13
	DW	offset func14	;function 14
	DW	offset func15	;function 15

;
;	error status set syori
;

;The manager detected a malfunction in the EMM software.
err80:
	pop	ax
	mov	ah,80h
	jmp	ngexit

;The manager detected a malfunction in the expanded memory hardware.
err81:
	pop	ax
	mov	ah,81h
	jmp	ngexit

;The EMM couldn't find the EMM handle your program specified.
err83:
	pop	ax
	mov	ah,83h
	jmp	ngexit

;The function code passed to the EMM is not defined.
err84:
	pop	ax
	mov	ah,84h
	jmp	ngexit

;All EMM handles are being used.
err85:
	pop	ax
	mov	ah,85h
	jmp	ngexit

;The EMM detected a "save" or "restore" page mapping context error.
err86:
	pop	ax
	mov	ah,86h
	jmp	ngexit

;There aren't enough expanded memory pages to satisfy your program's request.
err87:
	pop	ax
	mov	ah,87h
	jmp	ngexit

;There aren't enough unallocated pages to satisfy your program's request.
err88:
	pop	ax
	mov	ah,88h
	jmp	ngexit

;Can't allocate zero (0) pages.
err89:
	pop	ax
	mov	ah,89h
	jmp	ngexit

;The logical page is out of the range of logical pages which are allocated to
;the EMM handle.
err8a:
	pop	ax
	mov	ah,8ah
	jmp	ngexit

;The physical page to which the logical page is mapped is out of the range of
;physical pages.
err8b:
	pop	ax
	mov	ah,8bh
	jmp	ngexit

;The page mapping hardware state save area is full.
err8c:
	pop	ax
	mov	ah,8ch
	jmp	ngexit

;The page mapping hardware state save area already has a state associated with
;the EMM handle.
err8d:
	pop	ax
	mov	ah,8dh
	jmp	ngexit

;The page mapping hardware state save area doesn't have a state associated with
;the EMM handle.
err8e:
	pop	ax
	mov	ah,8eh
	jmp	ngexit

;The subfunction parameter passed to the function isn't defined.
err8f:
	pop	ax
	mov	ah,8fh
	jmp	ngexit

;------ function 2 --------------------------------------------------
;
; Get page frame address
;
; output
;	AH	: status
;	BX	: page segment address
;--------------------------------------------------------------------
func2:
	mov	bx,[emmseg]		;get EMM phygical page segment address
f21:					;
	mov	[bp.bxsv],bx		;
	jmp	noerr			;

;------ function 3 --------------------------------------------------
;
; Get unallocated page count
;
; output
;	AH	: status
;	BX	: unallocate page
;	DX	: all page
;--------------------------------------------------------------------
func3:
	mov	si,offset rpage		;
	xor	bx,bx			;
	mov	dx,bx			;
	mov	cx,PAGE_MAX		;
f33:					;
	cmp	[si],0ffffh		;logical page end ?
	jz	f31			;
	inc	dx			;
	cmp	[si],NOT_USE		;unallocated page ?
	jnz	f32			;
	inc	bx			;
f32:					;
	add	si,PAGESIZE		;
	loop	f33			;
f31:					;
	mov	[total],dx		;Save total page count
	mov	[unalloc],bx		;Save unallocated page count
	mov	[bp.dxsv],dx		;Save all page count
	jmp	f21			;

;------ function 4 --------------------------------------------------
;
; Allocate pages
;
;	BX	: request allocate page
; output
;	AH	: status
;	DX	: EMM handle
;--------------------------------------------------------------------
func4:
	mov	bx,[bp.bxsv]		;get BX
	or	bx,bx			;request page size 0 ?
	jnz	f41			;no
	jmp	err89			;
f41:					;
	cmp	[total],bx		;request total size over ?
	jnb	f42			;no
	jmp	err87			;
f42:					;
	cmp	[unalloc],bx		;request unallocate size over ?
	jnb	f43			;no
	jmp	err88			;
f43:					;
	xor	dx,dx			;
	mov	si,offset hndlfg	;
	mov	cx,MAX_HNDL		;
f45:					;
	inc	dx			;
	cmp	byte ptr [si],0		;not used EMM handle ?
	jz	f44			;
	inc	si			;
	loop	f45			;
	jmp	err85			;
f44:					;
	inc	[hndlcnt]		;used EMM handle count up
	sub	[unalloc],bx		;unallocated page - bx
	mov	byte ptr [si],1		;EMM handle used flag set
	mov	si,offset rpage		;
	xor	di,di			;
	mov	cx,bx			;
f47:					;
	cmp	[si],NOT_USE		;unallocated page ?
	jz	f46			;
	add	si,PAGESIZE		;
	jmp	f47			;
f46:					;
	mov	[si],dx			;EMM handle set
	mov	[si.l_hpageno],di	;logical page no. set
	inc	di			;
	add	si,PAGESIZE		;
	loop	f47			;
	mov	[bp.dxsv],dx		;return EMM handle
	mov	[bp.bxsv],bx		;allocated pages count set
	jmp	noerr			;

;------ function 5 --------------------------------------------------
;
; Map handle pages
;
;	AL	: phygical page no.
;	BX	: logical page no.
;	DX	: EMM handle
; output
;	AH	: status
;--------------------------------------------------------------------
func5:
	mov	bx,[bp.bxsv]		;get BX
	cmp	al,4			;phygical page no. ok ?
	jb	f51			;yes
	jmp	err8b			;
f51:					;
	xor	di,di			;
	mov	cx,PAGE_MAX		;
	mov	si,offset rpage		;
f55:					;
	cmp	[si],dx			;EMM handle same ?
	jnz	f53			;no
	inc	di			;
	cmp	[si.l_hpageno],bx	;logical page no. same ?
	jz	f54			;yes
f53:					;
	add	si,PAGESIZE		;
	loop	f55			;
f52:					;
	or	di,di			;
	jnz	f56			;
	jmp	err83			;
f56:					;
	jmp	err8a			;
f54:					;
	xor	ah,ah			;
	mov	di,offset maptbl	;
	mov	cl,TBLSIZE		;
	mul	cl			;
	add	di,ax			;
	push	dx			;
	mov	dx,[di.ioadr]		;
	mov	ax,[si.l_lpageno]	;
	add	al,[pageofs]		;
	call	emsout			;
	pop	dx			;
	mov	[di.p_hpageno],bx	;handle page no.
	mov	[di.p_handle],dx	;handle
	mov	[di.p_lpageno],al	;logical page no.
	jmp	noerr			;

;------ function 6 --------------------------------------------------
;
; Deallocate pages
;
;	DX	: EMM handle
; output
;	AH	: status
;--------------------------------------------------------------------
func6:
	mov	si,offset bkmap		;
	mov	cx,BACK_MAX		;
f62:					;
	cmp	[si],dx			;EMM handle same ?
	jnz	f61			;no
	jmp	err86			;
f61:					;
	add	si,BACKSIZE		;
	loop	f62			;
	mov	si,offset rpage		;logical page deallocate
	mov	cx,PAGE_MAX		;
	xor	bx,bx			;
f65:					;
	cmp	[si],0ffffh		;
	jz	f63			;
	cmp	[si],dx			;
	jnz	f64			;
	inc	bx			;
	mov	[si],NOT_USE		;
f64:					;
	add	si,PAGESIZE		;
	loop	f65			;
f63:					;
	or	bx,bx			;
	jnz	f66			;
	jmp	err83			;
f66:					;
	mov	cx,4			;phygical page deallocate
	mov	si,offset maptbl	;
f6a:					;
	cmp	[si.p_handle],dx	;
	jnz	f67			;
	push	dx			;phygical page wo mukou ni suru ---
	mov	dx,[si.ioadr]		;
	mov	al,DIS_EMS		;
	call	emsout			;
	pop	dx			;----------------------------------
	mov	[si.p_handle],0ffffh	;
	mov	[si.p_hpageno],0ffffh	;
	mov	[si.p_lpageno],0ffh	;
f67:					;
	add	si,TBLSIZE		;
f69:					;
	loop	f6a			;
	add	[unalloc],bx		;unallocate page add
	mov	si,offset hndlfg	;handle use flag clear
	dec	dx			;
	add	si,dx			;
	mov	byte ptr [si],0		;
	dec	[hndlcnt]		;not use handle count up
	jmp	noerr			;

;------ function 7 --------------------------------------------------
;
; Get EMM version
;
; output
;	AH	: status
;	AL	: EMM handle
;--------------------------------------------------------------------
func7:
	pop	ax			;
	mov	ax,[emmver]		;get version no.
	jmp	ngexit			;

;------ function 8 --------------------------------------------------
;
; Save page map
;
;	DX	: EMM handle
; output
;	AH	: status
;--------------------------------------------------------------------
func8:
	call	ckhandl			;handle data check
	jnb	f81			;
	jmp	err83			;
f81:					;
	mov	di,offset bkmap		;check backup handle
	mov	cx,BACK_MAX		;
f83:					;
	cmp	[di],dx			;
	jnz	f82			;
	jmp	err8d			;
f82:					;
	add	di,BACKSIZE		;
	loop	f83			;
	cmp	[backcnt],BACK_MAX	;
	jnz	f84			;
	jmp	err8c			;
f84:					;
	mov	di,offset bkmap		;found mapping data save area
	mov	cx,BACK_MAX		;
f86:					;
	cmp	[di],0ffffh		;
	jz	f85			;
	add	di,BACKSIZE		;
	loop	f86			;
f85:					;
	push	ds			;copy mapping data -> [di]
	pop	es			;
	mov	si,offset maptbl	;
	mov	[di],dx			;
	inc	di			;
	inc	di			;
	mov	cx,TBLSIZE * 4		;
	repz	movsb			;
	inc	[backcnt]		;
	jmp	noerr			;

;------ function 9 --------------------------------------------------
;
; Restore page map
;
;	DX	: EMM handle
; output
;	AH	: status
;--------------------------------------------------------------------
func9:
	call	ckhandl			;handle data check
	jnb	f91			;
	jmp	err83			;
f91:					;
	push	ds			;
	pop	es			;
	cmp	[backcnt],0		;
	jz	f92			;
	mov	si,offset bkmap		;
	mov	cx,BACK_MAX		;
f94:					;
	cmp	[si],dx			;
	jz	f93			;
	add	si,BACKSIZE		;
	loop	f94			;
f92:					;
	jmp	err8e			;
f93:					;
	mov	[si],0ffffh		;
	inc	si			;
	inc	si			;
	mov	di,offset maptbl	;disable phygical pages
	mov	dx,[di.ioadr]		;
f96:					;
	cmp	dx,0ffffh		;
	jz	f95			;
	mov	al,DIS_EMS		;
	call	emsout			;
f95:					;
	add	di,TBLSIZE		;
	loop	f96			;
;
	mov	cx,TBLSIZE * 4		;move mapping data
	mov	di,offset maptbl	;
	push	di			;
	repz	movsb			;
	pop	di			;
;
	mov	cx,4			;enable phygical pages
f98:					;
	mov	dx,[di.ioadr]		;
	cmp	dx,0ffffh		;
	jz	f97			;
	mov	al,[di.p_lpageno]	;
	add	al,[pageofs]		;
	call	emsout			;
f97:					;
	add	di,TBLSIZE		;
	loop	f98			;
	dec	[backcnt]		;backup mapping count dec
	jmp	noerr			;

;------ function 10 -------------------------------------------------
;
; Get page mapping register I/O port array
;
;	ES:DI	: buffer address point
; output
;	AH	: status
;	AL	: board count
;--------------------------------------------------------------------
func10:
	mov	si,offset ioport	;
	mov	cx,4 * 4		;
	xor	ax,ax			;
f102:					;
	cmp	[si],0ffffh		;
	jz	f101			;
	movsw				;
	inc	al			;
	loop	f102			;
f101:					;
	shr	al,1			;
	shr	al,1			;
	pop	cx			;
	jmp	ngexit			;

;------ function 11 -------------------------------------------------
;
; Get logical-to-phygical page translation array
;
;	DX	: EMM handle
;	ES:DI	: buffer address point
; output
;	AH	: status code
;	BX	: number of pages allocated EMM handle.
;--------------------------------------------------------------------
func11:
	mov	si,offset rpage		;
	mov	cx,PAGE_MAX		;
	xor	bx,bx			;
f113:					;
	cmp	[si],0ffffh		;
	jz	f111			;
	cmp	[si],dx			;
	jnz	f112			;
	mov	ax,[si.l_lpageno]	;
	add	al,[pageofs]		;
	xchg	ah,al			;
	stosw				;
	inc	bx			;
f112:					;
	add	si,PAGESIZE		;
	loop	f113			;
f111:					;
	or	bx,bx			;
	jnz	f114			;
	jmp	err83			;
f114:					;
	mov	[bp.bxsv],bx		;
	jmp	noerr			;

;------ function 12 -------------------------------------------------
;
; Get EMM handle count
;
; output
;	AH	: status
;	BX	: active EMM handles
;--------------------------------------------------------------------
func12:
	mov	bx,[hndlcnt]		;
f121:					;
	mov	[bp.bxsv],bx		;
	jmp	noerr			;

;------ function 13 -------------------------------------------------
;
; Get EMM handle pages
;
;	DX	: EMM handle
; output
;	AH	: status
;	BX	: pages EMM handle
;--------------------------------------------------------------------
func13:
	call	getpsiz			;
	or	bx,bx			;
	jnz	f121			;
	jmp	err83			;

;------ function 14 -------------------------------------------------
;
; Get all EMM handle pages
;
;	ES:DI	: buffer address point
; output
;	AH	: status
;	BX	: number of active EMM handles
;--------------------------------------------------------------------
func14:
	mov	si,offset hndlfg	;
	mov	cx,MAX_HNDL		;
	xor	bx,bx			;
	xor	dx,dx			;
f142:					;
	push	cx			;
	cmp	byte ptr [si],1		;
	jnz	f141			;
	push	bx			;
	push	cx			;
	push	si			;
	call	getpsiz			;
	mov	es:[di],dx		;
	mov	es:[di+2],bx		;
	add	di,4			;
	pop	si			;
	pop	cx			;
	pop	bx			;
	inc	bx			;
f141:					;
	inc	dx			;
	inc	si			;
	pop	cx			;
	loop	f142			;
	jmp	f121			;

;------ function 15 -------------------------------------------------
;
; Get/set page map
;
;	AL	: request subfunction no.
;	ES:DI	: mapping registers buffer address point
; output
;	AH	: status
;--------------------------------------------------------------------
func15:
	cmp	al,0			;
	jz	f151			;
	cmp	al,1			;
	jz	f152			;
	cmp	al,2			;
	jz	f153			;
	cmp	al,3			;
	jz	f154			;
	jmp	err8f			;
f151:					;Get the contents of the page mapping
	mov	si,offset maptbl	;registers.
	mov	cx,TBLSIZE * 4		;
	repz	movsb			;
	jmp	noerr			;
f152:					;Set the contents of the page mapping
	push	cs			;registers.
	pop	es			;
	push	ds			;
	mov	ax,[dssav]		;
	mov	ds,ax			;
	mov	di,offset maptbl	;
	mov	cx,TBLSIZE * 4		;
	repz	movsb			;
	pop	ds			;
	jmp	noerr			;
f153:					;Get and set the contents of the page
	push	si			;mapping registers.
	mov	si,offset maptbl	;
	mov	cx,TBLSIZE * 4		;
	repz	movsb			;
	pop	si			;
	jmp	f152			;
f154:					;Return the size of the mapping context
	pop	ax			;array for the operating system.
	mov	ax,TBLSIZE * 4		;mapping data size set
	jmp	ngexit			;

;------------------------------------------------------------------------

;--------------------------------------------------------------------
; EMM handle no. check
;
;	DX	: EMM handle
; output
;	CF = 0	: OK
;	CF = 1	: NG
;--------------------------------------------------------------------
ckhandl proc	near
	mov	si,offset rpage		;
	mov	cx,PAGE_MAX		;
ckhand3:				;
	cmp	[si],0ffffh		;
	jz	ckhand1			;
	cmp	[si],dx			;
	jnz	ckhand2			;
	clc				;
	ret				;
ckhand2:				;
	add	si,PAGESIZE		;
	loop	ckhand3			;
ckhand1:				;
	stc				;
	ret				;
ckhandl endp

;--------------------------------------------------------------------
; get handle page count
;
;	DX	: EMM handle
; output
;	BX	: page count
;--------------------------------------------------------------------
getpsiz proc	near
	mov	si,offset rpage		;
	mov	cx,PAGE_MAX		;
	xor	bx,bx			;
getpsiz3:				;
	cmp	[si],0ffffh		;
	jz	getpsiz1		;
	cmp	[si],dx			;
	jnz	getpsiz2		;
	inc	bx			;
getpsiz2:				;
	add	si,PAGESIZE		;
	loop	getpsiz3		;
getpsiz1:				;
	ret				;
getpsiz endp

;--------------------------------------------------------------------
; EMS i/o port out sub
;	DX	: i/o port address
;	AL	: output data
; output
;	CF = 0	: OK
;	CF = 1	: NG
;--------------------------------------------------------------------
emsout	proc near
	push	ax			;
	push	cx			;
	push	dx			;
	out	dx,al			;
	mov	cx,1000			;
	mov	dx,cs:[emsio]		;
emsout2:				;
	in	al,dx			;
	test	al,80h			;check TRS flag
	jz	emsout1			;
	loop	emsout2			;
	stc				;
	jmp	emsout3			;
emsout1:				;
	clc				;
emsout3:				;
	inc	dx			;v1.02 - out 00 -> R1
	xor	al,al			;v1.02
	out	dx,al			;v1.02
	pop	dx			;
	pop	cx			;
	pop	ax			;
	ret				;
emsout	endp

;..............................................................

;
emsio	label	word
ems_io	DW	98h	;EMS i/o port address
pageofs	DB	?	;logical page no. offset data
backcnt	DB	0	;mapping data backup count

;
;	i/o port address buffer
;
ioport	label	word
	DW	4 * 4 dup (-1)

;
;	phygical pages struct
;
p_page	struc
	p_handle	DW	?	;handle
	p_hpageno	DW	?
	ioadr		DW	?	;phygical page i/o port address
	p_lpageno	DB	?
p_page	ends

;
;	phygical page status data area
;
maptbl	label	word
	DB	TBLSIZE * 4 dup (-1)

;
;	handle used flag
;
hndlfg	label	byte
	DB	MAX_HNDL dup (0)

;
;	mapping data backup area
;
bkmap	label	word
	DB	BACKSIZE * BACK_MAX dup (-1)

;
;	logical pages struct
;
l_page	struc
	l_handle	DW	?	;EMM handle
	l_hpageno	DW	?	;
	l_lpageno	dw	?	;logical page no.
l_page	ends

;
;	logical page kanri data
;
rpage	label	word
	DB	PAGESIZE * PAGE_MAX dup (-1)

;********************************************************************

;--------------------------------------------------------------------
;
; EMS driver initilize program
;	v1.02 - three nop delete
;--------------------------------------------------------------------
emminit:
	call	getprm		;get parameter
	call	ckemsio		;EMS i/o check
	jb	errems		;error ?
	call	ramchk		;EMS ram check
	jb	errems		;error ?
	xor	ax,ax		;
	mov	es,ax		;
	mov	si,19ch		;
	mov	word ptr es:[si],offset int67;int67 offset set
	mov	es:[si+2],cs	;int67 segment set

	call	instmsg

	mov	ax,offset emminit;
	add	ax,0fh		;
	mov	cl,4		;
	shr	ax,cl		;
	mov	cx,ax		;
	mov	ax,cs		;
	add	ax,cx		;
	mov	[emmsts],1	;EMM install flag set
	lds	bx,ptrsav	;
	mov	[bx.brkoff],0	;break address offset set
	mov	[bx.brkseg],ax	;break address segment set
	jmp	exit		;
errems:				;
	mov	si,offset notinst;
	call	strdsp		;
	mov	[emmsts],0	;EMM install flag set
	lds	bx,ptrsav	;
	mov	ax,offset emminit;break address offset set
	mov	[bx.brkoff],ax	;break address offset set
	mov	[bx.brkseg],cs	;break address segment set
	jmp	exit		;

;--------------------------------------------------------------------
; CONFIG.SYS parameter get
;--------------------------------------------------------------------
getprm	proc	near
	push	ds		;option data read
	int	12h		;system memory size get
	mov	[sysmem],ax	;
	les	bx,[ptrsav]	;
	mov	di,es:[bx.count];
	mov	ax,es:[bx.start];
	mov	es,ax		;
	xor	cl,cl		;
getpr2:				;
	mov	al,es:[di]	;
	cmp	al,'/'		;
	jnz	getpr3		;
	inc	di		;
	mov	al,es:[di]	;
	cmp	al,'M'		;System memory size
	jnz	getpr1		;
	inc	di		;
	mov	al,es:[di]	;
	cmp	al,':'		;
	jnz	getpr3		;
	inc	di		;
	call	ascbin1		;
	jb	getpr5		;
	mov	[sysmem],ax	;
	jmp	getpr5		;
getpr1:				;
	cmp	al,'P'		;Phygical page segment address
	jnz	getpr3		;
	inc	di		;
	mov	al,es:[di]	;
	cmp	al,':'		;
	jnz	getpr1		;
	inc	di		;
	call	ascbin2		;
	jnb	getpr8		;error ?
	jmp	getpr5		;
getpr8:				;
	mov	[pageseg],ax	;
;
	mov	[emmseg],ax	;
;
	or	cl,2		;phygical page segment auto flag set
	jmp	getpr5		;
getpr3:				;
	cmp	al,'I'		;EMS i/o port address
	jnz	getpr6		;
	inc	di		;
	mov	al,es:[di]	;
	cmp	al,':'		;
	jnz	getpr1		;
	inc	di		;
	call	ascbin2		;
	mov	[rqioadr],ax	;
;
	mov	[emsio],ax	;
;
	or	cl,1		;EMS i/o port address change flag set
	jmp	getpr5		;
getpr6:				;
	cmp	al,CR		;
	jz	getpr4		;
getpr5:				;
	inc	di		;
	jmp	getpr2		;
getpr4:				;
	mov	[sysflg],cl	;system option flag set
	pop	ds		;
	ret			;
getprm	endp

;--------------------------------------------------------------------
; EMS i/o port check
; output
;	cf = 0 : OK
;	cf = 1 : NG
;--------------------------------------------------------------------
ckemsio	proc	near
	mov	dx,EMSIOA	;
	mov	al,CHKIOA	;
	out	dx,al		;
	mov	cx,1000		;
ckems5:				;
	in	al,dx		;
	test	al,80h		;check TRS flag
	jz	ckems6		;
	loop	ckems5		;
	jmp	ckems1		;
ckems6:				;
	mov	bl,CHKIOA	;
	and	bl,7fh		;
	cmp	al,bl		;
	jz	ckems2		;
ckems1:				;
	mov	dx,EMSIOB	;
	mov	al,CHKIOB	;
	out	dx,al		;
ckems2:				;
	mov	cs:[emsio],dx	;i/o port address set
	test	byte ptr cs:[sysflg],1;
	jz	ckems4		;
	mov	ax,cs:[rqioadr]	;
	mov	bx,ax		;EMS i/o port addess change
	mov	dx,cs:[emsio]	;
	mov	cl,3		;
	shr	ax,cl		;
	or	al,80h		;
	call	emsout		;
	mov	cs:[emsio],bx	;
ckems4:				;
	clc			;
	ret			;
ckems3:				;
	stc			;
	ret			;
ckemsio	endp

;--------------------------------------------------------------------
;
; System memory size check
;	v1.02 - three nop delete
;--------------------------------------------------------------------
sysram	proc	near
	push	cs		;
	pop	ds		;
	push	es		;
	xor	ax,ax		;system memory size set
	mov	es,ax		;
	mov	ax,[sysmem]	;
	cmp	es:[MEMSIZE],ax	;memory size same ?
	jz	sysram1		;
	mov	es:[MEMSIZE],ax	;
	int	19h		;reboot...
sysram1:			;
	pop	es		;
	mov	bx,40h		;
	mul	bx		;
	mov	dx,[emsio]	;
	add	dx,2		;
	mov	al,ah		;
	call	emsout		;EMS i/o port data out
	shr	al,1		;
	shr	al,1		;
	mov	[pageofs],al	;page no. offset data
	mov	dx,STATSEG 	;;;v1.01
	ret			;
sysram	endp

;--------------------------------------------------------------------
; Expanded memory check
; output
;	cf = 0 : OK
;	cf = 1 : NG
;--------------------------------------------------------------------
ramchk	proc	near
	push	ax		;
	push	bx		;
	push	cx		;
	push	dx		;
	push	bp		;
	push	cs		;
	pop	ds		;
	mov	si,offset ioport;phygical page i/o port address set ------
	mov	di,offset maptbl;
	mov	cx,4		;
	mov	bx,[emsio]	;
	add	bx,4		;
ramch14:			;
	mov	[si],bx		;
	mov	[di.ioadr],bx	;
	add	di,TBLSIZE	;
	mov	dx,bx		;phygical pages disable ---
	xor	al,al		;
	call	emsout		;--------------------------
	inc	si		;
	inc	si		;
	inc	bx		;
	loop	ramch14		;-----------------------------------------
;
	call	SYSRAM    	;;;v1.01
ramch4:				;
	mov	bp,dx		;
	xor	bx,bx		;
ramch6:				;
	call	ckromram	;
	jnb	ramch1		;
	inc	bx		;
	cmp	bx,20h		;
	jnb	ramch2		;
	cmp	dx,ROMSEG	;
	jb	ramch6		;
	jmp	ramch3		;
ramch1:				;
	cmp	dx,ENDSEG	;
	jb	ramch4		;
ramch3:				;
	push	cs		;
	pop	ds		;
	mov	si,offset nomem	;
	call	strdsp		;
	stc			;
	jmp	ramch5		;
ramch2:				;
	push	cs		;
	pop	ds		;
	mov	ax,bp		;
	mov	[emmseg],ax	;
	test	byte ptr [sysflg],2;
	jz	ramch12		;
	mov	ax,[pageseg]	;phygical page segment set
	mov	[emmseg],ax	;-----------------------------------------
ramch12:			;
	mov	ax,[emmseg]	;segment start address set
	mov	al,ah		;
	mov	dx,[emsio]	;
	add	dx,3		;
	call	emsout		;EMS i/o port data out
;
	mov	si,offset pagemsg;display page test msg
	call	strdsp		;
	in	al,I8042+1	;logical page test
	or	al,4		;system memory parity disable
	out	I8042+1,al	;
	mov	si,offset ioport;
	mov	di,offset rpage	;
	mov	dx,[si]		;
	cmp	dx,0ffffh	;
	jz	ramch7		;
	xor	ah,ah		;
	mov	al,[pageofs]	;EMS logical page start no. get
	mov	cx,PAGE_MAX	;
	xor	bx,bx		;
ramch9:				;
	push	cx		;
	call	emsout		;EMS i/o port data out
	call	imgchk		;
	jnb	ramch17		;
	pop	cx		;
	jmp	ramch16		;
ramch17:			;
	call	pagetst		;
	jb	ramch8		;
	mov	[di],NOT_USE	;
	mov	[di.l_lpageno],bx;
	inc	bx		;
	add	di,PAGESIZE	;
ramch8:				;
	pop	cx		;
	inc	al		;
	loop	ramch9		;
ramch16:			;
	mov	[total],bx	;
	mov	[unalloc],bx	;
	in	al,I8042+1	;
	and	al,0fbh		;system memory parity enable
	out	I8042+1,al	;
ramch7:				;
	clc			;
ramch5:				;
	pop	bp		;
	pop	dx		;
	pop	cx		;
	pop	bx		;
	pop	ax		;
	ret			;
ramchk	endp

;--------------------------------------------------------------------
; EMM install message display
;--------------------------------------------------------------------
instmsg	proc	near
	push	ax
	push	bx
	push	cx
	push	di
	push	cs
	pop	es
	mov	si,offset openmsg
	call	strdsp
	mov	ax,[total]
	mov	di,offset total_pg
	call	dbinasc
	mov	si,offset install_msg
	call	strdsp
	mov	ax,[emmseg]
	mov	di,offset segadr
	call	hbinasc
	mov	si,offset page_msg
	call	strdsp
	mov	ax,[emsio]
	mov	di,offset pioadr
	call	hbinasc
	mov	si,offset port_msg
	call	strdsp
	pop	di
	pop	cx
	pop	bx
	pop	ax
	ret
instmsg	endp

;--------------------------------------------------------------------
; Check rom/ram area
;	DX : segment address
; output
;	cf = 0 : OK
;	cf = 1 : NG
;--------------------------------------------------------------------
ckromram proc	near
	mov	es,dx
	xor	si,si
	mov	word ptr es:[si],0000h
	cmp	es:[si],0aa55h
	jnz	ckrom1
	xor	ah,ah
	mov	al,es:[si+2]
	mov	cl,5
	shl	ax,cl
	add	dx,ax
ckrom3:
	clc
	ret
ckrom1:
	add	dx,80h
	mov	es:[si],55aah
	cmp	es:[si],55aah
	jz	ckrom3
	stc
	ret
ckromram endp

;--------------------------------------------------------------------
; memory size check ( 1024 kbyte)
;--------------------------------------------------------------------
imgchk	proc	near
	cld
	push	ax
	push	cx
	push	si
	push	di
	push	ds
	mov	ax,[emmseg]
	mov	es,ax
	push	cx
	xor	ax,ax
	mov	ds,ax
	xor	si,si
	xor	di,di
	mov	cx,2000h
	repz	cmpsw
	cmp	cx,0
	jnz	imgch1
	pop	cx
	stc
	jmp	imgch3
imgch1:
	pop	cx
	clc
imgch3:
	pop	ds
	pop	di
	pop	si
	pop	cx
	pop	ax
	ret
imgchk	endp

;--------------------------------------------------------------------
; logical page ram check
;	AX	: logical page no.
;--------------------------------------------------------------------
pagetst	proc	near
	cld
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	ds
	push	cs
	pop	es
	sub	al,[pageofs]
	mov	di,offset tstpage
	call	dbinasc
	mov	si,offset tstpage
	call	strdsp
	mov	ax,[emmseg]
	mov	es,ax
	push	cx
;
	xor	ax,ax
	mov	ds,ax
	xor	si,si
	xor	di,di
	mov	cx,2000h
	repz	cmpsw
	cmp	cx,0
	jz	pagets2
	push	cs
	pop	ds
;
	mov	bx,offset chkchr
	mov	ax,cs:[bx]
	xor	di,di
	mov	cx,2000h
	repz	stosw
	mov	cx,2000h
	xor	di,di
	repz	scasw
	cmp	cx,0
	jz	pagets1
pagets2:
	pop	cx
	stc
	jmp	pagets3
pagets1:
	xor	ax,ax
	xor	di,di
	mov	cx,2000h
	repz	stosw
	pop	cx
	clc
pagets3:
	pop	ds
	pop	di
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	ret
pagetst	endp

;--------------------------------------------------------------------
; logical page set
;	DX	: I/O port address
;	SI	: logical page number
;--------------------------------------------------------------------
pageset	proc	near
	push	ax
	mov	ax,si
	call	emsout
	pop	ax
	ret
pageset	endp

;--------------------------------------------------------------------
; BYNARY -> ASCII
;	AX	: binary data
; output
;	ES:DI	: ascii data (DEC)
;--------------------------------------------------------------------
dbinasc:
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	mov	si,di
	mov	cx,4
	mov	bx,1000
	xor	dx,dx
dbinas1:
	div	bx
	or	al,al
	jnz	dbinas2
	cmp	cl,4
	jz	dbinas4
	cmp	byte ptr [si],' '
	jnz	dbinas2
dbinas4:
	mov	al,' '
	cmp	cl,1
	jnz	dbinas3
	xor	al,al
dbinas2:
	add	al,'0'
dbinas3:
	mov	si,di
	stosb
	push	dx
	xor	dx,dx
	mov	ax,bx
	mov	bx,10
	div	bx
	mov	bx,ax
	pop	ax
	loop	dbinas1
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	dec	di
	ret

;--------------------------------------------------------------------
; BYNARY -> ASCII
;	AX	: binary data
; output
;	ES:DI	: ascii data (HEX)
;--------------------------------------------------------------------
hbinasc:
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	mov	si,di
	mov	cx,4
	mov	bx,1000h
	xor	dx,dx
hbinas1:
	div	bx
	cmp	al,10
	jb	hbinas2
	add	al,7
hbinas2:
	add	al,'0'
hbinas3:
	mov	si,di
	stosb
	push	dx
	xor	dx,dx
	mov	ax,bx
	mov	bx,10h
	div	bx
	mov	bx,ax
	pop	ax
	loop	hbinas1
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	dec	di
	ret

;--------------------------------------------------------------------
; ASCII -> BINARY
;	(ES:DI) = ascii data address (DEC)
; output
;	AX = binary data
;--------------------------------------------------------------------
ascbin1:
	push	bx
	push	cx
	push	dx
	xor	dl,dl
	mov	cx,4
	xor	bx,bx
ascbin11:
	mov	al,es:[di]
	cmp	al,' '
	jz	ascbin13
	cmp	al,TAB
	jnz	ascbin14
ascbin13:
	or	dl,dl
	jnz	ascbin12
	inc	di
	jmp	ascbin11
ascbin12:
	or	dl,dl
	jnz	ascbin16
	stc
	jmp	ascbin15
ascbin14:
	cmp	al,'0'
	jb	ascbin12
	cmp	al,':'
	jnb	ascbin12
	mov	dl,1
	sub	al,'0'
	xor	ah,ah
	xchg	bx,ax
	push	dx
	mov	dx,10
	mul	dx
	add	bx,ax
	pop	dx
	inc	di
	loop	ascbin11
ascbin16:
	mov	ax,bx
	clc
ascbin15:
	pop	dx
	pop	cx
	pop	bx
	ret

;--------------------------------------------------------------------
; ASCII -> BINARY
;	(ES:DI) = ascii data address (HEX)
; output
;	AX = binary data
;--------------------------------------------------------------------
ascbin2:
	push	bx
	push	cx
	push	dx
	xor	dl,dl
	mov	cx,4
	xor	bx,bx
ascbin21:
	mov	al,es:[di]
	cmp	al,' '
	jz	ascbin23
	cmp	al,TAB
	jnz	ascbin24
ascbin23:
	or	dl,dl
	jnz	ascbin22
	inc	di
	jmp	ascbin21
ascbin22:
	or	dl,dl
	jnz	ascbin27
	stc
	jmp	ascbin25
ascbin24:
	cmp	al,'0'
	jb	ascbin22
	cmp	al,':'
	jb	ascbin26
	cmp	al,'A'
	jb	ascbin22
	cmp	al,'G'
	jnb	ascbin22
	sub	al,7
ascbin26:
	mov	dl,1
	sub	al,'0'
	xor	ah,ah
	xchg	bx,ax
	push	dx
	mov	dx,10h
	mul	dx
	add	bx,ax
	pop	dx
	inc	di
	loop	ascbin21
ascbin27:
	mov	ax,bx
	clc
ascbin25:
	pop	dx
	pop	cx
	pop	bx
	ret

;--------------------------------------------------------------------
; STRINGS DISPLAY SUB
;	DS:SI	: strings datas
;--------------------------------------------------------------------
strdsp:
	push	ax
	push	dx
	mov	dx,si
	mov	ah,9
	call	bdos
	pop	dx
	pop	ax
	ret

;--------------------------------------------------------------------
;  BDOS CALL SUB
;--------------------------------------------------------------------
bdos:
	int	21h
	ret

;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

;
;
;
openmsg:	DB	CR
		DB	'*********************************************',CR,LF
		DB	'*  SUNTAC-62 Chip Set EMS Driver Rev. 1.02  *',CR,LF
		DB	'*  SUN ELECTRONICS CORP. 1987               *',CR,LF
		DB	'*********************************************',CR,LF
		DB	'$'
install_msg:	DB	'EMS DRIVER INSTALL TOTAL PAGES : '
total_pg:	DB	'0000',CR,LF,'$'
page_msg:	DB	'EMS PAGE SEGMENT : '
segadr:		DB	'0000H',CR,LF,'$'
port_msg:	DB	'EMS PORT ADDRESS : '
pioadr:		DB	'0000H',CR,LF,LF,'$'
notinst:	DB	'EMS DRIVER FAIL TO INSTALL ABORTED !',CR,LF,'$'
errinst:	DB	'EMS DRIVER INSTALLED !',CR,LF,'$'
nomem:		DB	'NO FREE MEMORY SEGMENT, EMS DRIVER CAN NOT INSTALL !'
		DB	CR,LF,'$'
pagemsg		DB	'0000 PAGES TESTING',CR,'$'
tstpage		DB	'0000',CR,'$'
pageseg		DW	0	;phygical page segment
rqioadr		DW	0	;request EMS i/o port address
sysflg		DB	0	;system option flag
sysmem		DW	0	;system memory size
chkchr		DW	55aah

code	ends
	end


;********************* end of file **************************