/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.log4j.helpers.FileHelper;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.ErrorCode;

/**
 * Responsible for compressing log files using a given compression algorithm,
 * adding checksums if specified.
 * 
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.3
 */
final class LogFileCompressor implements Runnable, FileRollEventListener {

  private static final int QUEUE_LIMIT = 64;

  private final TimeAndSizeRollingAppender appender;

  private final AppenderRollingProperties properties;

  private Thread threadRef = null;

  private final List queue;

  LogFileCompressor(final TimeAndSizeRollingAppender rollingAppender,
      final AppenderRollingProperties appenderRollingProperties) {
    super();
    this.appender = rollingAppender;
    this.properties = appenderRollingProperties;
    this.queue = Collections.synchronizedList(new ArrayList());
  }

  public final void run() {
    LogLog.debug("Log file compressor started");
    try {
      while (this.isRunning()) {
        try {
          this.compressNext();
        } catch (InterruptedException e) {
          // game over
          Thread.currentThread().interrupt();
          break;
        }
      }
    } catch (Exception e) {
      this.appender.getErrorHandler().error("Log file compressor failed", e,
          ErrorCode.GENERIC_FAILURE);
    }
    this.threadRef = null;
    LogLog.debug("Log file compressor stopped");
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.apache.log4j.appender.FileRollEventListener#onFileRoll(org.apache.log4j.appender.FileRollEvent)
   */
  public final void onFileRoll(final FileRollEvent fileRollEvent) {
    this.compress(fileRollEvent.getBackupFile());
  }

  /**
   * For test purposes only
   * 
   * @return max files in the backup file queue
   */
  final int getQueueLimit() {
    return QUEUE_LIMIT;
  }

  /**
   * Starts the compressor.
   */
  final void begin() {
    if (LogFileCompressionStrategy.existsFor(this.properties)) {
      final Thread thread = new Thread(this, "Log4J File Compressor");
      thread.setDaemon(true);
      thread.start();
      this.threadRef = thread;
    }
  }

  /**
   * Stops the compressor.
   */
  final void end() {
    final Thread thread = this.threadRef;
    if (thread != null) {
      thread.interrupt();
      try {
        thread.join();
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
    }
    this.threadRef = null;
  }

  final void compress(final File file) {
    if (this.isRunning()) {
      synchronized (this.queue) {
        while (this.queue.size() > QUEUE_LIMIT) {
          if (this.properties.isCompressionBlocking()) {
            try {
              this.queue.wait(this.properties.getCompressionMaxWait());
            } catch (InterruptedException e) {
              Thread.currentThread().interrupt();
              return; // stop
            }
          } else {
            this.queue.remove(0);
          }
        }
        this.queue.add(file);
        this.queue.notify();
      }
    }
  }

  private boolean isRunning() {
    return (this.threadRef != null) ? (!this.threadRef.isInterrupted()) : false;
  }

  private boolean queueBelowMinSize() {
    synchronized (this.queue) {
      if (this.properties.getCompressionMinQueueSize() > 0) {
        return (this.queue.size() < this.properties
            .getCompressionMinQueueSize());
      } else {
        return this.queue.isEmpty();
      }
    }
  }

  private void compressNext() throws InterruptedException {
    if (this.queue != null) {
      File file = null;
      synchronized (this.queue) {
        while (this.queueBelowMinSize()) {
          this.queue.wait(this.properties.getCompressionMaxWait());
        }
        file = (File) this.queue.remove(0);
        this.queue.notify();
      }
      if (file != null) {
        this.doCompression(file);
      }
    }
  }

  private void doCompression(final File file) {
    if (new FileHelper().isWriteable(file)) {
      final LogFileCompressionStrategy compressionStrategy = LogFileCompressionStrategy
          .findCompressionStrategy(this.properties);
      if (compressionStrategy != null) {
        compressionStrategy.compress(file, this.properties);
      }
    }
  }
}
