package org.apache.log4j.appender;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.log4j.helpers.FileHelper;

import junit.framework.Assert;
import junit.framework.TestCase;

public class TestCompressionStrategies extends TestCase {

  private static final String BACKUP_FILE = "mylog.log.10.1";

  private final FileTestHelper fileTestHelper = new FileTestHelper();

  protected void setUp() throws Exception {
    this.fileTestHelper.setUp().writeFile(BACKUP_FILE, 10);
  }

  protected void tearDown() throws Exception {
    this.fileTestHelper.tearDown();
  }

  public void testIsZipRequiredStrategy() {
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("ZIP");
    Assert.assertTrue(LogFileCompressionStrategy.ZIP
        .isRequiredStrategy(properties));
    LogFileCompressionStrategy compressionStrategy = LogFileCompressionStrategy
        .findCompressionStrategy(properties);
    Assert.assertEquals(LogFileCompressionStrategy.ZIP, compressionStrategy);
    properties.setCompressionAlgorithm("zip");
    Assert.assertTrue(LogFileCompressionStrategy.ZIP
        .isRequiredStrategy(properties));
    compressionStrategy = LogFileCompressionStrategy
        .findCompressionStrategy(properties);
    Assert.assertEquals(LogFileCompressionStrategy.ZIP, compressionStrategy);
    properties.setCompressionAlgorithm("WOMBLE");
    Assert.assertFalse(LogFileCompressionStrategy.ZIP
        .isRequiredStrategy(properties));
    properties.setCompressionAlgorithm("wimbledon");
    Assert.assertFalse(LogFileCompressionStrategy.ZIP
        .isRequiredStrategy(properties));
  }

  public void testIsGzipRequiredStrategy() {
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("GZ");
    Assert.assertTrue(LogFileCompressionStrategy.GZIP
        .isRequiredStrategy(properties));
    LogFileCompressionStrategy compressionStrategy = LogFileCompressionStrategy
        .findCompressionStrategy(properties);
    Assert.assertEquals(LogFileCompressionStrategy.GZIP, compressionStrategy);
    properties.setCompressionAlgorithm("gz");
    Assert.assertTrue(LogFileCompressionStrategy.GZIP
        .isRequiredStrategy(properties));
    compressionStrategy = LogFileCompressionStrategy
        .findCompressionStrategy(properties);
    Assert.assertEquals(LogFileCompressionStrategy.GZIP, compressionStrategy);
    properties.setCompressionAlgorithm("WOMBLE");
    Assert.assertFalse(LogFileCompressionStrategy.GZIP
        .isRequiredStrategy(properties));
    properties.setCompressionAlgorithm("wimbledon");
    Assert.assertFalse(LogFileCompressionStrategy.GZIP
        .isRequiredStrategy(properties));
  }

  public void testGetAlgorithmName() {
    Assert.assertEquals("zip", LogFileCompressionStrategy.ZIP
        .getAlgorithmName());
    Assert.assertEquals("gz", LogFileCompressionStrategy.GZIP
        .getAlgorithmName());
  }

  public void testCompressUsingZip() {
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("ZIP");
    File logFile = this.fileTestHelper.fileForTest(BACKUP_FILE);
    Assert.assertTrue(logFile.exists());
    File deflatedFile = this.fileTestHelper.fileForTest(BACKUP_FILE + '.'
        + LogFileCompressionStrategy.ZIP.getAlgorithmName());
    Assert.assertFalse(deflatedFile.exists());
    InputStream logFileStream = this.fileTestHelper.readFile(logFile);
    Assert.assertNotNull(logFileStream);
    LogFileCompressionStrategy.ZIP.compress(logFile, properties);
    Assert.assertFalse(logFile.exists());
    Assert.assertTrue(deflatedFile.exists());
    Assert.assertTrue(new FileHelper().isZip(deflatedFile));
    ZipFile zipFile = null;
    try {
      zipFile = new ZipFile(deflatedFile);
      Enumeration enumeration = zipFile.entries();
      Assert.assertNotNull(enumeration);
      Assert.assertTrue(enumeration.hasMoreElements());
      ZipEntry zipEntry = (ZipEntry) enumeration.nextElement();
      Assert.assertFalse(enumeration.hasMoreElements());
      Assert.assertNotNull(zipEntry);
      Assert.assertEquals(BACKUP_FILE, zipEntry.getName());
      Assert.assertFalse(zipEntry.isDirectory());
      Assert.assertTrue(zipEntry.getSize() > zipEntry.getCompressedSize());
      zipEntry = zipFile.getEntry(BACKUP_FILE);
      Assert.assertNotNull(zipEntry);
      InputStream zipFileStream = zipFile.getInputStream(zipEntry);
      while (zipFileStream.available() > 0) {
        Assert.assertEquals(logFileStream.read(), zipFileStream.read());
      }
    } catch (ZipException e) {
      e.printStackTrace(System.err);
      Assert.fail(e.getMessage());
    } catch (IOException e) {
      e.printStackTrace(System.err);
      Assert.fail(e.getMessage());
    } finally {
      if (zipFile != null) {
        try {
          zipFile.close();
        } catch (IOException e) {
          // ignore
        }
      }
    }
  }

  public void testCompressUsingGzip() {
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("GZ");
    File logFile = this.fileTestHelper.fileForTest(BACKUP_FILE);
    Assert.assertTrue(logFile.exists());
    File deflatedFile = this.fileTestHelper.fileForTest(BACKUP_FILE + '.'
        + LogFileCompressionStrategy.GZIP.getAlgorithmName());
    Assert.assertFalse(deflatedFile.exists());
    InputStream logFileStream = this.fileTestHelper.readFile(logFile);
    Assert.assertNotNull(logFileStream);
    LogFileCompressionStrategy.GZIP.compress(logFile, properties);
    Assert.assertFalse(logFile.exists());
    Assert.assertTrue(deflatedFile.exists());
    Assert.assertTrue(new FileHelper().isGZip(deflatedFile));
    InputStream zipFileStream = null;
    try {
      zipFileStream = new GZIPInputStream(new FileInputStream(deflatedFile));
      while (zipFileStream.available() > 0) {
        Assert.assertEquals(logFileStream.read(), zipFileStream.read());
      }
    } catch (ZipException e) {
      e.printStackTrace(System.err);
      Assert.fail(e.getMessage());
    } catch (IOException e) {
      e.printStackTrace(System.err);
      Assert.fail(e.getMessage());
    } finally {
      if (zipFileStream != null) {
        try {
          zipFileStream.close();
        } catch (IOException e) {
          // ignore
        }
      }
    }
  }
}
