PROGRAM PIA_8255; { Intel 8255 Parallel Interface Adapter }
{ Demonstrate 8255's operation form a parallel printer port.  program outputs
  data to 8255 Ports 1 and  2, while reading data form Port 3.  This IC
  expands an 8-bit parallel printer port into three byte-wide I/O ports.
}
USES CRT, PRINTER, My_TPU;

TYPE     Data_Type = ARRAY[0..16] OF BYTE;

CONST    A0 = 1; { Bit 1 = D0, STROBE at Base-Address + 2                    }
         A1 = 2; { Bit 2 = D1, AUTO-FEED at Base-Address + 2                 }
         RD = 4; { Bit 3 = D2, *INITIALIZE at Base-Address + 2               }
         WR = 8; { Bit 4 = D3, *SELECT-INPUT at Base-Address + 2             }
                 { NOTE: an asterisk, (*),  denotes inactive high logic      }
   T : BYTE = 5;

VAR  A, E, X, Port_Number, Control_Byte, Data_Byte, Control_Lines  : BYTE;
                          Base_Addr_LO_Nibble, Base_Addr_HI_Nibble : BYTE;
                                                   Lptx, Data_Word : WORD;
                                                               Num : INTEGER;
                                                   Inverted_Logic  : Data_Type;
                                                                Ch : CHAR;

PROCEDURE Generate_Data(VAR Inverted_Logic : Data_Type);
{ makes "look-up" table for Base + 2's inverted signals on D0, D1, and D3    }
{ faster than doing repetitive logic conversions for all new data            }

VAR Count, Temp, D0, D1, D2, D3 : BYTE;
BEGIN
Count := 0;
FOR D3 := 1 DOWNTO 0 DO         { D3 of Base + 2 is inverted                 }
  FOR D2 := 0 TO 1 DO           { D2 of Base + 2 is NOT inverted             }
    FOR D1 := 1 DOWNTO 0 DO     { D1 of Base + 2 is inverted                 }
      FOR D0 := 1 DOWNTO 0 DO   { D0 of Base + 2 is inverted                 }
        BEGIN
        Temp := (8 * D3) + ( 4 * D2) + ( 2 * D1 ) + D0;
        Inverted_Logic[Count] := Temp;
        Count := Count + 1;
        END;
END; { generate data }

PROCEDURE Send_Control_Word(Control_BYTE : BYTE);
{ issues control byte to the 8255.  Tells the 8255 the Mode, which ports       }
{ are inputs, and which parts are outputs.                                     }
VAR Temp : BYTE;
BEGIN
Base_Addr_LO_Nibble := $0F;              { Initialize Control Lines, 0000 1111 }
PORT[LPTx] := Base_Addr_LO_Nibble;
Base_Addr_HI_Nibble := Control_Byte AND $F0;       { HI nibble of control byte }
PORT[LPTx] := Base_Addr_HI_Nibble OR Base_Addr_LO_Nibble;
Temp := Control_Byte AND $0F;                      { LO nibble of control byte }
Temp := Inverted_Logic[Temp];
PORT[LPTx + 2] := Temp;
PORT[LPTx] := PORT[LPTx] AND $F7; { 1111 0111, set Write Line to active LOW    }
delay(T);     { the LM311 RC network forces the use of a prolonged WR signal   }
PORT[LPTx] := PORT[LPTx] OR $08; {  0000 1000, set Write Line to inactive HIGH }
END;

PROCEDURE Send_Data(Data_Byte, Port_Number : Byte);
{ Send Data_Byte to Port [ Port_Number ]                                       }
VAR Temp : BYTE;
BEGIN
Base_Addr_LO_Nibble := $0C OR Port_Number; { Initialize Control Lines,00001100 }
PORT[LPTx] := Base_Addr_LO_Nibble;
PORT[LPTx] := ( Data_Byte AND $F0 ) OR Base_Addr_LO_Nibble;
Temp := Data_Byte AND $0F;
Temp := Inverted_Logic[Temp];
PORT[LPTx + 2] := Temp;
PORT[LPTx] := PORT[LPTx] AND $F7; { 1111 0111, set Write Line to active LOW    }
delay(T);     { the LM311 RC network forces the use of a prolonged WR signal   }
PORT[LPTx] := PORT[LPTx] OR $08;  { 0000 1000, set Write Line to inactive HIGH }
END;

PROCEDURE Output_Data;
  { Generate data to set each pin of Port 1, and Port 2, HIGH in sequence    }
VAR Ct : BYTE;
BEGIN { output data }
For A := 0 to 1 dO
  BEGIN
  Ct := 1;
  FOR E := 1 TO 8 DO
    BEGIN
    Send_Data(Ct, A); Ct := Ct + Ct; IF KeyPressed Then Exit;
    END; { for e }
  Ct := 0; Send_Data(Ct, A);
  END; { for a }
For A := 1 DOWNTO 0 DO
  BEGIN
  Ct := 128;
  FOR E := 1 TO 8 DO
    BEGIN
    Send_Data(Ct, A); Ct := Ct DIV 2; IF KeyPressed Then Exit;
    END; { for e }
  Ct := 0; Send_Data(Ct, A);
  END; { for a }
END; { output data }

PROCEDURE Input_Data(VAR Data_Byte : BYTE; Port_Number : BYTE);
{ Read data from selected port, invert inverted data on lines D0, D1, and D3 }
{ Return, via Data_Byte, a correct reading byte of data                      }
VAR Temp, TempB : BYTE;
BEGIN { input demo }
Base_Addr_LO_Nibble := $0C;  { Initialize Control Lines,           0000 1100 }
Base_Addr_LO_Nibble := $0C OR Port_Number; { select port for input           }
PORT[LPTx] := Base_Addr_LO_Nibble;                              {  0000 1101 }
PORT[LPTX] := PORT[LPTx] AND $0B;                { Rd line = LOW;  0000 1001 }
PORT[LPTx + 2] := $04;             { set all lines high for read;  0000 0100 }
Data_Byte := ( PORT[LPTx + 2] AND $0F );
Data_Byte := Data_Byte OR ( PORT[LPTx + 1] AND $F0);
PORT[LPTx] := Base_Addr_LO_Nibble;              { Rd line = HIGH;  0000 11xx }
  { invert inverted BASE + 2 data lines: D0, D1, and D3; }
Temp := Data_Byte AND $0F;
Temp := Inverted_Logic[Temp];
Data_Byte := (Data_Byte AND $F0 ) OR Temp;
END;  { input demo }

PROCEDURE Process_Speed_Request(VAR Ch : CHAR);
{
  Check for PAGE-UP, PAGE-DOWN, or ESC key pressing.
  Process PAGE-UP, and PAGE-DOWN  as request for faster or slower display
}
VAR Ch1, Ch2 : CHAR;
BEGIN
Ch1 := READKEY;
IF KEYPRESSED THEN Ch2 := READKEY;
If Ch1 IN ['9', '3', CHAR(27)] THEN Ch := Ch1
  ELSE IF Ch2 IN ['I', 'Q'] THEN Ch := Ch2;
CASE Ch OF
  '9', 'I' : { Go Faster }   IF T > 4 THEN  T := T - 3;
  '3', 'Q' : { Go Slower } IF T < 200 THEN  T := T + 3;
  CHAR(27) : EXIT;
  END; { case }
END; { process speed request }

BEGIN { main }
ClrScr;
LPTx := Find_LPTx(Num);
Generate_Data(Inverted_Logic);
Send_Control_Word($89); { PORT 1 & 2 Output, PORT 3 Input,         1000 1001 }
X := 0;
  REPEAT
  Output_Data;
  Port_Number := 2;
  Input_Data(Data_Byte, Port_Number);
  If X <> Data_Byte THEN    { only print data if it differs from old data, x }
    BEGIN WRITELN('Data = ',Data_Byte); X := Data_Byte; END;
  IF KEYPRESSED THEN Process_Speed_Request(Ch);
  UNTIL ORD(Ch) = 27;                          { exit if ESC key was pressed }
END. { main }
