
;
;                          DISKTEST
;
;
;    Reads sectors from the specified disk to test how BIOS
;    responds.  Specificly the test looks for problems when
;    more than one track is read, as many systems show different
;    limitations.
;
;    Most BIOSs destroy AL, although some return the number of
;    sectors read.
;
;    (c) Copyright 1994, 1996  Frank van Gilluwe
;    All Rights Reserved.
;
;    V2.00 - Aligns data segments on a physical 64 KB boundary

include undocpc.inc

cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stackt


;  data strings 

info     db     CR, LF, CR, LF
         db     'DISKTEST - Interrupt 13h Sector'
         db     ' Read Test        v2.00 (c) 1994, 1996 FVG'
         db     CR, LF
         db     ''
         db     ''
         db     CR, LF, '$'

novalue  db     'Must specify the drive number in hex on '
         db     'the command line (0, 1, 80, 81).'
         db     CR, LF, CR, LF, '$'

loadboot db     'Loading diskette boot sector to determine '
         db     'media type.', CR, LF, '$'

badmedia db     'Unknown media type from boot sector. Cannot '
         db     'proceed.', CR, LF, '$'

loadref  db     'Loading 128 KB, 1 sector at a time, for '
         db     'reference data in tests.', CR, LF, '$'
loaddot  db     '.$'


tried    db     'Number of sectors being read: '
sectry   db     '00h   $'

testok   db     'Compare passed, returns AH=0, AL='
sectr    db     '  h.', CR, LF, '$'

testfail db     'Failed. Error number = '
errnum   db     '  h', CR, LF, '$'

badcomp  db     'Compare error. Sector '
badsect  db     '00h did not match reference'
         db     ' sector.', CR, LF, '$'

crlf     db     CR, LF, '$'


drive       db          0          ; drive number to use
max_sectors db          0          ; number of sectors per track
max_heads   db          0          ; total number of heads
drive_type  db          0          ; diskette drive type
                                   ;   1 = 360 KB
                                   ;   2 = 1.2 MB
                                   ;   3 = 720 KB
                                   ;   4 = 1.44 MB
                                   ;   5 = 2.88 MB

sect        db          0          ; sector to read
head        db          0          ; head to read
cyln        dw          0          ; cylinder to read

dest        dw          0          ; destintion
count       db          0          ; sectors per single transfer
total_count dw          0          ; sectors to process
sec_read    db          0          ; temp register, number of
                                   ;   sectors read by int 13h

disktest proc   far

start:
        push    cs
        pop     ds
        OUTMSG  info               ; display information
        cmp     byte ptr es:[80h], 1  ; chars on command line ?
        jb      bad_command_line   ; jump if not
        mov     si, 81h            ; start of string
        call    ascii_bytes
        jnc     test_begin         ; jump if valid value

bad_command_line:
        OUTMSG  novalue            ; bad value
        jmp     done


test_begin:
        mov     [drive], dl        ; save value

; get drive parameters

        mov     ah, 8
        int     13h                ; get parameters, drive in dl
        and     cl, 3Fh            ; get sectors per track
        mov     [max_sectors], cl  ; save
        mov     [max_heads], dh
        mov     [drive_type], bl

; if diskette drive, we need to find the diskette media type, in case
; the diskette holds less than the drive capacity

        test    [drive], 80h       ; hard disk ?
        jnz     load128            ; jump if so

        OUTMSG  loadboot
        mov     ax, refseg1        ; REFSEG1 used for temporary
                                   ;  storage of boot sector
        and     ax, 0F000h         ; align to physical boundary
        mov     es, ax
        mov     [total_count], 0   ; total number of sectors
        mov     [count], 1         ; transfer 1 sector
        mov     [cyln], 0
        mov     [head], 0
        mov     [sect], 1          ; starting location
        call    read_sector        ; read 1 sector

        mov     al, es:[15h]       ; get media descriptor byte
        cmp     [drive_type], 3    ; 3.5" drive ?
        jae     check_720          ; jump if so
        cmp     al, 0F9h           ; 1.2 MB ?
        je      load128            ; then media same as max drive
        cmp     al, 0FDh           ; 360 KB diskette ?
        jne     unknown_media      ; jump if not
        mov     [max_sectors], 9
        jmp     load128

check_720:
        cmp     al, 0F9h           ; 720 KB diskette ?
        jne     check_144          ; jump if not
        mov     [max_sectors], 9
        jmp     load128

check_144:
        cmp     al, 0F0h           ; 1.44 or 2.88 MB ?
        jne     unknown_media      ; jump if not
        mov     ax, es:[13h]       ; get number of sectors on diskette
        cmp     ax, 0B40h          ; 720 KB has B40h sectors or less
        ja      load128            ; must be 2.88 diskette in 2.88 drive
        mov     [max_sectors], 18  ; must be 1.44 diskette
        jmp     load128

unknown_media:
        OUTMSG  badmedia
        jmp     failure_exit


; load 128 KB from disk (we'll start at sector 1, head 0)

load128:
        OUTMSG  loadref
        mov     ax, refseg1
        and     ax, 0F000h         ; align to physical boundary
        mov     es, ax
        mov     [total_count], 0   ; total number of sectors
        mov     [count], 1         ; transfer 1 sector per loop
        mov     [cyln], 0
new_cylinder:
        mov     [head], 0
new_head:
        mov     [sect], 1          ; starting location
new_sector:
        call    read_sector        ; read 1 sector
        mov     ax, es
        add     ax, 20h            ; next free area
        mov     es, ax
        inc     [total_count]
        cmp     [total_count], 100h  ; 128 KB read in yet ?
        je      begin_test
        test    [total_count], 3
        jnz     no_dot
        OUTMSG  loaddot            ; display dot every 4 sectors
no_dot:
        inc     [sect]
        mov     al, [sect]
        cmp     al, [max_sectors]  ; end of track ?
        jbe     new_sector
        inc     [head]
        mov     al, [head]
        cmp     al, [max_heads]
        jbe     new_head
        inc     [cyln]
        jmp     new_cylinder

; A full 128 KB of data has been read into the reference area,
;   so begin tests of reading and comparing multiple sector
;   reads in a single operation

begin_test:
        OUTMSG  crlf
        OUTMSG  crlf
        mov     ax, tempseg1       ; where to load test data
        and     ax, 0F000h         ; align to physical boundary
        mov     es, ax
        mov     [count], 1
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     al, [max_sectors]  ; read track
        mov     [count], al
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     al, [max_sectors]  ; read 2 tracks
        mov     [count], al
        shl     [count], 1
        call    readit             ; read sector & compare
        jc      comp_failure

        test    [drive], 80h       ; diskette drive ?
        jz      disk_skp1          ; skip if so, 2 tracks already tested
        mov     al, [max_heads]    ; read cylinder
        inc     al
        mul     [max_sectors]
        cmp     ax, 7Fh            ; more than 127 sectors ?
        jae     disk_skp1          ; skip if so
        mov     [count], al
        call    readit             ; read sector & compare

; now test boundry conditions

disk_skp1:
        OUTMSG  crlf
        mov     [count], 7Fh
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     [count], 80h
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     [count], 81h
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     [count], 0C0h
        call    readit             ; read sector & compare
        jc      comp_failure

        mov     [count], 0FFh
        call    readit             ; read sector & compare
        jc      comp_failure

        jmp     done

comp_failure:                      ; gets here if compare bad
failure_exit:
done:
        mov     ah, 4Ch
        int     21h                ; DOS terminate
disktest endp

;
;    READ SECTORS INTO MEMORY
;
;       Called with:    [count] = number of sectors to read
;                       [drive] = drive to test
;
;       Returns:        message of return status on screen
;                       carry = 1 if compare failure

readit  proc    near
        push    es
        mov     al, [count]
        mov     bx, offset sectry
        call    hex
        OUTMSG  tried              ; # of sectors attempting

; first clear entire 128 KB data area

        mov     bx, 4
        push    es
        mov     ax, tempseg1
        and     ax, 0F000h         ; align to physical boundary
        mov     es, ax
read_loop1:
        mov     cx, 4000h         ; 32 KB (16 K words)
        mov     ax, 5A5Bh         ; initialization pattern
        mov     di, 0
        cld
        rep     stosw
        mov     ax, es
        add     ax, 800h
        mov     es, ax
        dec     bx
        jnz     read_loop1         ; initialize next 32 KB
        pop     es

; now perform single read (with re-tries, if error)

        mov     [cyln], 0          ; starting address
        mov     [head], 0
        mov     [sect], 1
        call    read_sector        ; read sectors into temparea

        mov     [sec_read], al     ; AL might be # of sectors
        mov     bx, offset sectr   ;   read, so display AL value
        call    hex                ; convert to hex
        OUTMSG  testok             ; display OK and AL value

; now see if data matches reference read

        mov     al, [count]        ; sectors to compare
        xor     ah, ah
        mov     bp, ax
        push    ds
        mov     ax, refseg1
        and     ax, 0F000h         ; align to physical boundary
        mov     es, ax
        mov     ax, tempseg1
        and     ax, 0F000h         ; align to physical boundary
        mov     ds, ax

readit_loop3:
        mov     cx, 512            ; 512 bytes per sector
        mov     di, 0
        mov     si, di
        cld
        repe    cmpsb              ; compare sector

        cmp     cx, 0
        jne     readit_bad         ; compare failed

        mov     ax, ds             ; adjust to next 512 bytes
        add     ax, 20h            ;  by changing segment values
        mov     ds, ax
        mov     ax, es
        add     ax, 20h
        mov     es, ax

        dec     bp
        jnz     readit_loop3       ; compare next sector
        pop     ds
        clc
        jmp     readit_exit        ; compare ok

; sector compare failed

readit_bad:
        pop     ds
        mov     ax, bp
        mov     bx, offset badsect
        call    hex
        OUTMSG  badcomp            ; failed compare message
        stc

readit_exit:
        pop     es
        ret
readit  endp


;
;    READ SECTORS INTO MEMORY
;
;       Called with:    [cyln]  = cylinder
;                       [head]  = head
;                       [sect]  = sector
;                       [count] = number of sectors to read
;                       es = segment were to read data to
;
;       Returns:        sector information read
;                       al = value returned from INT 13h
;
;
read_sector proc    near
        push    bp
        mov     cx, 6              ; retry count
d_loop1:
        mov     bp, cx             ; save retry count
        push    ds
        mov     ch, byte ptr [cyln]   ; lower 8-bits of cylinder
        mov     cl, byte ptr [cyln+1] ; upper 2-bits of cylinder
        ror     cl, 1              ; put cylinder bits at 7 & 6
        ror     cl, 1
        or      cl, [sect]         ; sector number
        mov     dh, [head]         ; head number
        mov     dl, [drive]        ; drive
        mov     ax, es
        mov     ds, ax
        mov     ah, 2              ; read sectors function
        mov     al, cs:[count]     ; number of sectors to read
        mov     bx, 0              ; ds:bx = where to xfer data
        int     13h                ; read sectors
        pop     ds
        mov     cx, bp
        jnc     read_ok
        loop    d_loop1            ; retry

        mov     bx, offset errnum
        mov     al, ah             ; get error number
        call    hex                ; convert to hex
        OUTMSG  testfail           ; display error code
        jmp     failure_exit       ; just exit if error

read_ok:
        pop     bp
        ret
read_sector endp


;
;    HEX SUBROUTINE
;       convert the hex number in al into two ascii characters
;       at ptr ds:bx.
;
;       Called with:    al = input hex number
;                       ds:bx = ptr where to put ascii
;
;       Regs Used:      al, bx

hex     proc    near
        push    bx
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al
        mov     ax, bx
        pop     bx
        mov     [bx], ax           ; transfer ascii bytes
        ret
hex     endp

;
;    CONVERT STRING INTO ASCII BYTE OR WORD
;       Skips leading blanks to find value, which can be from
;       1 to 4 hex digits long (also skips tabs)
;
;       Called with:    es:si = ptr to start of value,
;                               format 0-FFFF
;                               term char = space, 0, or ','
;
;       Returns:        carry = 0 if valid value found
;                         dx = value
;                         si = ptr to byte beyond value
;                       carry = 1 if value invalid
;                         si unchanged
;
;       Regs used:      ax, cx, dx, si

ascii_bytes  proc near
        push    si
asbyskp1:
        cmp     byte ptr es:[si], ' ' ; space ?
        je      asbyskp2
        cmp     byte ptr es:[si], 9   ; tabs ?
        jne     asbyskp3
asbyskp2:
        inc     si
        cmp     byte ptr es:[si], 0Dh ; end of command ?
        je      asbyskp9              ; exit if so
        cmp     byte ptr es:[si], 0   ; end of command ?
        jne     asbyskp1              ; loop if not
        jmp     asbyskp9              ; error

asbyskp3:
        xor     dx, dx
        mov     ch, 4              ; up to 4 digits max
        mov     cl, ch             ; cl = 4 for nibble rotate
asbyloop:
        mov     al, es:[si]
        cmp     al, 40h            ; letter ?
        jb      asbyskp4           ; jump if not
        and     al, 0DFh           ; convert to upper case
        sub     al, 'A'            ; convert 'A' to 0
        cmp     al, 6
        jae     asbyskp8           ; jump if out of range
        add     al, 10             ; convert to 10 to 15
        jmp     asbyskp5

asbyskp4:
        sub     al, 30h            ; convert to digit
        cmp     al, 9              ; digit 0-9 ?
        ja      asbyskp8           ; jump if not

asbyskp5:
        xor     ah, ah             ; zero
        add     dx, ax             ; insert nibble
        rol     dx, cl             ; rotate left 1 nibble (cl=4)
        dec     ch
        cmp     ch, 0              ; all 4 digits processed ?
        je      asbyskp6           ; jump if so
        inc     si
        cmp     byte ptr es:[si], 0Dh ; end of line ?
        je      asbyskp6
        cmp     byte ptr es:[si], 0
        jne     asbyloop           ; loop up to 4 characters
asbyskp6:
        inc     si
asbyskp7:
        ror     dx, cl             ; adjust back
        pop     ax                 ; get rid of saved si
        clc                        ; valid value flag
        jmp     asbyExit

asbyskp8:                          ; invalid char found
        cmp     ch, 4              ; at least 1 digit ?
        je      asbyskp9           ; jump if not
        mov     al, es:[si]
        cmp     al, ' '            ; last invalid char a space?
        je      asbyskp7           ; jump if so (value ok)
        cmp     al, 9              ; tab ?
        je      asbyskp7           ; jump if so
        cmp     al, 0              ; end of line ?
        je      asbyskp7           ; jump if so
        cmp     al, ','            ; comma ?
        je      asbyskp7           ; jump if so
        and     al, 0DFh           ; convert to upper case
        cmp     al, 'H'
        je      asbyskp7
asbyskp9:
        stc                        ; return error
        pop     si
asbyExit:
        ret
ascii_bytes endp



cseg    ends

; This segment is used as an alignment segment so that all
; of the following segments start on a physical boundary.
; All segments after this segment are adjusted downward to
; start on a physical 64 KB segment.  This means that tempseg1
; will actually start somewhere in adjustseg.

adjustseg  segment para public

          db    0FFFFh dup (?)

adjustseg ends

; These two segments are used for temporary storage of
;    up to 128KB

tempseg1  segment para public

          db    0FFFFh dup (?)

tempseg1  ends

tempseg2  segment para public

          db    0FFFFh dup (?)

tempseg2  ends


; These two segments are used for reference storage of 128 KB

refseg1   segment para public

          db    0FFFFh dup (?)

refseg1    ends

refseg2   segment para public

          db    0FFFFh dup (?)

refseg2   ends



stackt  segment para stack

        db      200 dup (?)

stackt  ends

        end     start


