
;
;                            SYSTYPE
;
;    Determine the system type, diskettes and hard drive
;    information and display type and information.
;
;    (c) Copyright 1994, 1996  Frank van Gilluwe
;    All Rights Reserved.
;
;    V2.00 - Adds detection for PCI and version of PCI
;            Adds BIOS vendor indentification
;            Adds BIOS date code
;            Adds LBA detection of IDE hard disks
;            Improved hard disk identification

include undocpc.inc


cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stacka

; DATA AREA 

;  general data

        db      'SYSTYPE v2.00 '
        db      '(c) 1994, 1996 Frank van Gilluwe',0

syshead db      CR, LF, CR, LF
        db      'SYSTEM ANALYSIS             '
        db      '               v2.00 (c) 1994, 1996 FVG'
        db      CR, LF
        db      ''
        db      ''
        db      CR, LF, '$'

systyp  db      'System type:    '
systypm db      '                                               '
        db      CR, LF, '$'

sysmsg  db      0
        db      'PC (8088 based)                               '
sysend  db      1
        db      'XT (8088 based)                               '
        db      2
        db      'PC convertible (8088 based)                   '
        db      3
        db      'PC jr (8088 based)                            '
        db      4
        db      'unknown pre-80286                             '
        db      8
        db      'XT (80286 based)                              '
        db      10h
        db      'AT / ISA (Industry Standard Architecture)     '
        db      20h
        db      'EISA (Extended Industry Standard Architecture)'
        db      40h
        db      'MCA (Micro Channel Architecture)              '
        db      0FFh
        db      'unknown                                       '

pcityp  db      '                PCI v'
pciver  db      ' X.XX (Peripheral Component Interconnect)'
        db      CR, LF, '$'

biosv   db      CR, LF
        db      'System BIOS:    '
biosv2  db      '                                               '
        db      CR, LF, '$'
biosd   db      '                Date: '
biosd2  db      '00-Jan-00  '
        db      CR, LF, '$'
biosl1  db      '                BIOS supports LBA disks (> 504 MB)'
        db      CR, LF, '$'
biosl2  db      '                BIOS has no LBA support for disks'
        db      CR, LF, '$'

manuf   db      '                Motherboard is made '
manuf2  db      '                        '
        db      CR, LF, '$'

where1  db      'outside USA', 0
where2  db      'by American Megatrends', 0
where3  db      'in USA', 0

manufc  db      '                Chipset: '
manufc2 db      '       '
crlf    db      CR, LF, '$'

no_drvs db      'No diskette drives ', CR, LF, '$'

drv     db      'Diskette '
drvltr  db               'a:     '
drvtext db      '360KB   5.25" (40 tracks, 9 sectors per track) '
        db      CR, LF, '$'

drvtyps db      '360KB   5.25" (40 tracks, 9 sectors per track) '
drvend  db      '1.2MB   5.25" (80 tracks, 15 sectors per track)'
        db      '720KB   3.5"  (40 tracks, 9 sectors per track) '
        db      '1.44MB  3.5"  (80 tracks, 18 sectors per track)'
        db      '2.88MB  3.5"  (80 tracks, 36 sectors per track)'
        db      '320KB   5.25" (40 tracks, 8 sectors per track) '
        db      'unknown type                                   '

blank_ln db     CR, LF, '$'

no_hdrvs db     'No hard drives ', CR, LF, '$'

hdrv    db      'Hard Drive '
hdrvnum db                 '0:   '
hdrvtxt db      'PC/XT type controller     '
        db      CR, LF
        db      '                  Total size = '
hsize   db                                     '          '
        db      '(with diagnostic cylinder)'
        db      CR, LF
        db      '                '
hcyldr  db                      '      Cylinders,'
hheads  db      '      Heads,'
hsector db                  '      Sectors per track'
        db      CR, LF, '$'


hdrvtyp db      'PC/XT type controller     '
hdrvend db      'AT or IDE type controller '
        db      'SCSI type controller      '
        db      'PS/2 ESDI type controller '
        db      'Unknown controller type   '
        db      'IDE controller, LBA active'

vidmsg  db      CR, LF
        db      'Video Adapter:  $'
vidtype db      '                                           '
        db      CR, LF, '$'

vidmfg  db      'Video Vendor:   '
vidmstr db      '                                           '
        db      CR, LF, '$'


vtype   db      'MDA$     '        ; 0
vtypend db      'HGA$     '        ; 1
        db      'CGA$     '        ; 2
        db      'MCGA$    '        ; 3
        db      'EGA$     '        ; 4
        db      'VGA$     '        ; 5
        db      'SVGA$    '        ; 6
        db      'XGA$     '        ; 7
        db      'VESA XGA$'        ; 8

vcolor  db      ', programs should use color attributes.     '
        db      CR, LF, '$'
vcolore db      ', programs should use monochrome attributes.'
        db      CR, LF, '$'
        db      ', programs should use grayscale attributes. '
        db      CR, LF, '$'

novalue db      'Unknown', 0

months  db      'JanFebMarAprMayJunJulAugSepOctNovDec'

old_int6_seg dw 0                  ; temp storage for old int 6
old_int6_off dw 0                  ;  vector (bad opcode)
badoff       dw 0                  ; temp return offset if bad offset
                                   ;  interrupt 6 called

; CODE START 

systype proc    far

start:
        mov     ax, cs
        mov     ds, ax
        mov     es, ax
        OUTMSG  syshead            ; display system type

; --- Find system type

        call    sysvalue           ; get system value in al
        mov     cx, offset sysend - offset sysmsg
        mov     si, offset sysmsg
syst_loop:
        cmp     byte ptr [si], 0FFh  ; no match ?
        je      syst_skp1          ; exit if so
        cmp     al, [si]           ; system type match ?
        je      syst_skp1          ; exit if so
        add     si, cx
        jmp     syst_loop          ; loop until match or end

syst_skp1:
        inc     si
        dec     cx
        mov     di, offset systypm ; where to insert type text
        cld
        rep     movsb              ; xfer message
        OUTMSG  systyp             ; display system type

; --- Find if PCI supported, and display verison if so

        call    pci_detect
        jc      syst_BIOS          ; jump if no PCI BIOS

        ; convert bx BCD version to ASCII

        mov     di, offset pciver
        mov     al, bh
        shr     al, 4
        cmp     al, 0              ; leading zero?
        je      sysp_skp1          ; jump if so (leave blank)
        add     al, 30h            ; convert to ascii digit
        mov     [di], al
sysp_skp1:
        inc     di
        mov     al, bh
        and     al, 0Fh
        add     al, 30h            ; convert to ascii digit
        mov     [di], al
        inc     di
        inc     di
        mov     al, bl
        shr     al, 4
        add     al, 30h            ; convert to ascii digit
        mov     [di], al
        inc     di
        mov     al, bl
        and     al, 0Fh
        add     al, 30h            ; convert to ascii digit
        mov     [di], al
        OUTMSG  pcityp             ; display PCI information

; --- Find information from main system BIOS

syst_BIOS:
        call    biosinfo           ; system type in al to biosinfo
        push    ax
        push    di
        push    es
        mov     di, offset biosv2  ; where to put vendor string
        cmp     dl, 0              ; unknown vendor?
        jne     sysb_skp1          ; jump if not
        push    ds
        pop     es
        mov     si, offset novalue ; insert "unknown" string
sysb_skp1:
        call    xferstring         ; move string from es:[si] to [di]
        pop     es
        pop     di
        OUTMSG  biosv              ; display BIOS vendor

        cmp     byte ptr es:[di], 0  ; any date of BIOS ?
        je      sysb_skp3          ; jump if not

        ; convert date to dd-mmm-yy format

        mov     si, offset biosd2
        mov     ax, es:[di+6]      ; get day
        mov     [si], ax
        mov     ax, es:[di]        ; get year
        mov     [si+7], ax
        mov     ax, es:[di+3]      ; get month
        xchg    al, ah
        sub     al, 30h            ; convert ones to 0-9
        cmp     ah, '1'            ; months 10-12?
        ja      sysb_skp3          ; jump if month value invalid
        jne     sysb_skp2          ; jump if months 1-9
        add     al, 10
sysb_skp2:
        dec     al                 ; al = 0-11
        mov     ah, 3
        mul     ah                 ; ax = al * 3
        mov     di, ax
        add     di, offset months
        mov     ax, [di]           ; get month text, 1st 2 chars
        mov     [si+3], ax
        mov     al, [di+2]         ; get 3rd month char
        mov     [si+5], al
        OUTMSG  biosd              ; output BIOS date string

sysb_skp3:
        pop     ax
        push    ax
        cmp     al, 1
        jb      sysb_skp5          ; unknown if LBA supported
        ja      sysb_skp4          ; jump if LBA supported
        OUTMSG  biosl2             ; no support
        jmp     sysb_skp5

sysb_skp4:
        OUTMSG  biosl1             ; LBA supported message


sysb_skp5:
        pop     ax
        cmp     ah, 0
        je      sysb_skp7          ; unknown Motherboard
        push    es
        push    ds
        pop     es
        mov     di, offset manuf2
        mov     si, offset where1  ; insert where motherboard made
        cmp     ah, 2
        jb      sysb_skp6
        mov     si, offset where2
        je      sysb_skp6
        mov     si, offset where3
sysb_skp6:
        call    xferstring         ; move string from es:[si] to [di]
        OUTMSG  manuf
        pop     es

sysb_skp7:
        cmp     byte ptr es:[bx], 0  ; any chipset string?
        je      sysb_skp8          ; jump if not

        mov     si, bx             ; chipset string
        mov     di, offset manufc2
        call    xferstring         ; move string from es:[si] to [di]
        OUTMSG  manufc

sysb_skp8:
        OUTMSG  crlf

; --- Find number of Diskette drives

syst_skp2:
        mov     dl, 0              ; drive 0 first
syst_loop2:
        push    dx
        call    dsktype            ; drive type in al
        cmp     al, 0
        je      syst_skp3          ; jump if no drive
        dec     al                 ; zero based
        mov     cx, offset drvend - offset drvtyps
        mul     cl                 ; ax = cl * al
        add     ax, offset drvtyps
        mov     si, ax
        mov     di, offset drvtext
        cld
        rep     movsb              ; xfer string
        OUTMSG  drv                ; display drive message
        pop     dx
        inc     dl                 ; handle next drive
        inc     [drvltr]           ; next drive letter
        cmp     dl, 4
        jb      syst_loop2
        jmp     syst_skp4

syst_skp3:
        pop     dx
        cmp     dl, 1              ; were we looking at a: or b:
        jae     syst_skp4          ; jump if there is one drive
        OUTMSG  no_drvs            ; no diskette drives

; --- now check hard drive information

syst_skp4:
        OUTMSG  blank_ln           ; insert blank line
        mov     dl, 80h            ; drive 0 first
syst_loop3:
        push    dx
        call    hdsktype           ; drive type in al
        cmp     al, 0
        jne     syst_skp5
        jmp     syst_skp6          ; jump if no drive

syst_skp5:
        push    ax
        mov     ax, cx             ; get cylinders
        mov     di, offset hcyldr
        mov     word ptr [di], '  '
        mov     word ptr [di+2], '  '
        mov     bl, 1              ; no left justification
        call    decw               ; insert cylinders (decimal)

        mov     ax, dx             ; get heads
        mov     hheads+3, ' '      ; blank from prior use
        mov     di, offset hheads
        mov     bl, 1              ; no left justification
        call    decw               ; insert heads (decimal)

        pop     ax                 ; sectors in ah
        push    ax
        mov     al, ah             ; get sectors
        xor     ah, ah
        mov     hsector+3, ' '     ; blank from prior use
        mov     di, offset hsector
        mov     bl, 1              ; no left justification
        call    decw               ; insert sectors (decimal)

        pop     ax                 ; sectors in ah
        push    ax
        mov     al, ah
        xor     ah, ah
        mul     dx                 ; dx:ax = heads * sectors
        mul     cx                 ; dx:ax = total bytes/512
        mov     cx, 11             ; divide to get Megabytes
syst_loop4:
        shr     dx, 1              ; shift right into carry
        rcr     ax, 1              ; rotate right with carry
        loop    syst_loop4
        mov     di, offset hsize
        xor     bl, bl             ; left justification
        call    decw               ; insert total size in MB
        mov     word ptr [di], 'M '  ; insert " MB   "
        mov     word ptr [di+2], ' B'
        mov     word ptr [di+4], '  '

        pop     ax                 ; get type back
        dec     al                 ; zero based
        mov     cx, offset hdrvend - offset hdrvtyp
        mul     cl                 ; ax = cl * al
        add     ax, offset hdrvtyp
        mov     si, ax
        mov     di, offset hdrvtxt
        cld
        rep     movsb              ; xfer string

        OUTMSG  hdrv               ; display hard drive message
        pop     dx
        inc     dl                 ; handle next drive
        inc     [hdrvnum]          ; next drive number
        cmp     dl, 88h            ; test for up to 8 drives
        jae     syst_skp7          ; exit if done
        jmp     syst_loop3

syst_skp6:
        pop     dx
        cmp     dl, 81h            ; looking at drive 0 ?
        jae     syst_skp7          ; jump if there is one drive
        OUTMSG  no_hdrvs           ; no disk drives

; --- Find the video display type, attribute and possible vendor

syst_skp7:
        OUTMSG  vidmsg
        call    video_type         ; get video type information
        mov     bx, offset vtypend - offset vtype
        mul     bl                 ; ax = bl * video type
        mov     dx, offset vtype
        add     dx, ax             ; ds:dx = video type string
        mov     ah, 9
        int     21h                ; display string
        mov     al, ch
        mov     bx, offset vcolore - offset vcolor
        mul     bl                 ; ax = bl * video attribute
        mov     dx, offset vcolor
        add     dx, ax
        mov     ah, 9
        int     21h                ; display attribute portion
        or      cl, cl             ; vendor string ? (cl = 1)
        jz      syst_done          ; jump if no vendor string

        cmp     byte ptr es:[si], 0
        je      syst_vendor        ; jump if no string
        mov     cx, 40             ; display a max of 40 chars
        mov     di, offset vidmstr
syst_vloop:
        mov     al, es:[si]        ; get string char
        or      al, al             ; end (zero)
        jz      syst_vendor
        mov     [di], al           ; transfer to output area
        inc     di
        inc     si
        loop    syst_vloop

syst_vendor:
        OUTMSG  vidmfg

syst_done:
        mov     ax,4C00h
        int     21h                ; exit with al return code
systype endp


;
;    SYSTEM TYPE DETECTION SUBROUTINE
;       Determine the type of system the software is running
;       on.
;
;       Called with:    nothing
;
;       Returns:        al = System type
;                             0 if PC (8088 based)
;                             1 if XT (8088 based)
;                             2 if PC convertible (8088 based)
;                             3 if PC jr (8088 based)
;                             4 other pre-80286 based machine
;                             8 if XT (80286 based)
;                            10h if AT or ISA
;                            20h if EISA
;                            40h if MCA
;
;       Regs used:      ax, bx
;                       eax, ebx (386 or later)
;
;       Subs called:    cpu386

sysvalue proc    near
        push    cx
        push    dx
        push    es

        call    far ptr cpu386     ; get the cpu type in al
        mov     cl, al             ; save cpu class

; Avoid directly reading BIOS ROM, because a few memory managers
; like 386MAX alter bytes at the end of the BIOS.

        push    cx                 ; save cpu number on stack
        mov     ah, 0C0h
        int     15h                ; get BIOS config data es:bx
        pop     cx
        jc      sys_skp1           ; jump if no config support
                                   ;   (old BIOS)
        mov     dl, es:[bx+2]      ; get model byte
        mov     dh, es:[bx+3]      ; get submodel byte

        mov     al, 40h            ; assume MCA
        test    byte ptr es:[bx+5], 2
        jnz     sys_Exit           ; exit if MCA
        jmp     sys_skp2

; we only get here on older PCs in which a memory manager
;  can not be run

sys_skp1:                          ; ok, get BIOS model directly
        mov     ax, 0F000h
        mov     es, ax             ; point into system BIOS
        mov     dx, es:[0FFFEh]    ; get model & submodel byte

; now use the model and submodel bytes im DX to determine machine

sys_skp2:
        xor     al, al             ; assume PC (al=0)
        cmp     dl, 0FFh
        je      sys_Exit           ; jump if PC
        inc     al                 ; assume XT (al=1)
        cmp     dl, 0FEh
        je      sys_Exit           ; jump if XT
        cmp     dl, 0FBh
        je      sys_Exit           ; jump if XT
        inc     al                 ; assume PC convertible (al=2)
        cmp     dl, 0F9h
        je      sys_Exit           ; jump if convertible
        inc     al                 ; assume PCjr (al=3)
        cmp     dl, 0FDh
        je      sys_Exit           ; jump if PCjr
        inc     al                 ; assume other pre-286 (al=4)
        cmp     cl, 2              ; cl=CPU type - pre-286 ?
        jb      sys_Exit           ; jump so
        ja      sys_skp3           ; jump if 386 or above

; possible a 286 XT - use the model and submodel bytes to
;  determine

        mov     al, 8              ; assumption for 286XT
        cmp     dx, 02FCh          ; model code for 286XT ?
        je      sys_exit           ; jump if so

; check if EISA system by looking for the "EISA" string at
;  address F000:FFD9

sys_skp3:
        mov     ax, 0F000h
        mov     es, ax
        mov     al, 10h            ; assume a standard AT/ISA
        cmp     word ptr es:[0FFD9h], 'IE'
        jne     sys_exit           ; jump if not EISA
        cmp     word ptr es:[0FFDBh], 'AS'
        jne     sys_exit           ; jump if not EISA
        mov     al, 20h            ; EISA machine
sys_Exit:
        pop     es
        pop     dx
        pop     cx
        ret
sysvalue endp


;
;    PCI DETECTION SUBROUTINE
;       Determine if the BIOS supports PCI, and if so, get the
;       PCI BIOS version.
;
;       Called with:    nothing
;
;       Returns:        carry = 0 if PCI BIOS present and:
;                         bx = version in BCD
;                       carry = 1 if no PCI BIOS support
;
;       Regs used:      ax, bx
;                       edx (386 or later)
;
;       Subs called:    cpu386

.386

pci_detect proc    near
        push    cx

        call    far ptr cpu386     ; get the cpu type in al
        cmp     al, 3              ; 386 or better ?
        jae     pci_check          ; jump if so
        stc                        ; set carry flag (no PCI)
        jmp     pci_exit

pci_check:
        mov     ax, 0B101h         ; function for PCI detect
        mov     edx, " PCI"        ; signifier
        int     1Ah                ; check if present

pci_exit:
        pop     cx
        ret
pci_detect endp

.8086


;
;    BIOS INFO SUBROUTINE
;       Determine the BIOS vendor and other key information
;
;       Called with:    al = system type (from sysvalue)
;
;       Returns:        al = BIOS
;                             0 unknown LBA support status
;                             1 no LBA support
;                             2 LBA support
;                       ah = Motherboard manufacturing location
;                             0 = unknown
;                             1 = outside USA (from AMI code)
;                             2 = by American Megatrends
;                             3 = in USA (from AMI code)
;                       dl = vendor value
;                             0 = unknown
;                             1 = American Megatrends
;                             2 = Award Software International
;                             3 = Phoenix Technologies
;                             4 = Chips & Technologies
;                             5 = Compaq
;                             6 = DTK
;                             7 = Eurosoft
;                             8 = Faraday (Western Digital)
;                             9 = Hewlett Packard
;                             10 = Landmark Research International
;                             11 = Microid Research
;                             12 = Olivetti
;                             13 = Quadtel
;                             14 = Toshiba
;                             15 = Western Digital
;                             16 = IBM
;                             17 = IBM (old)
;                       es:bx = Chipset string (ASCIIZ, 10 char max)
;                       es:si = Vendor string (ASCIIZ, 40 char max)
;                       es:di = BIOS Date (ASCIIZ, form yy-mm-dd)
;
;       Regs used:      ax, bx
;
;       Subs called:    find_string, compare_date


; table of strings, length of string appears first (31 chars max)
;   biosven strings must be in caps, and only symbols below 30h
;   allowed are spaces or an ampersand
;   Recommend AMI, Award, and Phoenix be placed first, since other
;   vendors have sometimes used one of these first three.
;   IBM should appear last, since many BIOSes include an IBM
;   string in the BIOS like "IBM Compatible".

biosven         db      19, 'AMERICAN MEGATRENDS'  ; near top
                db      5,  'AWARD'                ; near top
                db      7,  'PHOENIX'              ; near top
                db      20, 'CHIPS & TECHNOLOGIES'
                db      6,  'COMPAQ'
                db      3,  'DTK'
                db      4,  'ERSO'
                db      8,  'EUROSOFT'
                db      7,  'FARADAY'
                db      7,  'HEWLETT'
                db      8,  'LANDMARK'
                db      7,  'MICROID'
                db      5,  'MYLEX'
                db      8,  'OLIVETTI'
                db      7,  'QUADTEL'
                db      7,  'TOSHIBA'
                db      15, 'WESTERN DIGITAL'
                db      8,  'IBM CORP'             ; near end
                db      7,  'IBM 198'              ; near end
                db      0

biosc_string    db      10 dup (0)
biosv_string    db      40 dup (0)
biosd_string    db      '00/00/00', 0   ; date in yy/mm/dd form
motherboard     db      0               ; temp for motherboard
LBA_support     db      0               ; temp for LBA

ami_LBA_date    db      '94/07/25'      ; LBA on AMI BIOS date & later

; the following strings are used to look for vendor specific information
;       * = any char, # = digit 0-9

ami1_string     db      31, '##-####-******-########-######-' ; new style
ami2_string     db      15, '-####-######-K#'                 ; old style

award_string    db      22, '##/##/##-***-####-####'

find_date1      db      8, '##/##*##'
find_date2      db      8, '##-##-##'

bios_size       dw      0               ; bytes to look at in BIOS

biosinfo proc    near
        push    ds
        push    cs
        pop     ds

        mov     bx, 0F000h         ; assume bios starts at F000
        mov     cx, 0FFE0h         ; number of bytes to check
        cmp     al, 8              ; old PC/XT ?
        ja      biosi_skp1         ; jump if not
        mov     bx, 0FE00h         ; use smaller BIOS limit
        mov     cx, 01FE0h         ; number of bytes to check
biosi_skp1:
        mov     [bios_size], cx
        mov     es, bx
        mov     si, offset biosven
        mov     dx, 1              ; dl is BIOS vendor number

biosi_loop1:
        cmp     byte ptr [si], 0   ; no more strings to check?
        je      biosi_skp5

        call    find_string        ; see if string [si] appears
                                   ;   in the BIOS at es:0
        jc      biosi_skp3         ; jump if found
        inc     dx
        mov     al, [si]
        xor     ah, ah
        add     si, ax             ; move to next string
        inc     si
        jmp     biosi_loop1        ; try next vendor string

        ; vendor string found, start of string at es:bx

biosi_skp3:
        mov     si, offset biosv_string
        mov     cx, 39             ; string limit
biosi_loop2:
        mov     al, es:[bx]
        cmp     al, 20h            ; allows spaces
        jb      biosi_skp5
        je      biosi_skp4
        cmp     al, '&'            ; allow ampersand
        je      biosi_skp4
        cmp     al, 30h            ; skip some symbols
        jb      biosi_skp5
        cmp     al, 7Eh            ; skip chars > 7Eh
        ja      biosi_skp5
biosi_skp4:
        mov     [si], al           ; save character
        inc     si
        inc     bx
        loop    biosi_loop2

biosi_skp5:
        cmp     [biosv_string], 0  ; no string found?
        jne     biosi_skp6
        mov     dl, 0              ; set unknown flag

        ; now look for vendor specific information

biosi_skp6:
        cmp     dl, 1              ; AMI?
        jne     biosi_skp7         ; jump if not

        mov     cx, [bios_size]
        mov     si, offset ami1_string
        call    find_string        ; see if string [si] appears
                                   ;   in the BIOS, start at es:0
        jnc     biosi_ami1         ; jump if not found
        mov     al, es:[bx]        ; get motherboard byte
        add     bx, 24             ; point to date string
        jmp     biosi_ami2

biosi_ami1:
        mov     si, offset ami2_string
        call    find_string        ; see if string [si] appears
                                   ;   in the BIOS at es:0
        jnc     biosi_skp7         ; jump if not found
        mov     ax, [bx-4]         ; get 4 chipset chars
        mov     word ptr [biosc_string], ax
        mov     ax, [bx-2]
        mov     word ptr [biosc_string+2], ax
        mov     al, es:[bx+1]      ; get motherboard byte
        add     bx, 6              ; point to date string
biosi_ami2:
        sub     al, 2Fh            ; convert AMI code to our
        and     al, 0FEh           ;  motherboard ids
        shr     al, 1
        cmp     al, 3
        ja      biosi_ami3         ; skip save if invalid value
        mov     [motherboard], al  ; save

        ; covert AMI date at es:bx to yy-mm-dd form
biosi_ami3:
        mov     di, offset biosd_string
        mov     ax, es:[bx]
        mov     [di+3], ax         ; insert month
        mov     ax, es:[bx+2]
        mov     [di+6], ax         ; insert day
        mov     ax, es:[bx+4]
        mov     [di], ax           ; insert year

        ; AMI began LBA support in BIOSes 25-07-94 and later

        mov     [LBA_support], 1   ; no LBA assumed
        mov     di, offset ami_LBA_date
        mov     si, offset biosd_string
        call    compare_date       ; compare
        jnc     biosi_done2        ; jump if date earlier
        mov     [LBA_support], 2   ; LBA should be supported
biosi_done2:
        jmp     biosi_done

biosi_skp7:
        cmp     dl, 2              ; Award
        jne     biosi_skp8         ; jump if not

        mov     cx, [bios_size]
        mov     si, offset award_string
        call    find_string        ; see if string [si] appears
        jnc     biosi_skp8         ; jump if not found

        mov     ax, es:[bx+9]      ; get chipset chars
        mov     word ptr [biosc_string], ax
        mov     al, es:[bx+11]
        mov     [biosc_string+2], al

        mov     di, offset biosd_string
        mov     ax, es:[bx]
        mov     [di+3], ax         ; insert month
        mov     ax, es:[bx+3]
        mov     [di+6], ax         ; insert day
        mov     ax, es:[bx+6]
        mov     [di], ax           ; insert year
        jmp     biosi_done

        ; generic search for BIOS date

biosi_skp8:
        push    es
        mov     ax, 0FFFFh
        mov     es, ax
        mov     cx, 15             ; see if string in last 15 chars
        mov     si, offset find_date2
        call    find_string
        jc      biosi_date_found   ; jump if date found
        mov     si, offset find_date1 ; try alternate format
        call    find_string
        jc      biosi_date_found
        pop     es
        push    es
        mov     cx, [bios_size]    ; look over all of the BIOS
        call    find_string
        jc      biosi_date_found   ; jump if date found
        mov     si, offset find_date2
        call    find_string
        jnc     biosi_skp10        ; jump if date not found

biosi_date_found:
        mov     di, offset biosd_string
        mov     ax, es:[bx+6]
        mov     [di], ax           ; insert year

        ; check if date in US or international order

        mov     ax, es:[bx]        ; if US this is the month
        mov     cx, es:[bx+3]      ; if US this is the day
        xchg    al, ah
        cmp     ax, '12'           ; greater than 12, not US
        xchg    al, ah
        jbe     biosi_skp9         ; jump if seems to be US order
        xchg    ax, cx             ; change to US order
biosi_skp9:
        mov     [di+3], ax         ; insert month
        mov     [di+6], cx         ; insert day
biosi_skp10:
        pop     es

biosi_done:
        cmp     byte ptr [biosd_string], '8'    ; 1980s ?
        je      biosi_skp11                     ; if no, NO LBA
        cmp     byte ptr [biosd_string], '9'    ; 1990s ?
        jne     biosi_skp12                     ; if not, unknown
        cmp     byte ptr [biosd_string+1], '4'  ; 1994 or after ?
        jae     biosi_skp12                     ; if so, unknown
biosi_skp11:
        mov     [LBA_support], 1   ; no LBA in 1980-1993 BIOSs
biosi_skp12:
        mov     ah, [motherboard]
        mov     al, [LBA_support]
        mov     bx, offset biosc_string ; BIOS chipset id
        mov     si, offset biosv_string ; BIOS vendor
        mov     di, offset biosd_string ; BIOS date string
        push    ds
        pop     es
        pop     ds
        ret
biosinfo endp


;
;    FIND STRING SUBROUTINE
;       See if string (any case) ds:si appears in area es:bx
;
;       Called with:    ds:si = string to find (in caps)
;                                 first byte is length of string
;                                 '*' = don't care (any char ok)
;                                 '#' = any digit 0-9 ok
;                       es:0  = where to begin serch
;                       cx    = number of bytes to search
;
;       Returns:        carry = 1 if found
;                               es:bx = start of string
;                       carry = 0 if not found
;
;       Regs used:      ax
;
;       Internal:       bx set to -1 as flag of no match
;                          if first char matches, address saved in bx
;                       dx = number of bytes that match so far

find_string proc    near
        push    cx
        push    dx
        push    di
        push    si
        mov     bx, -1             ; bx = -1 as flag of no string
        xor     di, di             ; start at es:0
        mov     ah, byte ptr [si]  ; save length of compare string
        inc     si                 ; move to start of compare string

finds_loop1:
        cmp     di, cx             ; all bytes checked?
        jae     finds_nofind

        mov     al, es:[di]
        cmp     al, 40h            ; check if a symbol or letter
        jb      finds_skp0         ; jump if can not be letter
        and     al, 0DFh           ; convert to upper case
finds_skp0:
        cmp     byte ptr [si], '*' ; any character ok?
        je      finds_match        ; jump if so, consider a match
        cmp     byte ptr [si], '#' ; any digit 0-9 ok?
        jne     finds_skp0a        ; jump if not, must be full match
        cmp     al, '0'            ; digit ?
        jb      finds_skp2         ; jump if not
        cmp     al, '9'
        ja      finds_skp2         ; not a digit
        jmp     finds_match        ; is a digit 0-9

finds_skp0a:
        cmp     al, [si]
        jne     finds_skp2         ; not matched

finds_match:
        cmp     bx, -1             ; first char match ?
        jne     finds_skp1
        mov     bx, di             ; save start if so
finds_skp1:
        inc     di
        inc     si
        mov     dx, di
        sub     dx, bx             ; dx = bytes matched so far
        cmp     dl, ah             ; entire string matched?
        jb      finds_loop1        ; not yet
        jmp     finds_found        ; jump if so

        ; string does not match, so reset if partial string matched

finds_skp2:
        cmp     bx, -1             ; no matches at all, so continue
        je      finds_skp3
        sub     di, dx             ; reset string pointers
        sub     si, dx
        mov     bx, -1             ; set flag, no string found
finds_skp3:
        inc     di
        jmp     finds_loop1

finds_found:
        stc                        ; set carry, string found
        jmp     finds_exit

finds_nofind:
        clc                        ; clear carry, no string found

finds_exit:
        pop     si
        pop     di
        pop     dx
        pop     cx
        ret
find_string endp


;
;    COMPARE DATE STRINGS SUBROUTINE
;       See if string ds:di is equal to or later than date
;         at ds:si
;
;       Called with:    ds:si = BIOS date string in form yy/dd/mm
;                       ds:di = date string to compare
;
;       Returns:        carry = BIOS equal or later than compare
;                                 string
;                       carry = 0 BIOS is older
;
;       Regs used:      ax

compare_date proc    near
        cmp     byte ptr [si], '8' ; see if years 2000 - 2070
        jb      cdate_later        ; if so, assume later
        mov     cx, 8
cdate_loop:
        mov     al, [si]           ; get BIOS date
        cmp     al, [di]           ; is BIOS older?
        jb      cdate_older        ; jump if so
        inc     si
        inc     di
        loop    cdate_loop

cdate_later:
        stc                        ; BIOS date is same or newer
        ret

cdate_older:
        clc                        ; BIOS date is older
        ret
compare_date endp


;
;    DISKETTE DRIVE TYPE DETECTION SUBROUTINE
;       Determine the type of diskette drive
;
;       Called with:    dl = drive to check (0=a:, 1=b:, etc.)
;
;       Returns:        al = drive type
;                               0 = no drive
;                               1 = 360KB
;                               2 = 1.2MB
;                               3 = 720KB
;                               4 = 1.44MB
;                               5 = 2.88MB
;                               6 = 320KB  (obsolete)
;                               7 = unknown type
;
;       Regs used:      ax

dsktype proc    near
        push    bx
        push    cx
        push    dx
        push    es

        int     11h                ; equipment check
        test    ax, 1              ; any drives ?
        jz      dskt_none          ; jump if not
        and     al, 0C0h           ; get # of drives bits
        mov     cl, 2
        rol     al, cl             ; convert to # of drives
        cmp     al, dl             ; is drive available ?
        jb      dskt_none          ; jump if not

; first we'll try getting the drive parameters from the BIOS

        mov     ah, 8
        int     13h                ; get drive parameters
        jc      dskt_skp1          ; jump if failed
        mov     al, bl             ; get drive type
        cmp     al, 5              ; unknown type if > 5
        jbe     dskt_done
        mov     al, 7              ; set to unknown type
        jmp     dskt_done

; Older systems can't supply drive info, so use diskette table.
;   First the drive is reset.  This forces systems that change
;   the diskette parameter pointer for different drive types
;   to update the pointer to the correct drive type. Interrupt
;   vector 1Eh now points to the correct diskette parameter
;   table.  Only 360KB and 1.2MB drives should get here.

dskt_skp1:
        xor     ah, ah
        int     13h                ; reset drive
        xor     ax, ax
        mov     es, ax
        mov     bx, es:[1Eh*4]     ; get offset of table
        mov     ax, es:[1Eh*4+2]   ; get segment of table
        mov     es, ax             ; es:bx points to table
        mov     ah, es:[bx+4]      ; get sectors per track
        mov     al, 1              ; assume 360K
        cmp     ah, 9              ; 9 sectors per track ?
        je      dskt_done          ; jump if so
        inc     al
        cmp     ah, 15             ; 15 sectors per track
        je      dskt_done          ; jump if so
        mov     al, 6              ; unknown type
        cmp     ah, 8              ; 8 sectors per track
        je      dskt_done          ; jump if so
        inc     al                 ; al=7, unknown type
        jmp     dskt_done

dskt_none:
        xor     al, al
dskt_done:
        pop     es
        pop     dx
        pop     cx
        pop     bx
        ret
dsktype endp


;
;    HARD DISK DRIVE TYPE DETECTION SUBROUTINE
;       Determine if a hard drive is attached, its likely type,
;       and the capacity of that drive.
;
;       Called with:    ds = cs
;                       dl = drive to check, 80h=drive 0, etc.
;
;       Returns:        al = drive type
;                               0 = no drive or no controller
;                               1 = XT type controller
;                               2 = AT or IDE type controller
;                               3 = SCSI type controller
;                               4 = PS/2 ESDI type controller
;                               5 = unknown controller type
;                               6 = IDE controller, LBA active
;                       cx = total number of cylinders (includes
;                              the diagnostic cylinder)
;                       dx = total number of heads (1-256)
;                       ah = number of 512 byte sectors (1-63)
;       Regs used:      ax

; local data for routine

hd_parm_seg     dw      0               ; pointer to disk
hd_parm_off     dw      0               ;  parameter table
hd_cylinder     dw      0               ; temp # from table
ESDIbuf         db      1024 dup (0)    ; ESDI info buffer


hdsktype proc   near
        push    bx
        push    es
        xor     bp, bp             ; used for temp drive type
        cmp     dl, 80h            ; at least 80h ?
        jb      hdsk_skp1          ; exit if not

; first we will get the number of drives attached

        push    dx
        mov     dl, 80h            ; ask for drive 0
        mov     ah, 8
        int     13h                ; read disk drive parameters
        pop     ax
        add     dl, 7Fh
        cmp     al, dl             ; drive for this number ?
        jbe     hdsk_skp2
hdsk_skp1:
        jmp     hdsk_none          ; jump if out of range

; now determine the controller/disk type

hdsk_skp2:
        inc     bp                 ; bp=1 assume XT type drive
        mov     dl, al             ; dl = drive number
        push    dx
        mov     ah, 15h
        int     13h
        pop     dx
        cmp     ah, 1              ; invalid request status ?
        je      hdsk_skp4          ; if so, XT type & exit
        inc     bp                 ; set type to AT
        cmp     ah, 3              ; confirm valid hard disk
        je      hdsk_skp3
        jmp     hdsk_none          ; exit if not

; let's check if it's a drive which does not use the drive
;   parameter table like SCSI and some ESDI drives.

hdsk_skp3:
        inc     bp                 ; assume SCSI
        cmp     dl, 87h            ; if above 87h, assume SCSI
        jbe     hdsk_skp5
hdsk_skp4:
        jmp     hdsk_info          ; jump if so

hdsk_skp5:
        mov     bx, 4*41h          ; assume ptr to vector 41h
        cmp     dl, 80h            ; drive 0 ?
        je      hdsk_skp6          ; jump if so
        mov     bx, 4*46h          ; pointer to vector 46h
hdsk_skp6:
        xor     ax, ax
        mov     es, ax
        mov     si, es:[bx]        ; offset of parameter table
        mov     ax, es:[bx+2]      ; segment of parameter table
        mov     [hd_parm_seg], ax  ; save pointer
        mov     [hd_parm_off], si
        mov     es, ax             ; es:si ptr to table

; we now have a pointer to the disk parameter table, but there
;   are two types!  Check if string at offset 2 is not 'IBM ', then
;   use normal table (cylinder at offset 0).  Otherwise use
;   old PS/2 style, where cylinder is at offset 19h.

        cmp     word ptr es:[si+2], 'BI' ; string 'IBM ' ?
        jne     hdsk_skp7          ; jump if not
        cmp     word ptr es:[si+4], ' M'
        jne     hdsk_skp7
        add     si, 19h            ; adjust to PS/2 style
hdsk_skp7:
        mov     ax, es:[si]        ; get cylinders from table
        mov     [hd_cylinder], ax  ; save
        cmp     ax, 0              ; invalid # of cylinders ?
        jne     hdsk_skp8          ; jump if ok (non-zero)
        jmp     hdsk_info          ; we will assume SCSI

hdsk_skp8:
        dec     bp                 ; assume AT type (2)
        push    dx
        mov     ah, 8
        int     13h                ; get disk parameters
        call    hdconvert          ; convert into useful values
        cmp     ax, 1021           ; near end of IDE limit?
        jb      hdsk_skp8a         ; jump if not
        cmp     [hd_cylinder], 1020
        jae     hdsk_skp8b         ; jump if ok
        jmp     hdsk_skp9          ; unlikely AT or IDE type

hdsk_skp8a:
        cmp     ax, [hd_cylinder]  ; are they within 3 cylinders?
        ja      hdsk_skp9          ; jump if not
        add     ax, 3
        cmp     ax, [hd_cylinder]  ; are they within 3 cylinders?
        jb      hdsk_skp9          ; jump if not

        ; is AT type, check if LBA active (heads > 16)

hdsk_skp8b:
        cmp     dx, 16             ; more than 16 heads ?
        jbe     hdsk_info2         ; jump if not
        mov     bp, 6              ; set to AT type with LBA on (6)
hdsk_info2:
        pop     dx
        jmp     hdsk_info          ; if so, likely AT type

; Not likely AT type, since cylinders do not match up!
;  First we'll try an older Future Domain SCSI test

; Future Domain SCSI test - Interrupt 13h, function 18h will
;  return invalid command with DL >= 80h, if no Future Domain
;  SCSI card is present

hdsk_skp9:
        pop     dx
        inc     bp                 ; assume SCSI
        push    dx
        mov     ah, 18h
        int     13h                ; Future Domain SCSI ?
        pop     dx
        jc      hdsk_skp10         ; jump if not
        cmp     ax, 4321h          ; confirmation number
        jne     hdsk_skp10
        jmp     hdsk_info

; Now check if possible PS/2 ESDI drive

hdsk_skp10:
        push    cs
        pop     es
        push    dx
        mov     ax, 1B0Ah
        mov     bx, offset ESDIbuf
        int     13h                ; Get ESDI config
        pop     dx
        jc      hdsk_skp11         ; jump if not PS/2 ESDI
        cmp     bx, offset ESDIbuf
        jne     hdsk_skp11
        mov     ax, cs
        mov     bx, es
        cmp     ax, bx             ; is CS = ES ?
        jne     hdsk_skp11         ; jump if not
        inc     bp
        jmp     hdsk_info

; General SCSI test (not drive specific)

hdsk_skp11:
        xor     ax, ax
        mov     es, ax
        cmp     word ptr es:[4*4Fh+2], 0  ; vector valid ?
        je      hdsk_skp12         ; jump if not
        push    dx
        mov     ax, 8200h
        mov     cx, 8765h
        mov     dx, 0CBA9h
        int     4Fh                ; check for SCSI CAM
        cmp     dx, 5678h
        pop     dx
        jne     hdsk_skp12         ; jump if not
        cmp     ah, 0
        jne     hdsk_skp12         ; jump if not
        cmp     cx, 9ABCh
        jne     hdsk_skp12         ; jump if not
        jmp     hdsk_info

hdsk_skp12:
        mov     bp, 5              ; Indicate unknown
                                   ;  controller type

; now get the disk parameter for the specified drive

hdsk_info:
        mov     ah, 8
        int     13h                ; read disk drive parameters
        call    hdconvert          ; convert to useful value
        xchg    cx, ax             ; put in proper registers
        mov     bl, al             ; sectors
        mov     ax, bp             ; get drive type
        mov     ah, bl             ; insert sectors
        jmp     hdsk_exit

hdsk_none:
        xor     al, al             ; no drive with this number
        xor     cx, cx
        xor     dx, dx
hdsk_exit:
        pop     es
        pop     bx
        ret
hdsktype endp


;
;    HARD DISK CONVERT
;
;       The cylinder number is in cx.  This routine
;       properly combines them back into a 10 bit
;       cylinder number in AX. The number is increased by
;       one for the diagnostic cylinder, and by one more, since
;       the number is zero based. The upper two bits in CL are
;       cleared to get the sector number.  One is added
;       to the head number, since it is zero based.
;
;       Called with:    ch = lower 8 bits of cylinder number
;                       cl upper two bits are
;                             bits 9 & 8 of cylinder
;                          lower six bits is sector number
;                       dh head number
;
;       Returns:        ax = total combined cylinders
;                       cl = sector number
;                       dx = total heads (1-256)
;
;       Regs used:      cx, dh

hdconvert proc   near
        mov     ax, cx             ; get fragmented cylinder #
        rol     al, 1
        rol     al, 1
        and     al, 3
        xchg    al, ah             ; ax = # of cylinders
        and     cx, 3Fh            ; mask for sector number
        mov     dl, dh
        xor     dh,dh
        inc     dx                 ; adjust for head count
        ret
hdconvert endp


;
;    VIDEO TYPE DETECT
;
;       Find the video type, type attributes and possible
;       vendor string.  This routine assumes the display is
;       in a text mode to determine the attribute byte CL.
;
;       The attributes option indicates what attributes a
;       program should use, color, monochrome, or grayscales.
;
;       Called with:    nothing
;
;       Returns:        al = video type
;                             0 = MDA
;                             1 = HGA
;                             2 = CGA
;                             3 = MCGA
;                             4 = EGA
;                             5 = VGA
;                             6 = SVGA
;                             7 = XGA
;                             8 = VESA XGA
;                       ch = attribute type
;                             0 = color
;                             1 = monochrome
;                             2 = gray scale (some MCGA or VGA+)
;                       cl = vendor string present in es:bx
;                             0 = no vendor string
;                             1 = vendor string
;                       es:si = vendor string, zero terminated
;                               (if cl = 1)
;
;       Regs used:      ax, cx, si, es

infobuf db      256 dup (0)        ; buffer for video info
hercstr db      'Hercules', 0      ; Vendor string if Hercules

video_type proc   near
        push    bx
        push    dx
        push    di
        push    bp
        mov     bp, 4              ; bp = temp video type, 4=EGA

; --- check if EGA or later using get video information function

        mov     ah, 12h            ; get video info function
        mov     bh, 5Ah            ; test value
        mov     bl, 10h            ; subfunction EGA+ info
        int     10h
        cmp     bh, 1              ; must be 0, color or 1, mono
        ja      below_EGA          ; jump if not EGA+

; --- it is an EGA or later, so now test for VGA

        push    bx                 ; save color info for later
        mov     ax, 1A00h          ; get display code
        int     10h
        cmp     al, 1Ah            ; is function supported ?
        je      vid_VGA            ; if so, at least a VGA
        jmp     type_found         ; jump if not (must be EGA)

; --- at least a VGA, now test for SVGA

vid_VGA:
        inc     bp                 ; assume VGA (5)
        push    cs
        pop     es
        mov     di, offset infobuf ; buffer for video info
        mov     ax, 4F00h          ; return SVGA info
        int     10h
        cmp     al, 4Fh            ; is function supported ?
        jne     XGA_test           ; jump if not SVGA
        inc     bp                 ; assume SVGA (6)

; --- at least a VGA/SVGA, now test for XGA/VESA XGA

XGA_test:
        mov     ax, 1F00h          ; get XGA information size
        int     10h
        cmp     al, 1Fh            ; is function supported ?
        jne     type_found         ; jump if not, is VGA or SVGA
        mov     bp, 7              ; set to XGA

; --- at least a XGA, now test for VESA XGA

        mov     di, offset infobuf ; buffer for video info
        mov     ax, 4E00h          ; return VESA XGA info
        int     10h
        cmp     ax, 004Eh          ; is function supported ?
        jne     type_found         ; jump if not
        inc     bp                 ; VESA XGA (8)
        jmp     type_found

; --- arrives here if adapter is below an EGA

below_EGA:
        mov     bp, 3              ; assume MCGA (3)
        mov     ax, 1A00h          ; get display code
        int     10h
        cmp     al, 1Ah            ; is function supported ?
        jne     vid_not_MCGA       ; jump if not MCGA
        mov     ah, 0Fh
        int     10h                ; get video mode
        mov     cx, 100h           ; assume mono, no vendor
        cmp     al, 7
        je      vid_chk_gray       ; if mono, check gray
        mov     ch, 0              ; looks like color!
        je      vid_chk_gray       ; check gray scales

; --- not MCGA, test for CGA or MDA/HGA

vid_not_MCGA:
        dec     bp                 ; assume CGA (2)
        mov     ah, 0Fh            ; get video mode
        int     10h
        cmp     al, 7              ; mode 7 monochrome ?
        je      vid_mono_type      ; jump if so
        xor     cx, cx             ; return color, no vendor
        jne     vid_mono_type
        jmp     vid_mono_chk       ; must be CGA

; --- Must be MDA or HGA, so find out which.  The HGA
;     (Hercules Graphics Adapter) toggles an undefined bit on
;     the MDA. Check to see ifthis bit changes state 10 times
;     or more.

vid_mono_type:
        dec     bp                 ; assume HGA (1)
        xor     bl, bl             ; start count at 0
        mov     dx, 3BAh           ; status port on HGA/MDA
        xor     ah, ah             ; ah used for prior status
        mov     cx, 0FFFFh         ; test for a long time
vid_loop:
        in      al, dx             ; read status port
        IODELAY
        and     al, 80h            ; isolate HGA toggle bit
        cmp     al, ah             ; has it changed ?
        je      vid_no_toggle      ; jump if not
        inc     bl                 ; bit changed, increment
        cmp     bl, 10             ; more than 10 toggles ?
        jae     vid_herc           ; if so, it is a Hercules card
vid_no_toggle:
        loop    vid_loop           ; read again until cx 0

; --- falls through if MDA (bit does not toggle)

        dec     bp                 ; set to MDA (0)
        mov     cx, 100h           ; mono attribute, no vendor
        jmp     vid_exit

; --- adapter is Hercules type

vid_herc:
        mov     cx, 101h           ; mono attribute, vendor ok
        push    cs
        pop     es
        mov     si, offset hercstr ; set string to Hercules
        jmp     vid_exit

; --- For MCGA/EGA/VGA/XGA put the attribute type in CL

type_found:
        pop     bx                 ; get attribute type (0 or 1)
        mov     ch, bh
vid_chk_gray:
        mov     ax, 1A00h          ; read display code
        int     10h
        cmp     al, 1Ah            ; check if supported
        jne     vid_string         ; jump if can't be gray scale
        cmp     bl, 7              ; gray scale monitor?
        je      vid_is_gray        ; jump if so
        cmp     bl, 0Bh            ; gray scale monitor?
        je      vid_is_gray        ; jump if so
        cmp     bp, 5              ; VGA or later ?

        mov     ax, 40h            ; BIOS data area
        mov     es, ax
        test    byte ptr es:[89h], 2  ; gray scale summing on ?
        jz      vid_string         ; not gray scale
vid_is_gray:
        mov     ch, 2              ; set ch to gray scale

vid_string:
        xor     cl, cl             ; assume no vendor string
        cmp     bp, 6              ; SVGA ?
        je      vid_skp1           ; jump if so
        cmp     bp, 8              ; VESA XGA ?
        jne     vid_mono_chk       ; jump if not
vid_skp1:
        mov     di, offset infobuf ; get buffer of SVGA info
        mov     si, cs:[di+6]      ; get offset and segment to
        mov     ax, cs:[di+8]      ;   the vendor string
        mov     es, ax             ; es:bx points to string
        inc     cl                 ; vendor string valid

; The last check is made to see if video mode 2 is set,
; indicating monochrome attributes should be used (if we
; haven't already detected monochrome operation)

vid_mono_chk:
        cmp     ch, 0              ; set to color ?
        jne     vid_exit           ; jump if not
        mov     ah, 0Fh
        int     10h                ; get video mode
        and     al, 7Dh
        cmp     al, 0              ; video mode 0 or 2 ?
        jne     vid_exit           ; jump if not
        mov     ch, 1              ; use monochrome attributes

vid_exit:
        mov     ax, bp             ; return adapter type in al
        pop     bp
        pop     di
        pop     dx
        pop     bx
        ret
video_type endp


;
;    CPU 386 IDENTIFICATION SUBROUTINE
;       Identify the CPU type, from 8088 to 386+.  This is
;       subset of the more extensive CPUVALUE program.  It is
;       used when identification of CPUs above the 386 is
;       not necessary (i.e. 32-bit support or not)
;
;       Called with:    nothing
;
;       Returns:        al = CPU type
;                             0 if 8088/8086 or V20/V30
;                             1 if 80186/80188
;                             2 if 80286
;                             3 if 80386 or better
;
;       Regs used:      ax, bx
;                       eax (32-bit CPU only)
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler

.8086   ; all instructions 8088/8086 unless overridden later

cpu386  proc    far
        push    cx
        push    dx
        push    ds
        push    es

; 8088/8086 test - Use rotate quirk - All later CPUs mask the CL
;   register with 0Fh, when shifting a byte by cl bits.  This
;   test loads CL with a large value (20h) and shifts the AX
;   register right.  With the 8088, any bits in AX are shifted
;   out, and becomes 0.  On all higher level processors, the
;   CL value of 20h is anded with 0Fh, before the shift.  This
;   means the effective number of shifts is 0, so AX is
;   unaffected.

        mov     cl, 20h            ; load high CL value
        mov     ax, 1              ; load a non-zero value in AX
        shr     ax, cl             ; do the shift
        cmp     ax, 0              ; if zero, then 8088/86
        jne     up186              ; jump if not 8088/86
        jmp     uP_Exit

; 80186/80188 test - Check what is pushed onto the stack with a
;   PUSH SP instruction.  The 80186 updates the stack pointer
;   before the value of SP is pushed onto the stack.  With all
;   higher level processors, the current value of SP is pushed
;   onto the stack, and then the stack pointer is updated.

up186:
        mov     bx, sp             ; save the current stack ptr
        push    sp                 ; do test
        pop     ax                 ; get the pushed value
        cmp     ax, bx             ; did SP change ?
        je      up286              ; if not, it's a 286+
        mov     ax, 1              ; set 80186 flag
        jmp     uP_Exit

; 80286 test A - We'll look at the top four bits of the EFLAGS
;   register.  On a 286, these bits are always zero.  Later
;   CPUs allow these bits to be changed.  During this test,
;   We'll disable interrupts to ensure interrupts do not change
;   the flags.

up286:
        cli                        ; disable interrupts
        pushf                      ; save the current flags

        pushf                      ; push flags onto stack
        pop     ax                 ; now pop flags from stack
        or      ax, 0F000h         ; try and set bits 12-15 hi
        push    ax
        popf                       ; set new flags
        pushf
        pop     ax                 ; see if upper bits are 0

        popf                       ; restore flags to original
        sti                        ; enable interrupts
        test    ax, 0F000h         ; were any upper bits 1 ?
        jnz     up386plus          ; if so, not a 286

; 80286 test B - If the system was in V86 mode, (386 or higher)
;   the POPF instruction causes a protection fault, and the
;   protected mode software must emulate the action of POPF. If
;   the protected mode software screws up, as occurs with a
;   rarely encountered bug in Windows 3.1 enhanced mode, the
;   prior test may look like a 286, but it's really a higher
;   level processor. We'll check if the protected mode bit is
;   on.  If not, it's guaranteed to be a 286.

.286P                              ; allow a 286 instruction
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      is286              ; jump if not (must be 286)

; 80286 test C - It's very likely a 386 or greater, but it is
;   not guaranteed yet.  There is a small possibility the system
;   could be in 286 protected mode so we'll do one last test. We
;   will try out a 386 unique instruction, after vectoring the
;   bad-opcode interrupt vector (int 6) to ourselves.

        call    hook_int6          ; do it!
        mov     [badoff], offset upbad_op3  ; where to go if bad
.386
        xchg    eax, eax           ; 32 bit nop (bad on 286)

        call    restore_int6       ; restore vector
        jmp     up386plus          ; only gets here if 386
                                   ;  or greater!

; Interrupt vector 6 (bad opcode) comes here if system is a
;   80286 (assuming the 286 protected mode interrupt 6 handler
;   will execute the bad-opcode interrupt).

upbad_op3:
        call    restore_int6
is286:
        mov     ax, 2              ; set 80286 flag
        jmp     uP_Exit

up386plus:
        mov     ax, 3              ; 32-bit CPU (386 or later)

up_Exit:
        pop     es
        pop     ds
        pop     dx
        pop     cx
        ret
cpu386  endp
.8086                              ; return to 8086 instructions


;
;    HOOK INTERRUPT 6
;       Save the old interrupt 6 vector and replace it with
;       a new vector to the bad_op_handler.  Vectors are handled
;       directly without using DOS.
;
;       Called with:    nothing
;
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Regs used:      none

hook_int6 proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     ax, es:[6*4]       ; get offset of int 6
        mov     cx, es:[6*4+2]     ; get segment
        mov     es:[6*4], offset bad_op_handler
        mov     word ptr es:[6*4+2], seg bad_op_handler
        sti                        ; enable interrupts
        mov     [old_int6_seg], cx ; save original vector
        mov     [old_int6_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_int6 endp


;
;    RESTORE INTERRUPT 6
;       Restore the previously saved old interrupt 6 vector.
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Returns:        vector restored
;
;       Regs used:      none

restore_int6 proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_int6_seg] ; get original vector
        mov     dx, [old_int6_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[6*4], dx       ; restore original int 6
        mov     es:[6*4+2], cx
        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_int6 endp


;
;    BAD OFFSET INTERRUPT HANDLER
;       If a bad opcode occurs (80286 or later) will come here.
;       The saved BADOFF offset is used to goto the routine
;       previously stored in BADOFF.
;
;       In a few cases, it is also used for double faults. A few
;       instructions (RDMSR & WRMSR) can issue a double fault if
;       not supported, so well come here as well.
;
;       Called with:    cs:[badoff] previously set
;
;       Returns:        returns to address stored in badoff


bad_op_handler proc far
        push    ax
        push    bp
        mov     ax, cs:[badoff]
        mov     bp, sp
        mov     ss:[bp+4], ax      ; insert new return offset
        pop     bp
        pop     ax
        iret
bad_op_handler endp


;
;    DECW
;       Convert the hex number in ax into decimal 1 to 5 ascii
;       characters and insert into [di].  Increment di ptr.  The
;       leading zeros are suppressed.
;
;       Called with:    ax = input hex number
;                       di = pointer where to store characters
;                       bl = 0 for left justification
;                            1 for no justification
;
;       Returns:        word converted to ascii at [di]
;
;       Regs used:      bx
;
;       Subs called:    hex2ascii

decw    proc    near
        push    ax
        push    cx
        push    dx
        cmp     ax, 0              ; check for zero
        jne     decskip0           ; jump if not
        mov     al, 4              ; if justify, make ax = 0
        mul     bl                 ;    no justify, ax = 4
        add     di, ax             ; move pointer
        mov     byte ptr [di], '0' ; put up ascii zero
        jmp     decskip15          ; done !

decskip0:
        xor     cl, cl             ; temp flag, 0=suppression on
        mov     ch, bl             ; save flag (0=left justify)
        xor     dx, dx             ; zero
        mov     bx, 10000
        div     bx                 ; (hex)/10000
        cmp     al, 0              ; 10000's ?
        je      decskip1           ; jump if zero
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1000's digit in
        jmp     decskip2
decskip1:
        cmp     ch, 0              ; left justify ?
        je      decskip3           ; jump if so
decskip2:
        inc     di
decskip3:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 1000
        div     bx                 ; (hex)/1000
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip3a          ; jump if not
        cmp     al, 0              ; 1000's ?
        je      decskip4           ; jump if zero
decskip3a:
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1000's digit in
        jmp     decskip5
decskip4:
        cmp     ch, 0              ; left justify ?
        je      decskip6           ; jump if so
decskip5:
        inc     di
decskip6:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 100
        div     bx                 ; (remainder in dx)/100
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip7           ; jump if not
        cmp     al, 0              ; zero ?
        je      decskip8           ; suppress zero
decskip7:
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 100's digit in
        jmp     decskip9
decskip8:
        cmp     ch, 0              ; left justify ?
        je      decskip10          ; jump if so
decskip9:
        inc     di
decskip10:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 10
        div     bx                 ; (remainder in dx)/10
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip11          ; jump if not
        cmp     al, 0              ; zero ?
        je      decskip12          ; suppress zero
decskip11:
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 10's digit in
        jmp     decskip13
decskip12:
        cmp     ch, 0              ; left justify ?
        je      decskip14          ; jump if so
decskip13:
        inc     di
decskip14:
        mov     al, dl             ; get 1's digit (remainder)
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1's digit in output
decskip15:
        inc     di
        pop     dx
        pop     cx
        pop     ax
        ret
decw    endp


;
;   HEX2ASCII
;       Convert the hex number in al into two ascii characters
;
;       Called with:    al = input hex number
;
;       Returns         bx = converted digits in ascii
;
;       Regs Used:      al, bx

hex2ascii       proc    near
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al             ; bx has two ascii bytes
        ret
hex2ascii       endp


;
;   TRANSFER STRING SUBROUTINE
;       Copy the ASCIIZ string from ES:SI to DS:DI, without the
;       zero.
;
;       Called with:    es:si = source string
;                       ds:di = destination string
;
;       Returns         string transferred
;
;       Regs Used:      di, si

xferstring     proc    near
        push    ax
xfers_loop:
        mov     al, es:[si]
        cmp     al, 0
        je      xfers_exit
        mov     [di], al
        inc     di
        inc     si
        jmp     xfers_loop

xfers_exit:
        pop     ax
        ret
xferstring endp


cseg    ends

;================================================== stack ======

stacka  segment para stack

        db      192 dup (0)

stacka  ends

        end     start



