WITH Tables_Generic;
WITH Tables_Generic.Backup;
WITH Dates;
WITH Dates.IO;
WITH Debugging_Support;
USE  Debugging_Support;
PROCEDURE Test_Backup IS
------------------------------------------------------------------------
--| Demonstration of generic backup/restore package.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: January 1996                                     
------------------------------------------------------------------------

  -- set up simple name/birthday record
  SUBTYPE NameType IS String(1..10);

  TYPE BirthRecord IS RECORD
    Name: NameType;
    Birthday: Dates.Date;
  END RECORD;

  -- The next two subprograms satisfy the KeyOf and Visit
  -- parameters of Tables_Generic

  FUNCTION KeyOf (Item: BirthRecord) RETURN NameType IS
  BEGIN
    RETURN Item.Name;
  END KeyOf;

  PROCEDURE Put(Item: IN BirthRecord) IS
  BEGIN
    Ada.Text_IO.Put(Item => Item.Name);
    Dates.IO.Put(Item => Item.Birthday);
    Ada.Text_IO.New_Line;
  END Put;

  -- Now we can instantiate Tables_Generic

  PACKAGE BirthdayTables IS 
    NEW Tables_Generic
      (Element => BirthRecord,
       KeyType => NameType,
       Capacity => 50,
       Visit => Put);

  -- We need file-oriented Get and Put to satisfy Tables_Generic.Backup

  PROCEDURE Get(File: IN Ada.Text_IO.File_Type; Item: OUT BirthRecord) IS
  BEGIN
    Ada.Text_IO.Get(File => File, Item => Item.Name);
    Dates.IO.Get(File => File, Item => Item.Birthday);
    Ada.Text_IO.Skip_Line(File => File);
  END Get;

  PROCEDURE Put(File: IN Ada.Text_IO.File_Type; Item: IN BirthRecord) IS
  BEGIN
    Ada.Text_IO.Put(File => File, Item => Item.Name);
    Dates.IO.Put(File => File, Item => Item.Birthday);
    Ada.Text_IO.New_Line(File => File);
  END Put;

  -- Now we can instantiate the backup package. Note that we refer to
  -- the instance BirthdayTables, not the generic Tables_Generic.

  PACKAGE BirthdayBackup IS
    NEW BirthdayTables.Backup;  -- note - instance used, not generic

  Friends: BirthdayTables.TableType;

BEGIN -- Test_Backup

  Debugging_Support.SetDebug(WhichWay => On);

  -- The file birthdays.new should be a copy of birthdays.dat

  BirthdayBackup.Restore(T => Friends, FileName => "birthdays.dat");
  BirthdayBackup.Save(T => Friends, FileName => "birthdays.new");

END Test_Backup;
