/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdlib.h>             
#include <stdio.h>
#include <jlib.h>

JINLINE int JLIB_do_intersect (int x1, int y1, UBYTE *r1, int x2, int y2, UBYTE *r2);
int global_biggest_rle;

/*+----------------------------------------------------------------------+*/
/*|sprite_init - intitialises the sprite system.                         |*/
/*+----------------------------------------------------------------------+*/
sprite_system *sprite_init (int max_sprites, int max_frames)
{
   sprite_system *tssp;
   int counter;

   JLIB_ENTER("sprite_init");

#ifndef JLIB_PRODUCTION
   if ((max_sprites == 0) && (max_frames == 0)){
       jlib_exit(jlib_msg(JLIB_ENULL));
   }

   if ((max_sprites < 0) || (max_frames < 0)){
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* create space for sprite system record */
   if ((tssp = (sprite_system *) malloc (sizeof (sprite_system))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }

   /* fill in max sprite info */
   tssp->no_sprites = max_sprites;

   JLIB_SPRINTF("%d sprites supported.",tssp->no_sprites);

   /* create 'is active' array */
   if(SPR_MAX_SPRITES(tssp) == 0){
      tssp->active_sprites =NULL;
   }
   else{
      int ns = SPR_MAX_SPRITES(tssp);

      if ((tssp->active_sprites = (UBYTE *) malloc(ns *sizeof (UBYTE))) == NULL){
          jlib_exit(jlib_msg(JLIB_EMALLOC));
      }
      /* turn 'em all off */
      MEM_STORE_LONG(tssp->active_sprites, 0, ns);
   }

   tssp->no_active = 0;

   /* create 'sprite' array */
   if(SPR_MAX_SPRITES(tssp)==0){
      tssp->sprites =NULL;
   }
   else{
      if ((tssp->sprites =                                                       
         (sprite_record **)malloc(SPR_MAX_SPRITES(tssp)*sizeof(sprite_record *)))==NULL){
         jlib_exit(jlib_msg(JLIB_EMALLOC));
      }
   }

   tssp->no_frames = max_frames;
   tssp->number_loaded = 0;
   
   JLIB_SPRINTF("%d frames supported.",tssp->no_frames);

   /* create 'sprite data' array */
   if ((tssp->sprite_data =
        (sprite_data_rec **) malloc (SPR_MAX_FRAMES (tssp) * sizeof (sprite_data_rec *))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }

   /* NULL them all */
   for(counter=0;counter<SPR_MAX_FRAMES(tssp);counter++){
       tssp->sprite_data[counter] = NULL;
   }

   /* initialise the sprites */
   for(counter=0;counter<SPR_MAX_SPRITES(tssp);counter++){
        if((tssp->sprites[counter]=(sprite_record *)calloc(sizeof(sprite_record),1))==NULL){
          jlib_exit(jlib_msg(JLIB_EMALLOC));
        }

        tssp->sprites[counter]->mode = SPR_MODE_WHOLE;
        tssp->sprites[counter]->animframes = NULL;

        if((tssp->sprites[counter]->buffer=(UBYTE *)malloc(SPR_MAX_X*SPR_MAX_Y*sizeof(UBYTE)))==NULL){
          jlib_exit(jlib_msg(JLIB_EMALLOC));
        }

        JLIB_SPRINTF("Init Sprite %d OK.",counter);
     }

   tssp->biggest_rle=0;  /* no loaded sprites yet */

   JLIB_LEAVE;
   return tssp; /* give back the handle */
}


/*+----------------------------------------------------------------------+*/
/*|sprite_optimise mem - decrease memory requirements of sprites.        |*/
/*+----------------------------------------------------------------------+*/
void sprite_optimise_mem(sprite_system * ssys)
{
 int i;
 
  JLIB_ENTER("sprite_optimise_mem");

#ifndef JLIB_PRODUCTION
  if (ssys == NULL) {
      jlib_exit(jlib_msg(JLIB_ENULL));
  }
#endif
 
  for(i=0;i<SPR_MAX_SPRITES(ssys);i++){
      if(ssys->sprites[i]->buffer != NULL){
         free(ssys->sprites[i]->buffer); 
         
         if (ssys->biggest_rle!=0){
             if ((ssys->sprites[i]->buffer =
                (UBYTE *) malloc (ssys->biggest_rle*sizeof (UBYTE))) == NULL){
                 jlib_exit(jlib_msg(JLIB_EMALLOC));
             }
          }
          else{
             if ((ssys->sprites[i]->buffer =
                (UBYTE *) malloc (SPR_MAX_X*SPR_MAX_Y*sizeof (UBYTE))) == NULL){
                 jlib_exit(jlib_msg(JLIB_EMALLOC));
             }
          }
      }
  }

  JLIB_LEAVE;
}

/*+----------------------------------------------------------------------+*/
/*|sprite_turn_on - turn on a sprite                                     |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_turn_on (sprite_system * spr_sys, int snum)
{
   JLIB_ENTER("sprite_turn_on");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* set the sprite as active if it isn't allready */
   if(spr_sys->active_sprites[snum] == 0){
      spr_sys->active_sprites[snum] = 1;
      spr_sys->no_active++;
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|sprite_set_an_frame - set a sprites animation frame                   |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_set_an_frame (sprite_system * ssys, int snum, int frame)
{
   JLIB_ENTER("sprite_set_an_frame");

#ifndef JLIB_PRODUCTION
   if (ssys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   ssys->sprites[snum]->frame = frame;

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|sprite_set_xy - change a sprites position                             |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_set_xy (sprite_system * spr_sys, int snum, int newx, int newy)
{
   JLIB_ENTER("sprite_set_xy");

#ifndef JLIB_PRODUCTION
    jlib_check_sprite_system(spr_sys);
    jlib_check_sprite(spr_sys,snum);
#endif

   spr_sys->sprites[snum]->y = newy;
   spr_sys->sprites[snum]->x = newx;

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|find_first_free - find the number of the first free sprite.           |*/
/*+----------------------------------------------------------------------+*/
JINLINE int sprite_find_first_free (sprite_system * spr_sys)
{
   int count;

   JLIB_ENTER("sprite_find_first_free");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* NOTE : this routine exits if there are no free sprites left. */
   
   if(SPR_NUM_ACTIVE(spr_sys)==(SPR_MAX_SPRITES(spr_sys)-1)){
      jlib_exit("No Free Sprites.");
   }
   
   for (count = 0; spr_sys->active_sprites[count] != 0; count++);

   JLIB_LEAVE;

   return count;
}


/*+----------------------------------------------------------------------+*/
/*|find_first_frame - find the number of the first free sprite frame.    |*/
/*+----------------------------------------------------------------------+*/
JINLINE int sprite_find_first_frame(sprite_system * spr_sys)
{
   JLIB_ENTER("sprite_find_first_frame");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   if(SPR_NUM_LOADED(spr_sys)>=SPR_MAX_FRAMES(spr_sys)){
      jlib_exit("No Free Sprite Frames.");
   }
   
   JLIB_LEAVE;

   return SPR_NUM_LOADED(spr_sys);
}


/*+----------------------------------------------------------------------+*/
/*|set_mode - Set the mode of a sprite (fixed point or whole).           |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_set_mode (sprite_system * spr_sys, int snum,int mode)
{
   JLIB_ENTER("sprite_set_mode");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   spr_sys->sprites[snum]->mode = mode;

   JLIB_LEAVE;
} 


/*+----------------------------------------------------------------------+*/
/*|set_move_info - Set up the movement info for a sprite.                |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_set_move_info (sprite_system * spr_sys, int snum,
				   UBYTE speed, BYTE xinc, BYTE yinc)
{
   JLIB_ENTER("sprite_set_move_info");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   spr_sys->sprites[snum]->speed = speed;
   
   if(spr_sys->sprites[snum]->mode == SPR_MODE_WHOLE){
      spr_sys->sprites[snum]->speed_counter = speed;
   }
   else{
      spr_sys->sprites[snum]->speed_counter = 0;
   }
   
   spr_sys->sprites[snum]->xinc = xinc;
   spr_sys->sprites[snum]->yinc = yinc;

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|set_anim_info - Set up the animation info for a sprite.               |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_set_anim_info (sprite_system * spr_sys, int snum,
			     UBYTE anspeed, UBYTE noframes, int *animpat)
{
   JLIB_ENTER("sprite_set_anim_info");

#ifndef JLIB_PRODUCTION
   if(spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   spr_sys->sprites[snum]->animspeed = anspeed;
   
   if(spr_sys->sprites[snum]->mode == SPR_MODE_WHOLE){
      spr_sys->sprites[snum]->animspeed_counter = anspeed;
   }
   else{
      spr_sys->sprites[snum]->animspeed_counter = 0;
   }

   spr_sys->sprites[snum]->noframes = noframes;
   spr_sys->sprites[snum]->frame = 0;
   spr_sys->sprites[snum]->frame_count = 0;
   spr_sys->sprites[snum]->animframes = animpat;

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|sprite_turn_off - turn off a sprite                                   |*/
/*+----------------------------------------------------------------------+*/
JINLINE void sprite_turn_off (sprite_system * spr_sys, int snum)
{
   JLIB_ENTER("sprite_turn_off");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* turn off if it was on */
   if(spr_sys->active_sprites[snum] == 1){
      /* restore default values to sprite */
      sprite_set_mode(spr_sys,snum,SPR_MODE_WHOLE);
      sprite_set_move_info(spr_sys,snum,0,0,0);
      sprite_set_anim_info(spr_sys,snum,0,0,NULL);
 
      spr_sys->active_sprites[snum] = 0;
      spr_sys->no_active--;
   };

   JLIB_LEAVE;
}


/*+---------------------------------------------------------------------+*/
/*|update_anim_and_move - update a sprites posistion and animframe.     |*/
/*+---------------------------------------------------------------------+*/
JINLINE void sprite_update_anim_and_move (sprite_system * spr_sys, int snum)
{
  sprite_record *current;

  JLIB_ENTER("sprite_update_anim_and_move");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

  current = spr_sys->sprites[snum];

  if(SPR_IS_ON (spr_sys, snum)){
      if(current->mode == SPR_MODE_WHOLE){
         /* check movement */
         if (current->speed){
            if ((--current->speed_counter) == 0){
               current->speed_counter = current->speed;
               current->x += current->xinc;
               current->y += current->yinc;
            }
         }

         /* check animation */
         if (current->animspeed){
             if ((--current->animspeed_counter) == 0){
                current->animspeed_counter = current->animspeed;
             
                if ((++current->frame_count) == current->noframes){
                   current->frame_count = 0;
                }
                current->frame = current->animframes[current->frame_count];
             }
          }
      }
      else{
         /* check fixed point movement */
         if (current->speed){
             current->speed_counter+=current->speed;
             current->x += ((current->speed_counter>>4)*current->xinc);
             current->y += ((current->speed_counter>>4)*current->yinc);
             current->speed_counter=(current->speed_counter & 0xf);

         }

         /* check fixed point animation */
         if (current->animspeed){
             current->animspeed_counter+=current->animspeed;
             current->frame_count+=(current->animspeed_counter>>4);
	     
             while(current->frame_count >= current->noframes){
                current->frame_count -= current->noframes;
             }
             current->frame = current->animframes[current->frame_count];
          }
          current->animspeed_counter=(current->animspeed_counter & 0xf);
      }
  }

  JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|sprite_update_all_anim_and_move                                       |*/
/*+----------------------------------------------------------------------+*/
void sprite_update_all_anim_and_move (sprite_system * spr_sys)
{
   int count;

   JLIB_ENTER("sprite_update_all_anim_and_move");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   for (count = 0; count < spr_sys->no_sprites; count++) {
      if (SPR_IS_ON (spr_sys, count)) {
	     sprite_update_anim_and_move (spr_sys, count);
      }
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|sprite_do_all_anim_and_move_n_times                                   |*/
/*+----------------------------------------------------------------------+*/
void sprite_do_all_anim_and_move_n_times (sprite_system * spr_sys, int n)
{
   int count, i;

   JLIB_ENTER("sprite_do_all_anim_and_move_n_times");

#ifndef JLIB_PRODUCTION
   if ((spr_sys == NULL) || (n < 0)){
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   for (count = 0; count < spr_sys->no_sprites; count++){
      if (SPR_IS_ON (spr_sys, count)){
          for (i = n; i != 0; i--){
             sprite_update_anim_and_move (spr_sys, count);
          }
      }
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|JLIB_do_intersect - this routine is private!                          |*/
/*+----------------------------------------------------------------------+*/
JINLINE int JLIB_do_intersect (int x1, int y1, UBYTE *r1, int x2, int y2, UBYTE *r2)
{
 if(((x1+r1[0]) < (x2 +r2[2])) &&
    ((x1+r1[2]) > (x2 +r2[0])) &&
    ((y1+r1[1]) < (y2 +r2[3])) &&
    ((y1+r1[3]) > (y2 +r2[1]))){
     return 1;
 }
 
 return 0;    
}


/*+----------------------------------------------------------------------+*/
/*|sprite_do_intersect                                                   |*/
/*+----------------------------------------------------------------------+*/
int sprite_do_intersect (sprite_system * spr_sys, int s1, int s2)
{
   int s1x, s1y, s2x, s2y;
   UBYTE *s1rect, *s2rect;
   int s1count, s2count;

   JLIB_ENTER("sprite_do_intersect");

#ifndef JLIB_PRODUCTION
   if (spr_sys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* if either sprite is off,  return no */
   if ((!SPR_IS_ON (spr_sys, s1)) || (!SPR_IS_ON (spr_sys, s2))){
       JLIB_LEAVE;
       return 0;
   }

   /* set up preliminary information */
   s1x = spr_sys->sprites[s1]->x;
   s1y = spr_sys->sprites[s1]->y;
   s2x = spr_sys->sprites[s2]->x;
   s2y = spr_sys->sprites[s2]->y;

   s1rect = spr_sys->sprite_data[spr_sys->sprites[s1]->frame]->rect_coords;
   s2rect = spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->rect_coords;

   /* for each bounding rectangle of sprite 1,  check each rect of sprite 2 */
   for (s1count = spr_sys->sprite_data[spr_sys->sprites[s1]->frame]->no_rects; s1count != 0; s1count--){
      for (s2count = spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->no_rects; s2count != 0; s2count--){
          if (JLIB_do_intersect (s1x, s1y, s1rect, s2x, s2y, s2rect)){
                JLIB_LEAVE;
                return 1;
           }
           else{
               /* move to the 2nd sprites next rectangle coordinates */
               s2rect += 4;
           }
      }
      s2rect -= (spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->no_rects * 4);
	
      s1rect += 4; /* move to the 1st sprites next rect coords */
   }

   JLIB_LEAVE;
   return 0;
}


/*+----------------------------------------------------------------------+*/
/*|sprites_do_intersect                                                  |*/
/*+----------------------------------------------------------------------+*/
int sprites_do_intersect (sprite_system *sys1, int s1,sprite_system *sys2,int s2)
{
   int s1x, s1y, s2x, s2y;
   UBYTE *s1rect, *s2rect;
   int s1count, s2count;

   JLIB_ENTER("sprites_do_intersect");

#ifndef JLIB_PRODUCTION
   if ((sys1 == NULL) || (sys2 == NULL)) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

   /* if either sprite is off,  return no */
   if ((!SPR_IS_ON (sys1, s1)) || (!SPR_IS_ON (sys2, s2))){
        JLIB_LEAVE;
        return 0;
   }

   /* set up preliminary information */
   s1x = sys1->sprites[s1]->x;
   s1y = sys1->sprites[s1]->y;
   s2x = sys2->sprites[s2]->x;
   s2y = sys2->sprites[s2]->y;

   s1rect = sys1->sprite_data[sys1->sprites[s1]->frame]->rect_coords;
   s2rect = sys2->sprite_data[sys2->sprites[s2]->frame]->rect_coords;

   /* for each bounding rectangle of sprite 1,  check each rect of sprite 2 */
   for (s1count = sys1->sprite_data[sys1->sprites[s1]->frame]->no_rects; s1count != 0; s1count--) {
        for (s2count = sys2->sprite_data[sys2->sprites[s2]->frame]->no_rects; s2count != 0; s2count--) {
           if (JLIB_do_intersect (s1x, s1y, s1rect, s2x, s2y, s2rect)){
               JLIB_LEAVE;
               return 1;
           }
           else{
              /* move to the 2nd sprites next rectangle coordinates */
              s2rect += 4;
           }
        }
        s2rect -= (sys2->sprite_data[sys2->sprites[s2]->frame]->no_rects * 4);
        s1rect += 4; /* move to the 1st sprites next rect coords */
   }

   JLIB_LEAVE;
   return 0;
}



/*+-----------------------------------------------------------------------+*/
/*|sprite_free - free the memory used by a sprite system.                 |*/
/*+-----------------------------------------------------------------------+*/
sprite_system *sprite_free (sprite_system * ssys)
{
   int counter;

   JLIB_ENTER("sprite_free");

   if (ssys == NULL){
       JLIB_LEAVE;
       return NULL;
   }

   if (ssys->sprites != NULL) {
      for (counter = 0; counter < SPR_MAX_SPRITES (ssys); counter++) {
          free (ssys->sprites[counter]->buffer);
          free (ssys->sprites[counter]);
      }
   }

   free (ssys->sprites);

   if (ssys->sprite_data != NULL) {
      for (counter = 0; counter < SPR_MAX_FRAMES (ssys); counter++) {
          if (ssys->sprite_data[counter] != NULL) {
             free(ssys->sprite_data[counter]->data);
             free(ssys->sprite_data[counter]->pattern);
             free(ssys->sprite_data[counter]->rect_coords);
             free(ssys->sprite_data[counter]);
          }
      }
   }

   free(ssys->sprite_data);

   free(ssys->active_sprites);

   free(ssys);

   JLIB_LEAVE;
   return NULL;
}


#define _MY_HEIGHT  (ssys->sprite_data[frame]->height)
#define _MY_WIDTH   (ssys->sprite_data[frame]->width)

/*+-----------------------------------------------------------------------+*/
/*|sprite_build_from_buff - build a sprite data frame from a buffer.      |*/
/*+-----------------------------------------------------------------------+*/
void sprite_build_from_buff(sprite_system *ssys,int frame,buffer_rec *bf,
			    int x1,int y1,int x2,int y2)
{
 int width;
 UBYTE *store;

 JLIB_ENTER("sprite_build_from_buff");

#ifndef JLIB_PRODUCTION
 if((bf==NULL) || (ssys==NULL)){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }

 if((frame >= SPR_MAX_FRAMES(ssys)) || (frame < 0)){
    jlib_exit("Invalid Frame");
 }
#endif

 width = B_X_SIZE(bf);

 if(x1 > x2){
    JLIB_SWAP(x2,x1);
 }

 if(y1 > y2){
    JLIB_SWAP(y2,y1);
 }

 /* check width & height,  truncate if too big */
 if((x2-x1) > SPR_MAX_X){
    x2=x1+SPR_MAX_X;
 }

 if((y2-y1) > SPR_MAX_Y){
    y2=y1+SPR_MAX_Y;
 }

 /* check to see if the sprite frame already exists */
 if (ssys->sprite_data[frame] != NULL){
    JLIB_SPRINTF("Overwriting Frame %d.",frame);

    /* kill existing sprite frame */
    free(ssys->sprite_data[frame]->data);
    free(ssys->sprite_data[frame]->data);

    if(ssys->sprite_data[frame]->rect_coords != NULL){
       free(ssys->sprite_data[frame]->rect_coords);
    }

    free(ssys->sprite_data[frame]);
 }
 else{
    /* increase the count of loaded sprites */
    ssys->number_loaded++;
 }

 /* make a sprite_data_rec for the sprite */
 if ((ssys->sprite_data[frame] =
	     (sprite_data_rec *) malloc (sizeof (sprite_data_rec))) == NULL){
     jlib_exit(jlib_msg(JLIB_EMALLOC));
 }

 /* set width and height */
 _MY_HEIGHT=y2-y1+1;
 _MY_WIDTH=x2-x1+1;

 /* create with no bounding rectangles */
 ssys->sprite_data[frame]->no_rects=0;
 ssys->sprite_data[frame]->rect_coords=NULL;

 if ((ssys->sprite_data[frame]->data =
      (UBYTE *) malloc ((_MY_HEIGHT * _MY_WIDTH) * sizeof (UBYTE))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
 }

 store=ssys->sprite_data[frame]->data;

 /* copy in the data */
 for(;y1<=y2;y1++){
     int j;
     UBYTE *start = B_OFFSET(bf,y1);
     for(j=x1;j<=x2;j++){
       *store=start[j];
       store++;
     }
 }

 /* generate the RLE pattern */
 ssys->biggest_rle=global_biggest_rle;
 generate_RLE (ssys->sprite_data[frame]);

 global_biggest_rle=ssys->biggest_rle;
 sprite_optimise_mem(ssys);

 JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Add a bounding rectangle to a sprite frame.                            |*/
/*+-----------------------------------------------------------------------+*/
void sprite_add_rect(sprite_system *ssys,int frame,UBYTE x1,UBYTE y1,UBYTE x2,UBYTE y2)
{
  UBYTE *rc;

#ifndef JLIB_PRODUCTION
   if ((ssys == NULL) || (frame >= SPR_NUM_LOADED(ssys)) || (frame <0)) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }

   if (ssys->sprite_data[frame] == NULL){
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

  /* add the rectangle to the sprite */
  ssys->sprite_data[frame]->no_rects++; /* increment count */
  
  /* increase memory to hold new coords */
  ssys->sprite_data[frame]->rect_coords = (UBYTE *)
  realloc(ssys->sprite_data[frame]->rect_coords,(ssys->sprite_data[frame]->no_rects * 4));

  if(ssys->sprite_data[frame] == NULL) {
     jlib_exit(jlib_msg(JLIB_EMALLOC));
  }

  /* write out the rectangles */
  rc =  ssys->sprite_data[frame]->rect_coords + ((ssys->sprite_data[frame]->no_rects - 1) * 4);
  rc[0] = x1;
  rc[1] = y1;
  rc[2] = x2;
  rc[3] = y2;
}


/*+-----------------------------------------------------------------------+*/
/*|Remove the background saving buffers from a sprite system.             |*/
/*+-----------------------------------------------------------------------+*/
void sprites_kill_buffers(sprite_system * ssys)
{
 int i;

#ifndef JLIB_PRODUCTION
   if (ssys == NULL) {
       jlib_exit(jlib_msg(JLIB_ENULL));
   }
#endif

  for(i=0;i<SPR_MAX_SPRITES(ssys);i++){
    if(ssys->sprites[i]->buffer != NULL){
       free(ssys->sprites[i]->buffer);
       ssys->sprites[i]->buffer = NULL;
    }
  }        
}

