;*
;* VESA.ASM - useful SVGA video functions
;* by Piotr Ulaszewski
;* I strated to write this code when I saw the vesa.asm file released
;* by Babyloon Revelations somewhere in 1997 (I think).
;* Modified functions (many modifications, but the idea stays the same):
;* CheckVESA, InitVESAMode
;* I like the idea of the buffer that holds all available vesa modes.
;*
;* This is an extended version for C/C++ support.
;*


; globals for the VESA interface (VESA.ASM)
global _VBE_TotalMemory                          ;word - size of video RAM in KB
global _VBE_VBEVersion                           ;word - BCD version of VESA interface
global _VBE_NumberOfMode                         ;dword - total number of graphics modes
global _VBE_Gfx_Mode                             ;word - physical number of mode found
global _VBE_VRAM_Address                         ;dword - pointer to Video RAM
global _VBE_BytesPerScanLine                     ;dword - width in bytes of VESA mode
global _VBE_LFBAvailable                         ;byte - if = 1 then LFB is available
global _VBE_PMODEInterface                       ;byte - if = 1 then PMODE is available
global _VBE_PMODEBankSwitch                      ;byte - if = 1 then PM Bank switch ok
global _VBE_Msg_Table                            ;dword - pointer to VESA errors
global _VBE_NumberOfMemoryPages                  ;byte - for the current mode
global _VBE_VirtualScreenAddress                 ;dword - pointer to Virtual Screen
global _VBE_FirstByteBeyondVirtualScreen         ;dword - pointer to end of Virtual Screen
global _VBE_CurrentPageMappingSize               ;dword - size in bytes of the current page
global _VBE_TotalMappingSize                     ;dword - size of video RAM in bytes
global _VBE_XResolution                          ;dword - X resolution of the current mode
global _VBE_YResolution                          ;dword - Y resolution of the current mode
global _VBE_BitsPerPixel                         ;dword - Bits per Pixel of the current mode
global _VBE_CalcBitsPerPixel                     ;dword - Bits per Pixel (if 15 then 16)
global _VBE_CalcBitsPerPixel2                    ;dword - Bits per Pixel shr 3
global _ROM_Font                                 ;dword - pointer to system font
global _VideoMemory_StartAddress                 ;dword - pointer to start of Linear Framebuffer
global _CheckVESA                                ;proc
global _InitVESAMode                             ;proc
global _SetVBEMode                               ;proc
global _CloseVBEMode                             ;proc
global _PutVESAScreen                            ;proc
global _WaitRetrace                              ;proc
global _Flip                                     ;proc
global _IncreaseResolution                       ;proc
global _DecreaseResolution                       ;proc
global _GetROMFont                               ;proc

%include "include\segments.inc"
%include "include\message.mac"

segment _DATA
; VBEInfoBlock 
struc   VBEInfoBlock
	VBESignature      resd 1        ; dd 0 - VBE Signature
	VBEVersion        resw 1        ; dw 0 - VBE Version
	OEMStringPtr      resd 1        ; dd 0 - Pointer to OEM string
	Capabilities      resd 1        ; dd 4 - Graphics capabilities of card
	VideoModePtr      resd 1        ; dd 0 - Pointer to Video Mode List
	TotalMemory       resw 1        ; dw 0 - Number of 64kb memory blocks (add for 2.0)
	OEMSoftwareRev    resw 1        ; dw 0 - VBE implementation Software revision (2.0)
	OEMVendorNamePtr  resd 1        ; dd 0 - Pointer to Vendor Name String        (2.0)
	OEMProductNamePtr resd 1        ; dd 0 - Pointer to Product Name String       (2.0)
	OEMProductRevPtr  resd 1        ; dd 0 - Pointer to Product Revision String   (2.0)
	Reserved          resb 222      ; db 222 dup (0) - Reserved for VBE implementation..
	OemData           resb 256      ; db 256 dup (0) - Data Area for OEM Strings     (2.0)
endstruc

; ModeInfoBlock 
     ; Mandatory information for all VBE revisions
     ModeAttributes      EQU 0   ; DW mode attributes
     WinAAttributes      EQU 2   ; DB window A attributes
     WinBAttributes      EQU 3   ; DB window B attributes
     WinGranularity      EQU 4   ; DW window granularity
     WinSize             EQU 6   ; DW window size
     WinASegment         EQU 8   ; DW window A start segment
     WinBSegment         EQU 10  ; DW window B start segment
     WinFuncPtr          EQU 12  ; DD pointer to window function
     BytesPerScanLine    EQU 16  ; DW bytes per scan line
     ; Mandatory information for VBE 1.2 and above
     XResolution         EQU 18  ; DW horizontal resolution in pixels or chars
     YResolution         EQU 20  ; DW vertical resolution in pixels or chars
     XCharSize           EQU 22  ; DB character cell width in pixels
     YCharSize           EQU 23  ; DB character cell height in pixels
     NumberOfPlanes      EQU 24  ; DB number of memory planes
     BitsPerPixel        EQU 25  ; DB bits per pixel
     NumberOfBanks       EQU 26  ; DB number of banks
     MemoryModel         EQU 27  ; DB memory model type
     BankSize            EQU 28  ; DB bank size in KB
     NumberOfImagePages  EQU 29  ; DB number of images
     Reserved_           EQU 30  ; DB reserved for page function
     ; Direct Color fields (required for direct/6 and YUV/7 memory models)
     RedMaskSize         EQU 31  ; DB size of direct color red mask in bits
     RedFieldPosition    EQU 32  ; DB bit position of lsb of red mask
     GreenMaskSize       EQU 33  ; DB size of direct color green mask in bits
     GreenFieldPosition  EQU 34  ; DB bit position of lsb of green mask
     BlueMaskSize        EQU 35  ; DB size of direct color blue mask in bits
     BlueFieldPosition   EQU 36  ; DB bit position of lsb of blue mask
     RsvdMaskSize        EQU 37  ; DB size of direct color reserved mask in bits
     RsvdFieldPosition   EQU 38  ; DB bit position of lsb of reserved mask
     DirectColorModeInfo EQU 39  ; DB direct color mode attributes
     ; Mandatory information for VBE 2.0 and above
     PhysBasePtr         EQU 40  ; DD physical address for flat frame buffer
     OffScreenMemOffset  EQU 44  ; DD pointer to start of off screen memory
     OffScreenMemSize    EQU 48  ; DW amount of off screen memory in 1k units


	VBE_RealSegment         dw 0
	VBE_MemSelector         dw 0

	VBE_Signature           db 0,0  ; Signature
				db 0,0
				db 13,10,'$'
	_VBE_TotalMemory         dw 0    ; Video Ram (in kb)
	_VBE_VBEVersion          dw 0    ; BCD Version
	VBE_OEMStringPtr        dd 0    ; Pointer to OEM string
	VBE_SoftwareRev         dw 0    ; BCD Software rev.
	VBE_OEMVendorNamePtr    dd 0    ; Pointer to Vendor Name String
	VBE_OEMProductNamePtr   dd 0    ; Pointer to Product Name String
	VBE_OEMProductRevPtr    dd 0    ; Pointer to Product Revision String
	_VBE_NumberOfMode        dd 0
	VBE_ModeList            times 1024 dw 0 ; dw 1024 dup (0) Video Mode List with :
				; Width, Height, bpp & mode number
	_VBE_Gfx_Mode            dw 0    ; Hexadecimal graphic mode found
	_VBE_VRAM_Address        dd 0    ; Pointer to VIDEO RAM

	_VBE_BytesPerScanLine    dd 0    ; Bytes per scan line
	VBE_BANKSetTable        times 128 dd 0  ; dd 128 DUP (0)
	VBE_WindowSize          dd 0
	VBE_NumberOfBanks       db 0
	VBE_CurrentBank         db 0    ; Current used bank
	_VBE_LFBAvailable        db 0    ; =1 if LFB available
	_VBE_PMODEInterface      db 0    ; =1 if Protected Mode interface found
	_VBE_PMODEBankSwitch     db 0    ; =1 if use P-Mode bank switch
	VBE_PMODESetStartPtr    dd 0    ; Pointer to P-Mode set start function
	VBE_PMODESetStartFactor db 0    ; Factor need for Start Address
	VBE_SetVBEBankPtr       dd 0    ; Pointer to BankSwitch function rm&pm
	VBE_Counter             db 0

	_VBE_Msg_Table           dd VBE_Err0,VBE_Err1,VBE_Err2
				dd VBE_Err3,VBE_Err4,VBE_Err5
				dd VBE_Err6,VBE_Err7,VBE_Err8,VBE_Err9

	VBE_Err0                db '$'
	VBE_Err1                db 'Error - DOS memory allocation failed$'
	VBE_Err2                db 'Error - $'
	VBE_Err3                db 'Error - VESA not supported$'
	VBE_Err4                db 'Error - DOS memory DeAllocation failed$'
	VBE_Err5                db 'Error - Requested video mode not found$'
	VBE_Err6                db 'Error - Wrong VESA mode$'
	VBE_Err7                db 'Error - Cannot map memory$'
	VBE_Err8                db 'Error - Bank Switching not available$'
	VBE_Err9                db 'Error - Bad Window A attributes$'
	VBE_ErrA                db 'Error - Could not allocate Virtual Screen$'

	es_segment dw 0

	VBE_PMODESetPagePtr dd 0                ; pointer to paging function
	VBE_Page db 0                           ; current VBE page
	_VBE_NumberOfMemoryPages db 0            ; total number of pages for a given mode
	VBE_NumberOfScanLines dd 0
	_VBE_VirtualScreenAddress dd 0           ; current virtual screen
	_VBE_FirstByteBeyondVirtualScreen dd 0   ; current end of virt scr + 1
	_VBE_CurrentPageMappingSize dd 0         ; memory size of one page
	_VBE_TotalMappingSize dd 0               ; video memory size in bytes
	_VBE_XResolution dd 0                    ; current X resolution
	_VBE_YResolution dd 0                    ; current Y resolution
	_VBE_BitsPerPixel dd 0                   ; current bits per pixel
	_VBE_CalcBitsPerPixel dd 0               ; calculated bpp (if 15 then 16)
	_VBE_CalcBitsPerPixel2 dd 0              ; bpp shr 3

	_ROM_Font dd 0				; Address of system font
	_VideoMemory_StartAddress dd 0
	VBE_VirtualScreenHandle dd 0		; only for system memory vbuf

;
;
segment _TEXT
;
;
Simulate_RM:                            ; this is an internal procedure
	push es
	clc
	push dword 0                    ; SS,SP
	lea esp,[esp-10]                ; CS,IP,GS,FS,DS - ignored
	push word [es_segment]
	pushfw
	pushad
	push ss
	pop es
	mov edi,esp                     ; ES:EDI - real mode block
	mov ax,300h
	xor cx,cx
	xor bh,bh
	mov bl,[esp+58]                 ; interrupt number
	int 31h
	popad
	popfw
	pop word [es_segment]
	lea esp,[esp+14]                ; CS,IP,SS,SP - ignored
	pop es
	ret 4                           ; discard int number



;******************************************************************************
;*
;* CheckVESA: Check if VBE is present and all available VESA modes.
;* In:
;*     Nothing
;* Out:
;*     EAX - Error code
;*              =0 VESA Found.
;*              =1 DOS Memory allocation failed (should never occur).
;*              =2 (not used any more)
;*              =3 VESA Not supported.
;*              =4 DOS Memory DeAllocation failed (should never occur).
;*
;******************************************************************************
_CheckVESA:
	push ebx
	push ecx
	push edx
	push esi
	push edi
	push ebp
	push fs

; *** Allocate DOS memory block for GET VESA INFO(4F00), GET MODE INFO(4F01)..
	mov ax,100h                 ; Allocate DOS memory block
	mov bx,32                   ; 32*16=512 bytes
	int 31h                     ;
	jnc .@@DOSAllocOk
	mov eax,1                   ; DOS memory allocation error
	jmp .@@CheckDone
.@@DOSAllocOk:
	mov [VBE_RealSegment],ax    ; Save segment and selector registers.
	mov [VBE_MemSelector],dx    ;

; preset VbeSignature field to 'VBE2' to have VBE 2.0 fields filled if possible
	mov fs,[VBE_MemSelector]
	xor edi,edi
	mov eax,'VBE2'
	mov dword [fs:edi+VBESignature],eax

	mov ax,[VBE_RealSegment]
	mov [es_segment],ax
	xor edi,edi
	mov eax,04F00h
	push dword 10h
	call Simulate_RM
	cmp ax,004Fh
	je .@@VBE_VESADriver_found
	mov eax,3                                ; VESA not supported
	jmp .@@EndCheck
.@@VBE_VESADriver_found:
	mov fs,[VBE_MemSelector]
	xor edi,edi

	mov eax,dword [fs:edi+VBESignature]
	mov dword [VBE_Signature],eax

	mov ax,word [fs:edi+VBEVersion]
	mov [_VBE_VBEVersion],ax

	mov ax,Word [fs:edi+TotalMemory]
	shl ax,6                                ; 64Ko blocks
	mov [_VBE_TotalMemory],ax
	movzx eax,ax
	shl eax,10                              ; convert to bytes
	mov [_VBE_TotalMappingSize],eax  

	mov ax,word [fs:edi+OEMSoftwareRev]
	mov [VBE_SoftwareRev],ax

	mov eax,dword [fs:edi+OEMStringPtr]     ; eax=Segment*65536+Offset
	mov edx,eax
	movzx ebx,ax
	xor dx,dx
	shr edx,12
	add edx,ebx
	mov [VBE_OEMStringPtr],edx              ; LinearAddress=Segment*4+Offset

	mov eax,dword [fs:edi+OEMVendorNamePtr]
	mov edx,eax
	movzx ebx,ax
	xor dx,dx
	shr edx,12
	add edx,ebx
	mov [VBE_OEMVendorNamePtr],edx

	mov eax,dword [fs:edi+OEMProductNamePtr]
	mov edx,eax
	movzx ebx,ax
	xor dx,dx
	shr edx,12
	add edx,ebx
	mov [VBE_OEMProductNamePtr],edx

	mov eax,dword [fs:edi+OEMProductRevPtr]
	mov edx,eax
	movzx ebx,ax
	xor dx,dx
	shr edx,12
	add edx,ebx
	mov [VBE_OEMProductRevPtr],edx

; *** VBE Mode List - This is the buffer that holds all available modes
	mov dword [_VBE_NumberOfMode],0
	mov esi,VBE_ModeList
	mov ebp,100h                        ; first VESA mode
.@@SetModeListLoop:
	mov ax,[VBE_RealSegment]
	mov [es_segment],ax
	xor edi,edi
	mov eax,04f01h
	mov ecx,ebp
	push ebp
	push esi
	push dword 10h
	call Simulate_RM
	pop esi
	pop ebp
	cmp ax,004Fh
	jne .@@BadVideoMode
	inc dword [_VBE_NumberOfMode]
	mov fs,[VBE_MemSelector]
	mov ax,[fs:XResolution]
	mov [esi],ax
	mov ax,[fs:YResolution]
	mov [esi+2],ax
	xor eax,eax
	mov al,[fs:BitsPerPixel]
	cmp al,8
	jbe .@@DontAddColorComponents
	mov al,[fs:RedMaskSize]              ; recalculate bpp field
	add al,[fs:GreenMaskSize]
	add al,[fs:BlueMaskSize]
.@@DontAddColorComponents:
	cmp al,byte 24                       ; check if we have a 32bit mode
	jne .@@DontAddColorComponents2
	add al,[fs:RsvdMaskSize]             ; 8:8:8:8 = 32bit organisation
.@@DontAddColorComponents2:
	mov [esi+4],ax
	mov [esi+6],bp
	add esi,byte 8                       ; adjust to next mode entry
.@@BadVideoMode:
	inc ebp
	cmp ebp,0201h
	jne near .@@SetModeListLoop

; *** Free allocated DOS memory block ***
	mov ax,0101h
	mov dx,[VBE_MemSelector]
	int 31h
	jnc .@@VBE_FreeMem_Ok
	mov eax,4
	jmp .@@CheckDone
.@@VBE_FreeMem_Ok:
	xor eax,eax
	jmp .@@CheckDone

.@@EndCheck:
	push eax
	mov ax,0101h
	mov dx,[VBE_MemSelector]
	int 31h
	pop eax
.@@CheckDone:
	pop fs
	pop ebp
	pop edi
	pop esi        
	pop edx
	pop ecx
	pop ebx
	ret



;******************************************************************************
;*
;* InitVESAMode: Verify if requested mode is available and initialize it..
;* In:
;*     push dword - Bits per pixel
;*     push dword - video mode height
;*     push dword - video mode width
;*
;* Out:
;*     EAX  - Error code
;*              =0 Initialisation ok.
;*              =1 DOS Memory allocation failed (should never occur).
;*              =2 (not used any more)
;*              =4 DOS Memory DeAllocation failed (should never occur).
;*              =5 Requested video mode not found.
;*              =6 Wrong VESA mode.
;*              =7 Can't map memory (DPMI error).
;*              =8 Bank Switching not really available !!!.
;*              =9 Bad Window A attributes (must be read/write).
;*              =A Virtual Screen allocation in system memory failed.
;*
;******************************************************************************
_InitVESAMode:
	push ebx
	push ecx
	push edx
	push esi
	push edi
	push ebp
	push fs
	mov ax,[esp+4+28]                            ; video mode width
	mov bx,[esp+8+28]                            ; video mode height
	mov cx,[esp+12+28]                           ; bits per pixel
	mov esi,VBE_ModeList
	mov ebp,[_VBE_NumberOfMode]
	or ebp,ebp
	jnz .@@FindModeLoop
	mov eax,5
	jmp .@@InitDone                              ; discard 3 dwords
.@@FindModeLoop:
	cmp ax,[esi]
	jne .@@SkipThisMode
	cmp bx,[esi+2]
	jne .@@SkipThisMode
	cmp cx,[esi+4]
	jne .@@SkipThisMode
	mov dx,[esi+6]
	mov word [_VBE_Gfx_Mode],dx                  ; Set Graphic Mode
	jmp .@@VideoModeFound
.@@SkipThisMode:
	add esi,byte 8
	dec ebp
	jnz .@@FindModeLoop
	mov eax,5
	jmp .@@InitDone
.@@VideoModeFound:
	movzx eax,ax
	movzx ebx,bx
	movzx ecx,cx
	mov [_VBE_XResolution],eax
	mov [VBE_NumberOfScanLines],ebx
	mov [_VBE_YResolution],ebx
	mov [_VBE_BitsPerPixel],ecx
	cmp ecx,byte 15
	jne .@@dont_add_bit
	inc ecx
.@@dont_add_bit:
	mov [_VBE_CalcBitsPerPixel],ecx
	shr ecx,3
	mov [_VBE_CalcBitsPerPixel2],ecx
	shl ecx,3
	imul eax,ebx
	imul eax,ecx
	shr eax,3
	mov [_VBE_CurrentPageMappingSize],eax
; *** Alloc DOS memory block for Mode Info Buffer ***
	mov ax,100h                         ; Allocate DOS memory block
	mov bx,32                           ; 32*16=512 bytes
	int 31h
	jnc .@@VBE_DOSAlloc_Ok
	mov eax,1                           ; can't alloc DOS memory
	jmp .@@InitDone
.@@VBE_DOSAlloc_Ok:
	mov [VBE_RealSegment],ax    ; Save segment and selector registers.
	mov [VBE_MemSelector],dx

; *** INITIALIZE VIDEO MODE !!!!
	mov ax,[VBE_RealSegment]
	mov [es_segment],ax
	xor edi,edi
	mov eax,04F01h
	movzx ecx,word [_VBE_Gfx_Mode]
	push dword 10h
	call Simulate_RM
	cmp ax,004Fh
	je .@@VBE_VideoMode_Ok
	Mov eax,6                          ; Wrong VESA mode
	jmp .@@EndInit
.@@VBE_VideoMode_Ok:

	mov fs,[VBE_MemSelector]

	movzx eax,word [fs:BytesPerScanLine]
	mov [_VBE_BytesPerScanLine],eax

	mov byte [VBE_PMODESetStartFactor],0
	cmp byte [fs:BitsPerPixel],4
	je .@@NoFactorNeeded
	mov byte [VBE_PMODESetStartFactor],2
.@@NoFactorNeeded:

	mov ax,[fs:ModeAttributes]         ; Test if LFB available
	test ax,010000000b
	jz near .@@VBE_BankSwitchMethod

; *************************************************************
; ***           LINEAR FRAME BUFFERING AVAILABLE            ***
; *************************************************************
	mov byte [_VBE_LFBAvailable],1
	mov eax,[fs:PhysBasePtr]
	mov [_VideoMemory_StartAddress],eax
; LFB Physical address converted into linear address..
	mov ecx,eax
	mov ebx,eax
	shr ebx,16                        ; BX:CX = Physical address of LFB

	mov eax,[_VBE_TotalMappingSize]
	mov edi,eax
	mov esi,eax
	shr esi,16                        ; SI:DI=size of zone to map in bytes

	mov ax,0800h                      ; DPMI Physical address mapping..
	int 31h
	jnc .@@VBE_MemMapping_ok
	Mov eax,7                         ; Can't map memory..
	jmp .@@EndInit
.@@VBE_MemMapping_ok:
	shl ebx,16
	mov bx,cx
	mov [_VBE_VRAM_Address],ebx
	
	add ebx,[_VBE_CurrentPageMappingSize]
	mov [_VBE_VirtualScreenAddress],ebx
	add ebx,[_VBE_CurrentPageMappingSize]
	mov [_VBE_FirstByteBeyondVirtualScreen],ebx
	mov eax,[_VBE_TotalMappingSize]
	cdq
	idiv dword [_VBE_CurrentPageMappingSize]
	mov [_VBE_NumberOfMemoryPages],al

; code added for system virtual screen support (release 4)
	dec al
	jnz .@@hardware_flip_ok
	mov eax,0501h
	mov ebx,[_VBE_CurrentPageMappingSize]
	add ebx,byte 16                     ; ensure that EMB is not overwr
	mov cx,bx
	shr ebx,16                          ; BX:CX - size of virtual scr
	int 31h
	jnc .@@vesa_system_flip_ok
	mov eax,0ah
	jmp .@@EndInit
.@@vesa_system_flip_ok:
	shl ebx,16
	mov bx,cx                           ; EBX - address of virtual scr
	mov [_VBE_VirtualScreenAddress],ebx
	add ebx,[_VBE_CurrentPageMappingSize]
	mov [_VBE_FirstByteBeyondVirtualScreen],ebx
	shl esi,16
	mov si,di                           ; ESI - handle of virtual scr
	mov [VBE_VirtualScreenHandle],esi

.@@hardware_flip_ok:
	Add word [_VBE_Gfx_Mode],4000h      ; Prepare for Linear Addressing.
	jmp .@@Check32BitInterface          ; Now set all Accelerated Functions.

; *************************************************************
; ***                BANK SWITCHING METHOD                  ***
; *************************************************************
.@@VBE_BankSwitchMethod:
	mov byte [_VBE_LFBAvailable],0
	mov fs,[VBE_MemSelector]
	mov ax,[fs:ModeAttributes]              ; Test if bank switching
	test ax,01000000b                       ; is really available.
	jz .@@VBE_windowing_supported
	mov eax,8                               ; BankSwitch not really available..
	jmp .@@EndInit
.@@VBE_windowing_supported:

	mov al,[fs:WinAAttributes]              ; Test if Window Attributes
	and al,0101b                            ; are OK.
	cmp al,0101b			          ; must be readable/writeable
	je .@@VBE_WinAttributes_ok
	mov eax,9                               ; Bad Window A attributes
	jmp .@@EndInit
.@@VBE_WinAttributes_ok:

	movzx eax,word [fs:WinSize]             ;\
	shl eax,10                              ; | Get window size.
	mov [VBE_WindowSize],eax                ;/

	mov eax,[_VBE_CurrentPageMappingSize]
	xor edx,edx
	mov ebx,[VBE_WindowSize]
	idiv ebx
	or edx,edx
	je .@@VBE_EndCalcBanks
	inc eax
.@@VBE_EndCalcBanks:
	mov [VBE_NumberOfBanks],AL

; Setup bank numering table..
	Movzx eax,word [fs:WinSize]
	Movzx ebx,word [fs:WinGranularity]
	xor edx,edx
	div ebx
	xor edx,edx
	xor ecx,ecx
.@@LoopSetUpBank:
	mov [VBE_BANKSetTable+ECX*4],edx
	add edx,eax
	inc ecx
	cmp ecx,128                            ; length VBE_BANKSetTable
	jb .@@LoopSetUpBank

; Window A physical Address converted into 32bits pointer. 0A000h -> 0A0000h
	xor eax,eax
	mov ax,[fs:WinASegment]
	shl eax,4
	mov [_VBE_VRAM_Address],eax

	mov eax,RealModeSetBank                ; Pointer to the standard
	mov [VBE_SetVBEBankPtr],eax            ; bankswitch function. (real mode).

	mov byte [VBE_CurrentBank],1           ;\
	xor dl,dl                              ; |-- Initialize to first bank..->0
	call SetVBEBank                        ;/

; added VESA 1.2 or below support for simulated page flipping (release 4)
	mov eax,0501h
	mov ebx,[_VBE_CurrentPageMappingSize]
	add ebx,byte 16                        ; ensure that EMB is not overwr
	mov cx,bx
	shr ebx,16                             ; BX:CX - size of virtual scr
	int 31h
	jnc .@@vesa_1_flip_ok
	mov eax,0ah
	jmp .@@EndInit
.@@vesa_1_flip_ok:
	shl ebx,16
	mov bx,cx                              ; EBX - address of virtual scr
	mov [_VBE_VirtualScreenAddress],ebx
	add ebx,[_VBE_CurrentPageMappingSize]
	mov [_VBE_FirstByteBeyondVirtualScreen],ebx
	shl esi,16
	mov si,di                              ; ESI - handle of virtual scr
	mov [VBE_VirtualScreenHandle],esi

	mov eax,[_VBE_TotalMappingSize]
	cdq
	idiv dword [_VBE_CurrentPageMappingSize]
	mov [_VBE_NumberOfMemoryPages],al

;******************************************************
;*** Now Check for a 32bit bank switch interface !! ***
;******************************************************
.@@Check32BitInterface:
	mov byte [_VBE_PMODEInterface],0
	mov byte [_VBE_PMODEBankSwitch],0

; *** Return VBE Protected Mode Interface - DOS interrupt !!!
	mov eax,04F0Ah
	xor ebx,ebx
	push dword 10h
	call Simulate_RM
	cmp ax,004Fh
	jne .@@No32BitInterface

	movzx esi,word [es_segment]
	shl esi,4
	and edi,0FFFFh
	add esi,edi                            ; ESI=Segment*16+Offset

; Check if there's a zero length memory list, it should be that for most video
; cards.
	movzx edi,word [ESI+06]                ; get port/memory table list
	or edi,edi
	jz .@@UsePmodeInterface                ; No port list..
	mov ax,Word [esi+edi]
	cmp ax,0FFFFh
	je .@@CheckMemoryList
.@@CheckPortList:
	add edi,byte 2
	mov ax,word [Esi+Edi]                  ; CMP WORD PTR [ESI+EDI],0FFFFh
	cmp ax,0FFFFh                          ; search port list
	jne .@@CheckPortList
.@@CheckMemoryList:
	add edi,byte 2
	mov ax,Word [Esi+Edi]
	cmp ax,0FFFFh                          ; see if mem list is zero
	jne .@@NoPMODEBankSwitch
.@@UsePmodeInterface:
	movzx eax,Word [ESI+00]
	add eax,esi
	mov [VBE_SetVBEBankPtr],eax            ; save SetBank code address
	mov byte [_VBE_PMODEBankSwitch],1
.@@NoPMODEBankSwitch:
	movzx eax,Word [ESI+02]
	add eax,esi
	mov [VBE_PMODESetStartPtr],eax         ; save SetStart code address
	movzx eax,word [esi+4]
	add eax,esi
	mov [VBE_PMODESetPagePtr],eax
	mov byte [_VBE_PMODEInterface],1
.@@No32BitInterface:
	mov ax,0101h
	mov dx,[VBE_MemSelector]
	int 31h
	jnc .@@VBE_FreeMem_Ok
	mov eax,4
	jmp .@@InitDone
.@@VBE_FreeMem_Ok:
	xor eax,eax                             ; All is ok !!
	jmp .@@InitDone
.@@EndInit:
	push eax
	mov ax,0101h
	mov dx,[VBE_MemSelector]
	int 31h
	pop eax
.@@InitDone:
	pop fs
	pop ebp
	pop edi
	pop esi
	pop edx
	pop ecx
	pop ebx
	ret



;******************************************************************************
;*
;* SetVBEMode: Set the previously initialised SVGA graphics mode.
;*
;*        In:  Nothing
;*        Out: Nothing
;*
;******************************************************************************
_SetVBEMode:
	pushad
	mov ax,4F02h
	mov bx,[_VBE_Gfx_Mode]
	int 10h
	popad
	Ret



;******************************************************************************
;*
;* CloseVBEMode: Close the active SVGA graphics mode - unmap LFB.
;*
;*        In:  Nothing
;*        Out: Nothing
;*
;*
;******************************************************************************
_CloseVBEMode:
	pushad
	cmp byte [_VBE_LFBAvailable],1
	jne .@@vesa1_or_below
	cmp byte [_VBE_NumberOfMemoryPages],1
	je .@@vesa1_or_below
	mov ax,0801h
	mov ebx,[_VideoMemory_StartAddress]
	mov cx,bx
	shr ebx,16
	int 31h
.@@exit:
	mov byte [VBE_Page],0
	popad
	ret

; code added for simulated page flipping (VESA 1.2 or below) in release 4

.@@vesa1_or_below:				; also when VBE Pages = 1
	mov eax,0502h
	mov esi,[VBE_VirtualScreenHandle]
	mov di,si
	shr esi,16				; SI:DI - virtual scr handle
	int 31h
	jmp short .@@exit	



;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;******************************************************************************
;*
;* SetVBEBank: Sets the actually visible bank at 0A0000h for VESA 1.2
;*             (internal to vesa.asm)
;*
;*        In:  DL - Bank number
;*        Out: Nothing
;*
;*
;******************************************************************************
SetVBEBank:
	cmp [VBE_CurrentBank],dl
	je .@@BankAlreadySet
	and edx,01fh
	mov [VBE_CurrentBank],dl
	mov edx,[VBE_BANKSetTable+edx*4]
	push ebx
	call [VBE_SetVBEBankPtr]
	pop ebx
.@@BankAlreadySet:
	ret



;******************************************************************************
;*
;* RealModeSetBank: Sets the position of Window A in the frame buffer memory
;*                  (internal to vesa.asm)
;*
;*        In:  DX - Window number in video memory in window granularity units
;*        Out: Nothing
;*
;*
;******************************************************************************
RealModeSetBank:
	xor ebx,ebx
	mov ax,4F05h        ; Use Real Mode interrupt
	int 10h
	ret



;******************************************************************************
;*
;* SetVBEStart: Setup start address (for Real Mode and Protected Mode interface)
;*              (internal to vesa.asm)
;*
;*        In:  ECX - Pixel in scanline
;*             EDX - scanline number
;*        Out: Nothing
;*
;*
;******************************************************************************
SetVBEStart:
	cmp byte [_VBE_PMODEInterface],1
	je .@@PMODESetStart
	mov eax,4F07h
	xor ebx,ebx
	int 10h
	ret
.@@PMODESetStart:
	imul edx,[_VBE_BytesPerScanLine]
	add edx,ecx
	mov cl,[VBE_PMODESetStartFactor]
	shr edx,cl
	mov cx,dx
	shr edx,16
	xor ebx,ebx
	call [VBE_PMODESetStartPtr]
	ret



;******************************************************************************
;*
;* SetVBEPage: Set the actually visible page (0=1st pge, 1=2nd page, etc.)
;*             Only PMODE interface for flipping pages is supported.
;*             Currently not used.
;*             (internal to vesa.asm)
;*
;*        In:  Set the page in [VBE_page] location.
;*        Out: Nothing
;*
;******************************************************************************
SetVBEPage:                                     ; 0=1st page, 1=2nd page, etc.
	cmp byte [_VBE_PMODEInterface],1
	jne .do_not_switch
	push eax                                ; only with PMODE interface!
	push ebx
	push ecx
	push edx
	movzx eax,byte [VBE_Page]
	mov edx,[_VBE_BytesPerScanLine]
	imul edx,[VBE_NumberOfScanLines]
	imul edx,eax
	mov cl,[VBE_PMODESetStartFactor]
	shr edx,cl
	mov cx,dx
	shr edx,16                              ; DX:CX - offset from start
	xor ebx,ebx
	call [VBE_PMODESetStartPtr]
	pop edx
	pop ecx
	pop ebx
	pop eax
.do_not_switch:
	ret



;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;******************************************************************************
;*
;* PutVESAScreen: Transfer picture from virtual buffer in system memory to the
;*                VESA screen in video memory (for VESA 1.2 or below) but
;*                it may also be used with VESA 2.0 or above,
;*                simply to show an image in system memory on the screen.
;*
;*        In:  push dword - Address of virtual screen in system memory
;*        Out: Nothing
;*
;******************************************************************************
_PutVESAScreen:
	pushad
	mov esi,[esp+4+32]                      ; address of virtual screen
	push es
	push ds
	pop es
	cmp byte [_VBE_LFBAvailable],1
	je .@@Use_LFB
	mov ebp,[_VBE_CurrentPageMappingSize]
	mov al,[VBE_NumberOfBanks]
	mov [VBE_Counter],al
.@@LoopBankSwitch:
	mov edi,[_VBE_VRAM_Address]
	mov dl,[VBE_NumberOfBanks]
	sub dl,[VBE_Counter]
	Call SetVBEBank

	sub ebp,[ds:VBE_WindowSize]
	js .@@EndLoopBankSwitch
	mov ecx,[VBE_WindowSize]
	shr ecx,2
	rep movsd
	dec byte [VBE_Counter]
	jnz .@@LoopBankSwitch
.@@EndLoopBankSwitch:
	or ebp,ebp
	jz .@@End_VBE
	add ebp,[ds:VBE_WindowSize]
	mov ecx,ebp
	shr ecx,2
	rep movsd
	jmp .@@End_VBE
.@@Use_LFB:
	mov edi,[_VBE_VRAM_Address]
	mov ecx,[_VBE_CurrentPageMappingSize]
	shr ecx,2
	rep movsd
.@@End_VBE:
	pop es
	popad
	ret



;******************************************************************************
;*
;* WaitRetrace: Additional wait for vertical retrace function.
;*              Used for VESA 1.2 flip simulation.
;*
;*        In:  Nothing
;*        Out: Nothing
;*
;******************************************************************************
_WaitRetrace:
	push eax
	push edx
	mov dx,3DAh
.@@WaitRetrace1:
	in al,dx
	test al,8
	jne .@@WaitRetrace1
.@@WaitRetrace2:
	in al,dx
	test al,8
	je .@@WaitRetrace2
	pop edx
	pop eax
	ret



;******************************************************************************
;*
;* Flip: Perform double buffering in video memory.
;*       Supports real and protected mode VESA interface.
;*       Please note that protected mode interface is much faster.
;*       See README.TXT for detailed explanation of multi buffering.
;*
;*        In:  Nothing
;*        Out: Nothing
;*
;******************************************************************************
_Flip:                               ; perform double buffering (protected mode)
	cmp byte [_VBE_LFBAvailable],1
	jne near .simulated_flip
	cmp byte [_VBE_PMODEInterface],1
	jne .flip2
	cmp byte [_VBE_NumberOfMemoryPages],1        
	je near .simulated_flip
	pushad
	xor byte [VBE_Page],1       
	jz .page_zero

	mov edx,[_VBE_CurrentPageMappingSize]
	jmp short .page_done
.page_zero:
	xor edx,edx
.page_done:      
	mov cl,[VBE_PMODESetStartFactor]
	shr edx,cl
	mov cx,dx
	shr edx,16
	xor ebx,ebx

	push eax
	push edx
	mov edx,3DAh
.@@Retrace1:
	in al,dx
	test al,8
	jne .@@Retrace1
	pop edx
	pop eax
	call [VBE_PMODESetStartPtr]
	mov edx,3DAh
.@@Retrace2:
	in al,dx
	test al,8
	je .@@Retrace2

	mov ecx,[_VBE_CurrentPageMappingSize]
	mov eax,[_VBE_VRAM_Address]
	mov edi,eax
	xchg eax,[_VBE_VirtualScreenAddress]
	mov [_VBE_VRAM_Address],eax
	mov eax,edi                             ; new VirtualScreenAddress
	add eax,ecx
	mov [_VBE_FirstByteBeyondVirtualScreen],eax      
	xor eax,eax
	shr ecx,2
	rep stosd                               ; clear virtual screen
	popad
	ret

.flip2:                                  ; perform double buffering (real mode)
	cmp byte [_VBE_NumberOfMemoryPages],1        
	je .simulated_flip
	pushad
	xor byte [VBE_Page],1       
	jz .page_zero2
	mov edx,[_VBE_YResolution]
	jmp short .page_done2
.page_zero2:
	xor edx,edx
.page_done2:     
	xor ecx,ecx
	mov eax,4f07h
	xor ebx,ebx

	push eax
	push edx
	mov dx,3DAh
.@@Retrace1_2:
	in al,dx
	test al,8
	jne .@@Retrace1_2
	pop edx
	pop eax
	int 10h
	mov dx,3DAh
.@@Retrace2_2:
	in al,dx
	test al,8
	je .@@Retrace2_2

	mov ecx,[_VBE_CurrentPageMappingSize]
	mov eax,[_VBE_VRAM_Address]
	mov edi,eax
	xchg eax,[_VBE_VirtualScreenAddress]
	mov [_VBE_VRAM_Address],eax
	mov eax,edi                             ; new VirtualScreenAddress
	add eax,ecx
	mov [_VBE_FirstByteBeyondVirtualScreen],eax      
	xor eax,eax
	shr ecx,2
	rep stosd                               ; clear virtual screen
	popad
	ret

.simulated_flip:                              ; only VESA 1.2 or below
	pushad
	mov esi,[_VBE_VirtualScreenAddress]
	call _WaitRetrace
	push esi
	call _PutVESAScreen
	add esp,byte 4
	mov edi,[_VBE_VirtualScreenAddress]
	xor eax,eax
	mov ecx,[_VBE_CurrentPageMappingSize]
	shr ecx,2
	rep stosd
	popad
	ret



;******************************************************************************
;*
;* IncreaseResolution: Increase the SVGA graphics resolution on the fly.
;*                     Only modes defined after color_modes are supported.
;*                     You may define more modes that will be supported.
;*
;*        In:  Nothing
;*        Out: If error carry flag set and resolution stays the same.
;*             If success carry flag cleared and resolution has been increased.
;*             EAX = FALSE on error
;*             EAX = TRUE on success
;*
;******************************************************************************
segment _DATA

; resolution values for mode switching on the fly
color_modes dw 320,200,320,240,400,300,320,400,320,480,512,384,640,400
dw 640,480,800,600

segment _TEXT

_IncreaseResolution:
	pushad
	mov edi,color_modes
	mov ecx,9
.@@find_actual_mode:
	movzx eax,word [edi]            ; XRes
	movzx edx,word [edi+2]          ; YRes
	cmp eax,[_VBE_XResolution]
	jne .@@prepare_for_next_search
	cmp edx,[_VBE_YResolution]
	jne .@@prepare_for_next_search
	jmp short .@@actual_mode
.@@prepare_for_next_search:
	add edi,byte 4
	dec ecx
	jnz .@@find_actual_mode
	stc
	jmp short .@@exit
.@@actual_mode:                         ; EAX = XRes   EBX = YRes
	cmp edx,600                     ; Maximum YResolution in (800 * 600)
	je .@@do_not_increase
	call _CloseVBEMode
	movzx eax,word [edi+4]          ; XRes
	movzx ebx,word [edi+6]          ; YRes
	mov ecx,[_VBE_BitsPerPixel]
	push ecx
	push ebx
	push eax
	call _InitVESAMode
	add esp,byte 12
	or eax,eax
	jnz .@@do_not_increase
	call _SetVBEMode
	clc
.@@exit:
	popad
	xor eax,eax
	adc al,0
	ret
.@@do_not_increase:
	stc
	jmp short .@@exit



;******************************************************************************
;*
;* DecreaseResolution: Decrease the SVGA graphics resolution on the fly.
;*                     Only modes defined after color_modes are supported.
;*                     You may define more modes that will be supported.
;*
;*        In:  Nothing
;*        Out: If error carry flag set and resolution stays the same.
;*             If success carry flag cleared and resolution has been decreased.
;*             EAX = FALSE on error
;*             EAX = TRUE on success
;*
;******************************************************************************
_DecreaseResolution:
	pushad
	mov edi,color_modes
	mov ecx,9
.@@find_actual_mode:
	movzx eax,word [edi]            ; XRes
	movzx edx,word [edi+2]          ; YRes
	cmp eax,[_VBE_XResolution]
	jne .@@prepare_for_next_search
	cmp edx,[_VBE_YResolution]
	jne .@@prepare_for_next_search
	jmp short .@@actual_mode
.@@prepare_for_next_search:
	add edi,byte 4
	dec ecx
	jnz .@@find_actual_mode
	stc
	jmp short .@@exit
.@@actual_mode:                         ; EAX = XRes   EBX = YRes
	cmp edx,200                     ; Minimum YResolution in (320 * 200)
	je .@@do_not_decrease
	call _CloseVBEMode
	movzx eax,word [edi-4]          ; XRes
	movzx ebx,word [edi-2]          ; YRes
	mov ecx,[_VBE_BitsPerPixel]
	push ecx
	push ebx
	push eax
	call _InitVESAMode
	add esp,byte 12
	or eax,eax
	jnz .@@do_not_decrease
	call _SetVBEMode
	clc
.@@exit:
	popad
	xor eax,eax
	adc al,0
	ret
.@@do_not_decrease:
	stc
	jmp short .@@exit



;******************************************************************************
;*
;* GetROMFont: Get the system font location using VGA BIOS call
;*
;*        In:  Nothing
;*        Out: Nothing
;*
;******************************************************************************
_GetROMFont:
	pushad
	mov eax,1130h
	mov bh,6
	push dword 10h
	call Simulate_RM
	movzx eax,word [es_segment]
	shl eax,4
	movzx ebx,bp
	add eax,ebx
	mov [_ROM_Font],eax
	popad
	ret



