{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}


unit pFIBCacheQueries;

interface

{$I FIBPlus.inc}
 uses
 {$IFDEF MSWINDOWS}
  Sysutils,Classes,FIBQuery,pFIBQuery,FIBDatabase;
 {$ENDIF}
 {$IFDEF LINUX}
  SysUtils,Classes,FIBQuery,pFIBQuery,FIBDatabase;
 {$ENDIF}

function  GetQueryForUse(aTransaction:TFIBTransaction; const SQLText:string):TpFIBQuery;
procedure FreeQueryForUse(aFIBQuery:TpFIBQuery);

implementation

 uses SQlTxtRtns,pFIBLists;

 type
{$WARNINGS OFF}
      TCacheQueries=class(TComponent)
      private
       FFIBDataBase:TFIBDataBase;
       FListUnused :TObjStringList;
       FListUsed   :TList;
       vInClear    :boolean;
       procedure   Clear;
       function    UseQuery(aTransaction:TFIBTransaction;
                    const SQLText:string
                   ):TpFIBQuery;
       procedure   UnUseQuery(aFIBQuery:TpFIBQuery);
      protected
       procedure   Notification(AComponent: TComponent; Operation: TOperation);override;
      public
       constructor Create(aFIBDataBase:TFIBDataBase);
       destructor  Destroy; override;
      end;
{$WARNINGS ON}

      TCacheList =class(TComponent)
      private
       FList:TList;
       function    GetCacheForDB(aDataBase:TFIBDatabase):TCacheQueries;
       procedure   RemoveDataBase(aDataBase:TFIBDatabase);
      protected
       procedure   Notification(AComponent: TComponent; Operation: TOperation);override;
      public
       constructor Create(AOwner:TComponent);override;
       destructor  Destroy; override;

       function  UseQuery(aTransaction:TFIBTransaction;
         const SQLText:string
       ):TpFIBQuery;
       procedure UnUseQuery(aFIBQuery:TpFIBQuery);
      end;

      THackQuery = class(TpFIBQuery);      

var
 CacheList:TCacheList;

{ TCacheQueries }

procedure TCacheQueries.Clear;
var i:integer;
begin
  try
   vInClear := true;
   with FListUsed do
   for i := 0 to Pred(Count) do
     TObject(FListUsed[i]).Free;
  finally
   vInClear := false;  
  end;
end;

constructor TCacheQueries.Create(aFIBDataBase:TFIBDataBase);
begin
  inherited Create(nil);
  FFIBDataBase:=aFIBDataBase;
  FListUnused :=  TObjStringList.Create(nil,true);
  FListUsed  :=TList.Create;
end;

destructor TCacheQueries.Destroy;
begin
  Clear;
  try
   vInClear := true;
   FListUnused.Free;
  finally
   vInClear := false;
  end;
  FListUsed.Free;
  inherited;
end;


procedure TCacheQueries.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (Operation=opRemove) and not vInClear then
  if Acomponent is TpFIBQuery then
  begin
    FListUsed.Remove(Acomponent);
    FListUnused.Remove(Trim(THackQuery(Acomponent).vNormalizedText));
  end;
  inherited;
end;

procedure TCacheQueries.UnUseQuery(aFIBQuery:TpFIBQuery);
var i:integer;
begin
 i:=FListUsed.IndexOf(aFIBQuery);
 if i>=0 then begin
  FListUsed.Delete(i);
  FListUnused.AddObject(
   Trim(THackQuery(aFIBQuery).vNormalizedText),aFIBQuery
  );
 end;
end;

function TCacheQueries.UseQuery(
  aTransaction: TFIBTransaction; const SQLText: string
): TpFIBQuery;
var i:integer;
begin
 if FListUnused.Find(Trim(SQLText),i) then
 begin
  Result:=TpFIBQuery(FListUnused.Objects[i]);
  if Result.Transaction<>aTransaction then
   Result.Transaction:=aTransaction;
  FListUsed.Add(Result);
  FListUnused.Delete(i)
 end
 else
 begin
   Result:=TpFIBQuery.Create(nil);
   Result.FreeNotification(Self);
   with Result do
   begin
     DataBase:=FFIBDataBase;
     Transaction:=aTransaction;
     SQl.Text :=SQlText;
     FListUsed.Add(Result);
   end;    
 end;
end;

{ TCacheList }

constructor TCacheList.Create(AOwner: TComponent);
begin
  inherited;
  FList:=TList.Create;
end;

destructor TCacheList.Destroy;
var i:integer;
begin
  with FList do
  for i := 0 to Pred(Count) do   
   TObject(FList[i]).Free;
  FList.Free;
  inherited;
end;

function TCacheList.GetCacheForDB(aDataBase: TFIBDatabase): TCacheQueries;
var j:integer;
begin
 with FList do
 for j := 0 to Pred(Count) do    // Iterate
 if TCacheQueries(FList[j]).FFIBDataBase=aDataBase then
 begin
  Result:=TCacheQueries(FList[j]);
  exit;
 end;
 Result:=TCacheQueries.Create(aDataBase);
 FList.Add(Result)
end;

procedure TCacheList.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
 if (Operation=opRemove)and (AComponent is TFIBDatabase) then
  RemoveDataBase(TFIBDatabase(AComponent));
 inherited;
end;

procedure TCacheList.RemoveDataBase(aDataBase: TFIBDatabase);
var i:integer;
begin
 with FList do
 for i := Pred(Count) downto 0 do
 if TCacheQueries(FList[i]).FFIBDataBase=aDataBase then
 begin
  TCacheQueries(FList[i]).Free;
  Delete(i)
 end;
end;

procedure TCacheList.UnUseQuery(aFIBQuery: TpFIBQuery);
begin
 if(aFIBQuery=nil) or(aFIBQuery.Database=nil) then exit;
 GetCacheForDB(aFIBQuery.Database).UnUseQuery(aFIBQuery);
end;

function TCacheList.UseQuery(aTransaction: TFIBTransaction;
  const SQLText: string): TpFIBQuery;
begin
 Result:=nil;
 if(aTransaction=nil) or(aTransaction.DefaultDatabase=nil) then exit;
 aTransaction.DefaultDatabase.FreeNotification(Self); 
 Result:=
  GetCacheForDB(aTransaction.DefaultDatabase).UseQuery(aTransaction,SQLText);
end;

// interface
function
 GetQueryForUse(aTransaction:TFIBTransaction; const SQLText:string):TpFIBQuery;
var NSQL:string;
begin
 NormalizeSQLText(SQLText,'@',  NSQL );
 Result:=CacheList.UseQuery(aTransaction,NSQL);
 if (Result<>nil)  and Result.Open then Result.Close; 
end;

procedure FreeQueryForUse(aFIBQuery:TpFIBQuery);
begin
  CacheList.UnUseQuery(aFIBQuery)
end;

initialization
 CacheList:=TCacheList.Create(nil);
finalization
 CacheList.Free;
end.


