//***************************************************************************
//
// Copyright (c) 1991-93 Sierra Semiconductor Corp.
//
// FILE:    aria.c
//
// LANGUAGES:
//          Microsoft C Versions 5.0, 6.0, and 7.0
//          Borland Turbo C Versions 1.5 and 2.0
//
// DESCRIPTION:
//
//          Aria Function Library
//
//          For Microsoft C 5.0, 6.0, and 7.0
//          =================================
//          compile:  cl /Zlp /Od /c aria.c    (for small model)
//
//                    for medium  model include /AM option
//                    for compact model include /AC option
//                    for large   model include /AL option
//
//          For Borland Turbo C
//          ===================
//          compile:  bcc -DTURBOC -a- -O- -c aria.c    (for small model)
//
//                    for medium  model include -mm option
//                    for compact model include -mc option
//                    for large   model include -ml option
//
// AdLib is a trademark of AdLib Inc.
// Aria, Aria Synthesizer and Aria Listener are trademarks of Sierra
//  Semiconductor Corp.
// QSound is a trademark of Archer Communications
// Sound Blaster is a trademark of Creative Labs Inc.
//
//***************************************************************************

// $Header:   F:\projects\ariai\dos\archives\aria.c_v   2.4   08 Nov 1993 15:49:28   golds  $
// $Log:   F:\projects\ariai\dos\archives\aria.c_v  $
// 
//    Rev 2.4   08 Nov 1993 15:49:28   golds
// Allowed for more options in the SetMixerConfig function, you can now
// specify mono or stereo mode even when the monitor is disabled.
// 
//    Rev 2.3   28 Sep 1993 14:49:44   golds
// Changed DSP commands for setting up recording mode.
// 
//    Rev 2.2   03 Sep 1993 09:51:36   golds
// Fixed recording from external input source
// 
//    Rev 2.1   13 Aug 1993 09:10:16   golds
// New reverb commands, 16-channel MIDI, other optimizations
// 
//    Rev 2.0   24 Jun 1993 12:15:34   golds
// Initial revision.
//

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>

#include "aria.h"

#define ON  1
#define OFF 2


//*** Global variables ***

WORD dspbase  = DEF_DSPBASE;    // Aria DSP base address
WORD midibase = DEF_MIDIBASE;   // MIDI port base address
WORD dspint   = DEF_ARIAINT;    // Aria DSP interrupt number
WORD midiint  = DEF_MIDIINT;    // MIDI interrupt number
WORD dmachan  = DEF_DMACHAN;    // Aria DMA channel
WORD sb_base  = DEF_SB_BASE;    // Sound Blaster base address
WORD sb_int   = DEF_SB_INT;     // Sound Blaster interrupt number
WORD sysver   = DEF_SYSVER;     // Aria system version
WORD version  = DEF_VERSION;    // Aria system ROM version
WORD DSPversion = DEF_DSPVER;   // Aria DSP version number - app should
                                //  call GetDSPVersion ()
WORD  wDSPpart  = 0;            // Aria DSP part number code
WORD  wROMpart  = 0;            // Aria ROM part number code
short DSPstatus = 1;            // DSP status: 0 = error, 1 = ok

// Audio buffer pointers and buffer sizes

short chCount   = 0;                // Active audio channel count
WORD packetSize = DEF_PACKETSIZE;   // Digital audio packet size
WORD audioRate  = SAMP_22K;         // DSP hardware audio rate
WORD recSource  = 0;                // Recording control port bits
WORD playMode   = MAXOPERATORS;     // Default playback mode
WORD speechMode = 0;                // Aria Listener mode
WORD reverbMode = 0;                // Reverb effect mode
WORD qsoundMode = 0;                // QSound(TM) effect mode

WORD wMidiSynthAllocated = NOSYNTH;
WORD wMaxSynthOp = DSPMAXOPS;       // Max number of allowable synth ops
WORD wMaxPercOp  = MAXPERCUSSIONON; // Max number of allowable percussion ops
WORD rt_ax, rt_bx, rt_cx, rt_dx, rt_di, rt_si;

                                    // Aria capabilities structure
ARIACAPS AriaCap = {0,0,0,0,0,0,FALSE,0};

//*** External variable declarations ***

extern ARIAWAVE waveCh[MODE0_CHANNELS];   // PCM wave channel structures
extern LPBYTE ISRvector[16];    // Interrupt vector table for
                                //   Aria Interrupt Manager
extern BOOL maxLoaded;          // Maximum DSP applications loaded flag

//*** Local variables ***
                                // Default audio format
static WORD audioFormat = SR22KHZ | MONO | EIGHTBIT;
static WORD gwIntCount;         // Interrupt counter

// Old interrupt handler pointer
static VOID (interrupt FAR *oldInt)() = 0L;


//*** Local function declarations ***

static VOID Delay (UINT);
static VOID interrupt FAR IntTest1 (VOID);
static VOID interrupt FAR IntTest2 (VOID);
static VOID interrupt FAR IntTest3 (VOID);

#ifdef TURBOC
VOID disableintr (VOID);
VOID enableintr (VOID);
#endif

//*** External function declarations ***

extern VOID  interrupt FAR AriaInt (VOID);
extern VOID  ChangeOpCount (UINT, UINT);
extern short SetMode (UINT, UINT);
extern short SetADC (UINT);


//***************************************************************************
//
// FUNCTION:      short GetAriaEnv (LPINT port, LPINT irq, LPINT dma, 
//                                  LPINT ver)
//
// DESCRIPTION:   Function to retrieve base port address, IRQ, DMA channel,
//                and system version number from DOS enviroment variable.
//
// PARAMETERS:    port = pointer to Aria base port address
//                irq  = pointer to Aria IRQ number
//                dma  = pointer to Aria DMA channel
//                ver  = pointer to Aria system version number
//
// RETURNS:        0 = environment parameters set
//                -1 = could not find environment variables,
//                     parameters will be set to current default values
//
//***************************************************************************

short GetAriaEnv (LPINT port, LPINT irq, LPINT dma, LPINT ver)
   {
   char *value;
   UINT temp;

   // Set parameters to current defaults
   *port = (UINT) dspbase;
   *irq  = (UINT) dspint;
   *dma  = (UINT) dmachan;
   *ver  = (UINT) sysver;

   // Environment variable defined?

   if ((value = getenv ("ARIA")) == NULL)
      return -1;

   // Parse the environment string

   while (*value != '\0')
      {
      // Skip blanks
      while (*value == ' ')
         ++value;

      switch (*value)
         {
         // Get base address port (hex string)
         case 'A':
         case 'a':
            value++;
            temp = (*value - '0') << 8;
            value++;
            temp += ((*value - '0') << 4);
            value++;
            temp += (*value - '0');
            value++;
            *port = temp;
            break;

         // Get IRQ number
         case 'I':
         case 'i':
            value++;
            temp = *value - '0';
            value++;
            if (*value != ' ' && *value != '\0')
               {
               temp *= 10;
               temp += (*value - '0');
               value++;
               }
            *irq = temp;
            break;

         // Get DMA channel
         case 'D':
         case 'd':
            value++;
            temp = *value - '0';
            value++;
            *dma = temp;
            break;

         // Get system version number
         case 'T':
         case 't':
            value++;
            temp = *value - '0';
            value++;
            if (*value != ' ' && *value != '\0')
               {
               temp *= 10;
               temp += (*value - '0');
               value++;
               }
            *ver = temp;
            break;

         default:
            while (*value != ' ' && *value != '\0')
               value++;
            break;
         }
      }
   return 0;
   }


//***************************************************************************
//
// FUNCTION:      UINT GetDSPVersion (VOID)
//
// DESCRIPTION:   Function to set and return Aria DSP version number.
//
// PARAMETERS:    None
//
// RETURNS:       Major version number in high byte
//                Minor version number in low byte
//                0xFFFF = error
//
//***************************************************************************

UINT GetDSPVersion (VOID)
   {
   DSPversion = getMem16 (DSP_VERSION);
   return DSPversion;
   }


//***************************************************************************
//
// FUNCTION:      VOID GetAriaCaps (LPBYTE caps)
//
// DESCRIPTION:   Function to get Aria system capabilities
//
// PARAMETERS:    caps = far pointer to Aria capabilities structure
//
// RETURNS:       None
//
//***************************************************************************

VOID GetAriaCaps (LPBYTE caps)
   {
   WORD hw_flag1, sw_flag1;
   ARIACAPS FAR *ac;

   hw_flag1 = getMem16 (HW_OFFSET1);
   sw_flag1 = getMem16 (SW_OFFSET1);
   wDSPpart = getMem16 (DSP_PARTCODE);
   wROMpart = getMem16 (ROM_PARTCODE);

   ac = (ARIACAPS FAR *) caps;

   ac->version = GetVersion ();
   ac->ROMsize = 1 << ((hw_flag1 & ROM_SIZEMASK) + 6);
   if (hw_flag1 & ROM_RESMASK)
      ac->ROMwidth = 16;
   else
      ac->ROMwidth = 8;
   ac->mixerType = (hw_flag1 & MIXER_MASK) >> 5;
   switch ((hw_flag1 & RAM_SIZEMASK) >> 8)
      {
      case RAM_8K:
         ac->RAMsize = 8;
         break;
      case RAM_16K:
         ac->RAMsize = 16;
         break;
      case RAM_32K:
         ac->RAMsize = 32;
         break;
      case RAM_40K:
         ac->RAMsize = 40;
         break;
      case RAM_64K:
         ac->RAMsize = 64;
         break;
      }
   switch ((hw_flag1 & ADC_RESMASK) >> 11)
      {
      case ADC_8BIT:
         ac->ADCbits = 8;
         break;
      case ADC_10BIT:
         ac->ADCbits = 10;
         break;
      case ADC_12BIT:
         ac->ADCbits = 12;
         break;
      case ADC_16BIT:
         ac->ADCbits = 16;
         break;
      }
   ac->phoneInterface = (hw_flag1 & PHONE_INTFC)? TRUE: FALSE;
   ac->DSPapps = sw_flag1;
   }


//***************************************************************************
//
// FUNCTION:      UINT GetVersion (VOID)
//
// DESCRIPTION:   Function to get Aria system ROM version
//
// PARAMETERS:    None
//
// RETURNS:       Major version number in high-order byte
//                Minor version number in low-order byte
//                0x0000 = cannot determine version
//                0xFFFF = DSP not responding
//
//***************************************************************************

UINT GetVersion (VOID)
   {
   UINT ver;
   UINT m, n;
   UINT ROMwavebuf;

   n = 1;                              // Get wave buffer address
   ROMwavebuf = getMem16 (WAVEBUFADDR);
   putMem16 (ROMWAVEFLAG, n);          // Set ROM wave flag to 1

   DISABLE
   sendDSP (DSPREADROM);               // Retrieve ROM block 0x8000
   sendDSP (WAVEBUFFER);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return (0xFFFF);

   m = 20;
   while (n && m)                      // Wait for block to copy to buffer
      {
      Delay (100);
      n = getMem16 (ROMWAVEFLAG);
      --m;
      }

   if (m)                              // If copied ok,
      {                                //   Read ROM version from buffer
      ver = getMem16 (ROMwavebuf+252);
      ver |= (getMem16 (ROMwavebuf+253) >> 8);
      version = ver;
      return (ver);
      }

   return (0);
   }


//***************************************************************************
//
// FUNCTION:      short GetInterrupt (VOID)
//
// DESCRIPTION:   Function to set and return Aria system IRQ.
//
// PARAMETERS:    None
//
// RETURNS:       Interrupt request number (10, 11 or 12)
//                -1 = not in Aria Wave Synthesis mode
//                -2 = unable to detect interrupt
//
//***************************************************************************

short GetInterrupt (VOID)
   {
   WORD i;
   BYTE oldctrl;
   UINT port, irq, dma, ver;
   VOID (interrupt FAR *oldInt1)();   // Old interrupt 10 vector pointer
   VOID (interrupt FAR *oldInt2)();   // Old interrupt 11 vector pointer
   VOID (interrupt FAR *oldInt3)();   // Old interrupt 12 vector pointer

   // First, check environment variable

   if (GetAriaEnv ((LPINT) &port, (LPINT) &irq, (LPINT) &dma, 
                   (LPINT) &ver) == 0)
      {
      dspint = irq;
      return (short) dspint;
      }

   if (INPW (DSPSTATUS) & C2MODE)
      {
      // Install dummy IRQ handlers to detect interrupt

      oldInt1 = GETVECT (ARIAINT1 + 0x68);
      oldInt2 = GETVECT (ARIAINT2 + 0x68);
      oldInt3 = GETVECT (ARIAINT3 + 0x68);
      DISABLE
      SETVECT (ARIAINT1 + 0x68, IntTest1);
      SETVECT (ARIAINT2 + 0x68, IntTest2);
      SETVECT (ARIAINT3 + 0x68, IntTest3);
      ENABLE

      // Enable all possible interrupts

      oldctrl = INP (INTCTRLR1);
      OUTP (INTCTRLR1, oldctrl & ~( (1 << (ARIAINT1 - 8)) |
                                    (1 << (ARIAINT2 - 8)) |
                                    (1 << (ARIAINT3 - 8)) ));

      // Force board to generate an interrupt

      dspint = 0;
      gwIntCount = 0;
      GenerateInt ();

      for (i = 0; i < SYSTEM_WAIT; ++i)   // Time delay to allow interrupt
         {
         if (dspint)
            break;
         INPW (DSPSTATUS);
         INPW (DSPSTATUS);
         }

      // Restore original IRQ mask

      OUTP (INTCTRLR1, oldctrl);

      // Restore original interrupts

      DISABLE
      SETVECT (ARIAINT1 + 0x68, oldInt1);
      SETVECT (ARIAINT2 + 0x68, oldInt2);
      SETVECT (ARIAINT3 + 0x68, oldInt3);
      ENABLE

      if (dspint && gwIntCount == 1)
         OUTP (INTCTRLR1, oldctrl | (1 << (dspint - 8)));
      else
         {
         dspint = DEF_ARIAINT;
         return -2;
         }

      return (short) dspint;
      }

   return -1;
   }


//***************************************************************************
//
// FUNCTION:      short GetDMAChannel (VOID)
//
// DESCRIPTION:   Function to set and return Aria DMA channel
//
// PARAMETERS:    None
//
// RETURNS:       DMA channel number (5 or 6)
//                -1 = unable to detect channel
//
//***************************************************************************

short GetDMAChannel (VOID)
   {
   UINT   port, irq, dma, ver;
   WORD   page, baseLSB, baseMSB;
   WORD   oldctrl;
   DWORD  absaddr;
   LPWORD valptr;
   BYTE   bDmaStatus;

   // First, check environment variable

   if (GetAriaEnv ((LPINT) &port, (LPINT) &irq, (LPINT) &dma, 
                   (LPINT) &ver) == 0)
      {
      dmachan = dma;
      return (short) dmachan;
      }

   // Set up PC's 8237A-5 DMA Controller

   ver = 0x5432;  // choose an unlikely number

   valptr  = (LPWORD) &ver;
   absaddr = ((DWORD) FP_SEG (valptr) << 4) + (DWORD) FP_OFF (valptr);
   page    = (WORD) (absaddr >> 16);
   baseLSB = (WORD) ((absaddr >> 1) & 0x00FF);
   baseMSB = (WORD) (((absaddr >> 1) & 0xFF00) >> 8);

   DISABLE
   bDmaStatus = INP (0xD0);   // Clear any TCs
   OUTP (0xD4, 5);            // Mask off DMA channel
   ASM nop
   OUTP (0xD4, 6);            // Mask off DMA channel
   ASM nop
   OUTP (0xD8, 0);            // Clear byte pointer flip-flop
   ASM nop
   OUTP (0xD6, 0x44 | 1);     // Set transfer mode (receive)
   ASM nop
   OUTP (0xD6, 0x44 | 2);     // Set transfer mode (receive)
   ASM nop

   OUTP (0xC4, baseLSB);      // Set LSB of base address
   ASM nop
   OUTP (0xC4, baseMSB);      // Set MSB of base address
   ASM nop
   OUTP (0x8B, page);         // Set page number
   ASM nop
   OUTP (0xC6, 0);            // Set LSB of data counter
   ASM nop
   OUTP (0xC6, 0);            // Set MSB of data counter
   ASM nop

   OUTP (0xC8, baseLSB);      // Set LSB of base address
   ASM nop
   OUTP (0xC8, baseMSB);      // Set MSB of base address
   ASM nop
   OUTP (0x89, page);         // Set page number
   ASM nop
   OUTP (0xCA, 0);            // Set LSB of data counter
   ASM nop
   OUTP (0xCA, 0);            // Set MSB of data counter
   ASM nop

   OUTP (0xD4, 1);            // Enable DMA channel
   ASM nop
   OUTP (0xD4, 2);            // Enable DMA channel
   ASM nop
   ENABLE

   // Wait for a value to show up

   oldctrl = INPW (DSPSTATUS);

   // Set up DSP's source address

   OUTPW (DMAADDRESS, DSP_VERSION);

   // Find the DMA channel by checking the status register of the PC's
   // 8237A-5 DMA controller, after making a DMA transfer request

   OUTPW (DSPCONTROL, oldctrl | C2MODE | DMA_DSPTOPC | DMA_XFR);

   for (page = 0; page < SYSTEM_WAIT; ++page)   // Time delay to allow DMA
      {
      if (ver != 0x5432)
         break;
      INPW (DSPSTATUS);
      }

   OUTPW (DSPCONTROL, oldctrl);

   OUTP (0xD4, 5);      // Mask off DMA channel
   ASM nop
   OUTP (0xD4, 6);      // Mask off DMA channel

   if (ver == getMem16 (DSP_VERSION))  // Got correct version number?
      {
      bDmaStatus = INP (0xD0);

      //** NOTE ************************************************************
      //  Different AT chipsets have different behavior after DMA is done.
      //  On one type of chip's status register, only the TC will be set and
      //  the rest are cleared.  On another type, only DMA request bit will
      //  be cleared.
      //********************************************************************

      if ((bDmaStatus & 0x06) == 0x06) // DMA 5 & 6 both TC are on
         {
         // Check the request bit, see which bit has been off
         if (bDmaStatus & 0x40)        // DMA 6 requested, DMA 5 served
            return 5;
         else if (bDmaStatus & 0x20)   // DMA 5 requested, DMA 6 served
            return 6;
         }
      else
         {
         if (bDmaStatus & 0x02)
            return 5;
         else if (bDmaStatus & 0x04)
            return 6;
         }
      }
   return -1;
   }


//***************************************************************************
//
// FUNCTION:      VOID InstallAriaInt (VOID)
//
// DESCRIPTION:   Function to install Aria Interrupt Manager (AIM)
//
// PARAMETERS:    None
//
// RETURNS:       None
//
//***************************************************************************

VOID InstallAriaInt (VOID)
   {
   // Install interrupt handler

   if (oldInt == 0L)
      {
      oldInt = GETVECT (dspint + 0x68);
      DISABLE
      SETVECT (dspint + 0x68, AriaInt);
      ENABLE
      }

   // Enable interrupts

   OUTP (INTCTRLR1, INP (INTCTRLR1) & ~(1 << (dspint - 8)));
   }


//***************************************************************************
//
// FUNCTION:      VOID RemoveAriaInt (VOID)
//
// DESCRIPTION:   Function to remove Aria Interrupt Manager (AIM)
//
// PARAMETERS:    None
//
// RETURNS:       None
//
//***************************************************************************

VOID RemoveAriaInt (VOID)
   {
   // Install old interrupt handler

   if (oldInt != 0L)
      {
      DISABLE
      SETVECT (dspint + 0x68, oldInt);
      ENABLE
      oldInt = 0L;
      }

   // Disable interrupts

   OUTP (INTCTRLR1, INP (INTCTRLR1) | (1 << (dspint - 8)));
   }


//***************************************************************************
//
// FUNCTION:      short RegisterISR (UINT appnum, LPBYTE addr)
//
// DESCRIPTION:   Function to register an interrupt service routine with
//                the Aria Interrupt Manager (AIM).
//
// PARAMETERS:    appnum = the DSP applicaton number (0 to 15)
//                addr   = the entry point address of the ISR
//
// RETURNS:       0 = ISR registered
//                1 = appnum out of range
//
//***************************************************************************

short RegisterISR (UINT appnum, LPBYTE addr)
   {
   // Validate parameter

   if (appnum > 15)
      return 1;

   // Set the new interrupt address in the vector table

   ISRvector[appnum] = addr;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      VOID RemoveISR (UINT appnum)
//
// DESCRIPTION:   Function to remove a previously registered ISR from the
//                Aria Interrupt Manager (AIM).
//
// PARAMETERS:    appnum = the application number (0 to 15)
//
// RETURNS:       0 = ISR removed
//                1 = appnum out of range
//
//***************************************************************************

short RemoveISR (UINT appnum)
   {
   // Validate parameter

   if (appnum > 15)
      return 1;

   // Clear the interrupt address from the vector table

   ISRvector[appnum] = 0L;
   return 0;
   }


//***************************************************************************
//
// FUNCTION:      LPBYTE GetISRAddress (UINT appnum)
//
// DESCRIPTION:   Function to return the address of a previously registered
//                ISR.
//
// PARAMETERS:    appnum = the application number (0 to 15)
//
// RETURNS:       0L = appnum out of range or
//                     no application registered for specified appnum
//                Otherwise, the ISR's entry point address
//
//***************************************************************************

LPBYTE GetISRAddress (UINT appnum)
   {
   // Validate parameter

   if (appnum > 15)
      return 0L;

   return ISRvector[appnum];
   }


//***************************************************************************
//
// FUNCTION:      short SystemInit (UINT parm1)
//
// DESCRIPTION:   Function to initialize DSP system application.
//
// PARAMETERS:    parm1 = application (DEFAULT_SYNTH or ARIA_SYNTH)
//
// RETURNS:       0 = system initialized properly
//                1 = parm1 out of range
//               -1 = DSP not responding
//               -2 = system could not initialize
//               -3 = Aria board not found
//
// NOTES:
//
//   This routine will also set the Aria base port location and version
//   number.
//
//***************************************************************************

short SystemInit (UINT parm1)
   {
   register WORD j, ready;
   WORD oldctrl;
   UINT port, irq, dma, ver;
   short ret=0;

   // Validate parameter

   if (parm1 != ARIA_SYNTH && parm1 != DEFAULT_SYNTH)
      return 1;

   audioRate = SAMP_22K;
   ready = INTDSP_PCWR | SAMP_22K | C2MODE;

   // Check environment variable

   if (GetAriaEnv ((LPINT) &port, (LPINT) &irq, (LPINT) &dma, 
                   (LPINT) &ver) == 0)
      {
      dspbase = (WORD) port;
      dspint  = (WORD) irq;
      dmachan = (WORD) dma;
      sysver  = (WORD) ver;
      OUTPW (DSPCONTROL, ready);
      }
   else
      {
      // Verify board port location

      j = ready;
      for (dspbase = DSPBASE1; dspbase <= DSPBASE4; dspbase+=0x10)
         {
         if ((oldctrl = (WORD) INPW (DSPSTATUS)) != 0xFFFF)
            {
            OUTPW (DSPCONTROL, j);
            if ((INPW (DSPSTATUS) & 0x7FFF) == ready)
               {
               j = 0;
               break;
               }
            }
         }
      if (j)
         {
         dspbase = DEF_DSPBASE;
         return -3;
         }
      }

   // Load DSP system application

   ret = SelectDSPApp (parm1, TRUE, NULL);
   if (ret == -1)
      return -1;
   else if (ret > 0)
      return -2;

   GetDSPVersion ();    // Initialize DSP version number

   if (parm1 == DEFAULT_SYNTH)
      // Set control port for AdLib/Sound Blaster emulation
      OUTPW (DSPCONTROL, audioRate);
   else
      {
      // Set control port for Aria mode
      OUTPW (DSPCONTROL, INTPC_DSPWR | INTDSP_PCWR | audioRate | C2MODE);

      // Initialize global variables for Aria PCM and Synthesizer functions
      chCount = 0;
      for (j = 0; j < MODE0_CHANNELS; ++j)
         {
         waveCh[j].status       = INACTIVE;
         waveCh[j].audioPending = INACTIVE;
         waveCh[j].audioSize    = 0L;
         waveCh[j].bufferSize   = 0L;
         waveCh[j].bufferOffset = 0L;
         waveCh[j].audioPtr     = 0L;
         waveCh[j].audioStart   = 0L;
         }
      packetSize  = DEF_PACKETSIZE;
      playMode    = MAXOPERATORS;
      wMaxSynthOp = DSPMAXOPS;
      wMaxPercOp  = MAXPERCUSSIONON;
      audioFormat = SR22KHZ | MONO | EIGHTBIT;

      GetAriaCaps ((LPBYTE) &AriaCap);
      }
   return ret;
   }


//***************************************************************************
//
// FUNCTION:      short SetWaveParameter (UINT wpt, ENVDEF FAR *env,
//                                        WPTLFO FAR *lfo, UINT pan)
//
// DESCRIPTION:   Set up wave parameter table
//
// PARAMETERS:    wpt = wave parameter table number (0 to 79)
//                env = pointer to array of 8 envelope stage structures
//                lfo = pointer to array of 3 lfo structures
//                pan = initial pan setting (0 to 127)
//
// RETURNS:       0 = wave parameter table set
//                1 = wpt out of range
//                2 = envelope stage one out of range
//                3 = envelope stage two out of range
//                "       "      "        "  "    "
//                9 = envelope stage eight out of range
//               10 = vibrato LFO number out of range
//               11 = vibrato LFO depth out of range
//               12 = tremolo LFO number out of range
//               13 = tremolo LFO depth out of range
//               14 = pan mod LFO number out of range
//               15 = pan mod LFO depth out of range
//               16 = pan out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetWaveParameter (UINT wpt, ENVDEF FAR *env, WPTLFO FAR *lfo, UINT pan)
   {
   register WORD i;
   ENVDEF FAR *envptr;
   WPTLFO FAR *lfoptr;

   // Validate parameters

   if (wpt >= MAXWPTS)
      return 1;
   envptr = env;
   for (i = 0; i < MAXSTAGES; ++i, ++envptr)
      if (envptr->index == 0x3F && envptr->length == 0x3FF)
         return (i + 2);
   lfoptr = lfo;
   for (i = 0; i < 3; ++i, ++lfoptr)
      {
      if (lfoptr->LFOnum >= MAXLFOS)
         return ((i << 1) + MAXSTAGES + 2);
      if (lfoptr->LFOdepth > MAXLFODEPTH)
         return ((i << 1) + MAXSTAGES + 3);
      }
   if (pan > MAXPAN)
      return ((i << 1) + MAXSTAGES + 2);

   // Send command sequence

   DISABLE
   sendDSP (DSPWAVEPARM);
   sendDSP (wpt);
   envptr = env;
   for (i = 0; i < MAXSTAGES; ++i, ++envptr)
      sendDSP (envptr->index | (envptr->length << 6));
   lfoptr = lfo;
   for (i = 0; i < 3; ++i, ++lfoptr)
      {
      sendDSP (lfoptr->LFOnum);
      sendDSP (lfoptr->LFOdepth);
      }
   sendDSP (pan);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetAudioFormat (UINT format)
//
// DESCRIPTION:   Set up digital audio format for recording and playback
//
// PARAMETERS:    format = sample rate | data format
//                         -----------   -----------
//                          00 (11KHz)    00 8-bit mono
//                          10 (22KHz)    01 8-bit stereo
//                          20 (44KHz)    02 16-bit mono
//                          30 ( 8KHz)    03 16-bit stereo
//                          40 (16KHz)    04 ADPCM mono
//                          50 (32KHz)    05 ADPCM stereo
//                                        06 Mu-Law mono
//                                        07 Mu-Law stereo
//                                        08 A-Law mono
//                                        09 A-Law stereo
//
// RETURNS:       0 = format set
//                1 = invalid format parameter
//               -1 = DSP not responding
//               -2 = audio channel currently active
//
//***************************************************************************

short SetAudioFormat (UINT format)
   {
   UINT fmt;            	

   // Validate parameter

   fmt = format & 0x0F;
   if ((fmt == 0x04 || fmt == 0x05) && ((format & SRMASK) != SR22KHZ))
      return 1;
   if (chCount > 0)
      return -2;

   fmt = format;
   switch (format & SRMASK)
      {
      case SR44KHZ:
         audioRate = SAMP_44K;
         break;
      case SR22KHZ:
         audioRate = SAMP_22K;
         break;
      case SR11KHZ:
         audioRate = SAMP_22K;
         break;
      case SR32KHZ:
         fmt = (format & 0x0F) | SR11KHZ;
         audioRate = SAMP_16K;
         break;
      case SR16KHZ:
         fmt = (format & 0x0F) | SR22KHZ;
         audioRate = SAMP_16K;
         break;
      case SR8KHZ:
         fmt = (format & 0x0F) | SR22KHZ;
         audioRate = SAMP_32K;
         break;
      default:
         return 1;
      }

   // Send command sequence

   DISABLE
   sendDSP (DSPAUDIOFORMAT);
   sendDSP (fmt);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   if (audioRate != SAMP_22K)
      {
      if (playMode != NOSYNTH)
         if (SetPlaybackMode (NOSYNTH) < 0)
            return -1;
      }
   OUTPW (DSPCONTROL, (INPW (DSPSTATUS) & ~SAMP_RATE) | audioRate);

   audioFormat = format;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetMasterVolume (UINT left, UINT right)
//
// DESCRIPTION:   Set up master volume of final output
//
// PARAMETERS:    left  = left channel volume (0 to 7FFFh)
//                right = right channel volume (0 to 7FFFh)
//
// RETURNS:       0 = master volume set
//                1 = left volume out of range
//                2 = right volume out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetMasterVolume (UINT left, UINT right)
   {
   // Validate parameters

   if (left > MAXVOL)
      return 1;
   if (right > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPMASTERVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetPacketSize (UINT size)
//
// DESCRIPTION:   Set up transfer packet size
//
// PARAMETERS:    size = number of words in a packet (4, 8, 16, 32, 64,
//                                                    128, 256, 512)
//
// RETURNS:       0 = packet size set
//                1 = size is an invalid value
//               -1 = DSP not responding
//
//***************************************************************************

short SetPacketSize (UINT size)
   {
   UINT i;

   for (i = 4; i <= 512; i*=2)
      {
      if (size == i)
         {
         // Send command sequence

         DISABLE
         sendDSP (DSPPACKETSIZE);
         sendDSP (size);
         sendDSP (DSPTERMINATOR);
         ENABLE

         if (!DSPstatus)
            return -1;
         packetSize = size;
         chCount = 0;
         for (i = 0; i < MODE0_CHANNELS; ++i)
            {
            waveCh[i].status       = INACTIVE;
            waveCh[i].audioPending = INACTIVE;
            waveCh[i].audioSize    = 0L;
            waveCh[i].bufferSize   = 0L;
            waveCh[i].bufferOffset = 0L;
            waveCh[i].audioPtr     = 0L;
            waveCh[i].audioStart   = 0L;
            }
         return 0;
         }
      }
   return 1;
   }


//***************************************************************************
//
// FUNCTION:      short SetPlaybackMode (UINT mode)
//
// DESCRIPTION:   Set up audio channel/synth operator mode
//
// PARAMETERS:    mode = 0  for 4 audio channels, no synthesis
//                       1  for 2 audio channel,  20 operators
//                       2  for 2 audio channels, 32 operators
//
// RETURNS:       0 = playback mode set
//                1 = mode out of range
//               -1 = DSP not responding
//               -2 = mode not available
//
//***************************************************************************

short SetPlaybackMode (UINT mode)
   {
   UINT newmode, opcount, ret=0;

   // Validate parameter

   newmode = mode & 0xFF;
   if (newmode > MAXOPERATORS)
      return 1;

   opcount = mode >> 8;
   if (opcount > DSPMAXOPS)
      return 2;

   if ((newmode > MINOPERATORS || opcount > DSPMINOPS) && maxLoaded)
      return -2;

   switch (mode)
      {
      case NOSYNTH:
         wMaxSynthOp = 0;
         wMaxPercOp  = 0;
         return SetMode (NOSYNTH, 0);

      case MINOPERATORS:
         wMaxSynthOp = DSPMINOPS;
         wMaxPercOp  = MINPERCUSSIONON;
         break;

      case MAXOPERATORS:
         wMaxSynthOp = DSPMAXOPS;
         wMaxPercOp  = MAXPERCUSSIONON;
         break;

      default:
         wMaxSynthOp = opcount;
         wMaxPercOp  = opcount >> 2;
         break;
      }

   if (playMode != NOSYNTH && (audioFormat & SRMASK) != SR22KHZ)
      ret = SetAudioFormat ((audioFormat & ~SRMASK) | SR22KHZ);

   if (ret == 0)
      ChangeOpCount (wMaxSynthOp, wMaxPercOp);

   return ret;
   }


//***************************************************************************
//
// FUNCTION:      short SetOperatorCount (UINT numops)
//
// DESCRIPTION:   Set number of operators for synthesizer
//
// PARAMETERS:    numops = number of operators (1 to 32)
//
// RETURNS:       0 = operator count set
//                1 = count out of range
//               -1 = DSP not responding
//               -2 = count not available
//
//***************************************************************************

short SetOperatorCount (UINT numops)
   {
   return SetPlaybackMode (numops << 8);
   }


///**************************************************************************
//
// FUNCTION:      short SetAuxVolume (UINT left, UINT right)
//
// DESCRIPTION:   Set up auxiliary volume (CD-ROM audio)
//
// PARAMETERS:    left  = left channel auxiliary volume (0 to 7FFFh)
//                right = right channel auxiliary volume (0 to 7FFFh)
//
// RETURNS:       0 = auxiliary volume set
//                1 = left channel out of range
//                2 = fight channel out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetAuxVolume (UINT left, UINT right)
   {
   // Validate parameters

   if (left > MAXVOL)
      return 1;
   if (right > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPAUXVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetExtVolume (UINT left, UINT right)
//
// DESCRIPTION:   Set up external volume (mic or line in audio)
//
// PARAMETERS:    left  = left channel external volume (0 to 7FFFh)
//                right = right channel external volume (0 to 7FFFh)
//
// RETURNS:       0 = external volume set
//                1 = left channel out of range
//                2 = fight channel out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetExtVolume (UINT left, UINT right)
   {
   // Validate parameters

   if (left > MAXVOL)
      return 1;
   if (right > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPEXTVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetMixerConfig (UINT source, UINT ext, UINT extmon,
//                                                   UINT aux, UINT auxmon);*
//
// DESCRIPTION:   Set up recording source, mixer input levels, and monitor
//                enable/disable
//
// PARAMETERS:    source = recording source:
//                          0000h  ADC off
//                          0001h  Microphone
//                          0002h  CD-ROM audio (AUX2)
//                          0004h  Synth/wave output
//                          0008h  External
//                          0010h  Auxiliary 1
//                          0020h  PC Speaker/Telephone input
//                          0005h  Microphone  and Synth/wave output
//                          0003h  Microphone  and CD-ROM audio
//                          0021h  Microphone  and PC Speaker/Telephone input
//                          0014h  Auxiliary 1 and Synth/wave output
//                          0012h  Auxiliary 1 and CD-ROM audio
//                          0030h  Auxiliary 1 and PC Speaker/Telephone input
//                          000Ch  External    and Synth/wave output
//                          000Ah  External    and CD-ROM audio
//                          0028h  External    and PC Speaker/Telephone input
//                mic    = 1 for microphone, 0 for line level
//                micmon = 01h for monitor enable, stereo
//                         11h for monitor enable, mono
//                         00h for monitor disable, stereo
//                         10h for monitor disable, mono
//                aux    = Not currently used
//                auxmon = 01h for monitor enable, stereo
//                         11h for monitor enable, mono
//                         00h for monitor disable, stereo
//                         10h for monitor disable, mono
//
// RETURNS:       0 = mixer levels set
//                1 = illegal source value
//                2 = microphone level out of range
//                3 = microphone monitor out of range
//                5 = auxiliary monitor out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetMixerConfig (UINT source, UINT mic, UINT micmon, UINT aux,
                      UINT auxmon)
   {
   register short i, j=3;
   static WORD validADC[16] = { MICROPHONE, AUX2AUDIO, DSPOUTPUT, 
                                ADCOFF, EXTERNAL, AUX1AUDIO, PHONEAUDIO, 
                                MICROPHONE|DSPOUTPUT,
                                MICROPHONE|AUX2AUDIO, 
                                MICROPHONE|PHONEAUDIO, 
                                AUX1AUDIO|DSPOUTPUT,  
                                AUX1AUDIO|AUX2AUDIO,
                                AUX1AUDIO|PHONEAUDIO, 
                                EXTERNAL|DSPOUTPUT, 
                                EXTERNAL|AUX2AUDIO, 
                                EXTERNAL|PHONEAUDIO };

   // Validate parameters

   if (MIXERCHIP)
      j = 16;
   for (i = 0; i < j; ++i)
      if ((WORD) source == validADC[i])
         break;
   if (i >= j)
      return 1;
   if (mic > AMPLEVEL)
      return 2;
   if (micmon > 0x11)
      return 3;
   if (auxmon > 0x11)
      return 5;

   // Send command sequence

   DISABLE
   sendDSP (DSPMIXERLEVEL);
   sendDSP (source);
   sendDSP (mic | (micmon << 4));
   sendDSP (auxmon << 4);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (j == 16)
      {
      DISABLE
      sendDSP (DSPADCSOURCE);
      sendDSP (source);
      sendDSP (DSPTERMINATOR);
      ENABLE
      }

   if (!DSPstatus)
      return -1;

   recSource = source;
   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short GenerateInt (VOID)
//
// DESCRIPTION:   Function to force DSP to generate a PC interrupt
//
// PARAMETERS:    none
//
// RETURNS:       0 = interrupt generated
//               -1 = DSP not responding
//
//***************************************************************************

short GenerateInt (VOID)
   {
   // Send command sequence

   DISABLE
   sendDSP (DSPGENPCINT);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetAudioTransfer (VOID)
//
// DESCRIPTION:   Function to signal end of packet transfer
//
// PARAMETERS:    none
//
// RETURNS:       0 = transfer command sent
//               -1 = DSP not responding
//
//***************************************************************************

short SetAudioTransfer (VOID)
   {
   // Send command sequence

   DISABLE
   sendDSP (DSPTRANSFER);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short StartPlayback (UINT ch, LPBYTE ptr, DWORD bufsize,
//                                     DWORD sampsize)
//
// DESCRIPTION:   Function to start playback of digital audio.
//                If sampsize is greater than bufsize, system will loop
//                back to start of buffer and continue playing.  This
//                assumes that the sample buffer has been refilled with new
//                sample data.
//
// PARAMETERS:    ch       = audio channel (0 to 3)
//                ptr      = pointer to sample buffer
//                bufsize  = size of sample buffer (in bytes)
//                           must be a multiple of (packetSize * 2)
//                sampsize = size of entire sample (in bytes)
//
// RETURNS:       0 = sample started playing
//                1 = channel out of range
//                2 = ptr is NULL
//                3 = buffer size is zero or not a multiple of packetsize*2
//                4 = sample size is zero
//               -1 = DSP not responding
//               -2 = audio channel is already active
//
//***************************************************************************

short StartPlayback (UINT ch, LPBYTE ptr, DWORD bufsize, DWORD sampsize)
   {
   // Validate parameters

   if ( ((audioFormat & STEREO)   && ch >= MAXSTEREOCHANNELS) ||
        (playMode == NOSYNTH      && ch >= MODE0_CHANNELS)    ||
        (playMode == MINOPERATORS && ch >= MODE1_CHANNELS)    ||
        (playMode == MAXOPERATORS && ch >= MODE2_CHANNELS) )
      return 1;
   if (ptr == 0L)
      return 2;
   if (bufsize == 0L || bufsize % (packetSize << 1))
      return 3;
   if (sampsize == 0L)
      return 4;

   if (waveCh[ch].status != INACTIVE)
      return -2;

   // Send command sequence

   DISABLE
   waveCh[ch].audioPtr     = ptr;
   waveCh[ch].audioStart   = ptr;
   waveCh[ch].bufferSize   = bufsize;
   waveCh[ch].bufferOffset = bufsize;
   waveCh[ch].audioSize    = sampsize;
   waveCh[ch].status       = PLAYING;
   waveCh[ch].audioPending = ON;

   // Make sure PCM audio ISR is registered

   if (GetISRAddress (PCM_ID) == 0L)
      RegisterISR (PCM_ID, (LPBYTE) PCMproc);

   sendDSP (DSPPLAYSTART);
   sendDSP (ch);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short StopPlayback (UINT ch)
//
// DESCRIPTION:   Function to stop audio playback
//
// PARAMETERS:    ch = channel to stop (0 to 3)
//
// RETURNS:       0 = sample stopped
//                1 = channel out of range
//               -1 = DSP not responding
//               -2 = channel is not playing
//
//***************************************************************************

short StopPlayback (UINT ch)
   {
   // Validate parameter

   if ( ((audioFormat & STEREO)   && ch >= MAXSTEREOCHANNELS) ||
        (playMode == NOSYNTH      && ch >= MODE0_CHANNELS)    ||
        (playMode == MINOPERATORS && ch >= MODE1_CHANNELS)    ||
        (playMode == MAXOPERATORS && ch >= MODE2_CHANNELS) )
      return 1;

   if (waveCh[ch].status != PLAYING)
      return -2;
   else
      waveCh[ch].audioPending = OFF;

   // Send command sequence

   if (oldInt == 0L)
      {
      DISABLE
      sendDSP (DSPPLAYSTOP);
      sendDSP (ch);
      sendDSP (DSPTERMINATOR);
      ENABLE

      if (!DSPstatus)
         return -1;
      }

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetChannelVolume (UINT ch, UINT vol)
//
// DESCRIPTION:   Function to set digital audio channel playback volume
//
// PARAMETERS:    ch  = audio channel to adjust (0 to 3)
//                vol = new volume (0 to 0x7FFF)
//
// RETURNS:       0 = volume changed
//                1 = channel out of range
//                2 = volume out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetChannelVolume (UINT ch, UINT vol)
   {
   // Validate parameters

   if (ch >= MODE0_CHANNELS)
      return 1;
   if (vol > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPCHANNELVOL);
   sendDSP (ch);
   sendDSP (vol);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetChannelPan (UINT ch, UINT pan)
//
// DESCRIPTION:   Function to set digital audio channel playback pan
//
// PARAMETERS:    ch  = audio channel to adjust (0 to 3)
//                pan = new pan setting (0 to 127)
//
// RETURNS:       0 = pan changed
//                1 = channel out of range
//                2 = pan value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetChannelPan (UINT ch, UINT pan)
   {
   // Validate parameters

   if (ch >= MODE0_CHANNELS)
      return 1;
   if (pan > MAXPAN)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPCHANNELPAN);
   sendDSP (ch);
   sendDSP (pan);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short StartRecording (UINT ch, LPBYTE ptr, DWORD bufsize,
//                                      DWORD sampsize)
//
// DESCRIPTION:   Function to start recording of digital audio.
//                If sample size is greater than buffer size, the system
//                will loop back to the start of the buffer and continue
//                recording.
//
// PARAMETERS:    ch       = audio channel for recording (0 or 1)
//                ptr      = sample buffer pointer
//                bufsize  = size of sample buffer (in bytes)
//                           must be a multiple of (packet size * 2)
//                sampsize = size of entire sample to record
//
// RETURNS:       0 = recording started
//                1 = channel out of range
//                2 = ptr is NULL
//                3 = buffer size is zero or not a multiple of packetsize*2
//                4 = sample size is zero
//               -1 = DSP not responding
//               -2 = channel is already active
//
//***************************************************************************

short StartRecording (UINT ch, LPBYTE ptr, DWORD bufsize, DWORD sampsize)
   {
   // Validate parameters

   if ( ((audioFormat & STEREO)   && ch >= MAXSTEREOCHANNELS) ||
        (playMode == NOSYNTH      && ch >= MAXRECCHANNELS)    ||
        (playMode == MINOPERATORS && ch >= MODE1_CHANNELS)    ||
        (playMode == MAXOPERATORS && ch >= MODE2_CHANNELS) )
      return 1;
   if (ptr == 0L)
      return 2;
   if (bufsize == 0L || bufsize % (packetSize << 1))
      return 3;
   if (sampsize == 0L)
      return 4;

   if (waveCh[ch].status == PLAYING || waveCh[0].status == RECORDING ||
       waveCh[1].status == RECORDING)
      return -2;

   // Check for ADPCM recording.  If so, ensure NOSYNTH playback mode

   if ((audioFormat & ADPCM) && playMode != NOSYNTH)
      if (SetPlaybackMode (NOSYNTH) < 0)
         return -1;

   // Make sure PCM audio ISR is registered

   if (GetISRAddress (PCM_ID) == 0L)
      RegisterISR (PCM_ID, (LPBYTE) PCMproc);

   // Send command sequence

   DISABLE
   waveCh[ch].audioPtr     = ptr;
   waveCh[ch].audioStart   = ptr;
   waveCh[ch].bufferSize   = bufsize;
   waveCh[ch].bufferOffset = bufsize;
   waveCh[ch].audioSize    = sampsize;
   waveCh[ch].status       = RECORDING;
   waveCh[ch].audioPending = ON;

   sendDSP (DSPRECSTART);
   sendDSP (ch);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short StopRecording (UINT ch)
//
// DESCRIPTION:   Function to stop audio recording
//
// PARAMETERS:    ch = audio channel to stop
//
// RETURNS:       0 = recording stopped
//                1 = channel out of range
//               -1 = DSP not responding
//               -2 = channel is not recording
//
//***************************************************************************

short StopRecording (UINT ch)
   {
   // Validate parameter

   if ( ((audioFormat & STEREO)   && ch >= MAXSTEREOCHANNELS) ||
        (playMode == NOSYNTH      && ch >= MAXRECCHANNELS)    ||
        (playMode == MINOPERATORS && ch >= MODE1_CHANNELS)    ||
        (playMode == MAXOPERATORS && ch >= MODE2_CHANNELS) )
      return 1;

   if (waveCh[ch].status != RECORDING)
      return -2;
   else
      waveCh[ch].audioPending = OFF;

   // Send command sequence

   if (oldInt == 0L)
      {
      DISABLE
      sendDSP (DSPRECSTOP);
      sendDSP (ch);
      sendDSP (DSPTERMINATOR);
      ENABLE

      if (!DSPstatus)
         return -1;
      }

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetPlaybackVolume (UINT left, UINT right)
//
// DESCRIPTION:   Function to set playback volume for all digital audio
//                channels
//
// PARAMETERS:    left  = left output channel volume (0 to 0x7FFF)
//                right = right output channel volume (0 to 0x7FFF)
//
// RETURNS:       0 = playback volume set
//                1 = left volume out of range
//                2 = right volume out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetPlaybackVolume (UINT left, UINT right)
   {
   // Validate parameters

   if (left > MAXVOL)
      return 1;
   if (right > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPPLAYVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetRecordingVolume (UINT source, UINT left,
//                                          UINT right)
//
// DESCRIPTION:   Function to set recording level.  For mono recording,
//                the left volume level is used.
//
// PARAMETERS:    source = recording source (EXTREC or AUXREC)
//                left   = left channel record level (0 to 0x7FFF)
//                right  = right channel record level (0 to 0x7FFF)
//
// RETURNS:       0 = recording level set
//                1 = source value out of range
//                2 = left channel level out of range
//                3 = right channel level out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetRecordingVolume (UINT source, UINT left, UINT right)
   {
   // Validate parameters

   if (source != EXTREC && source != AUXREC)
      return 1;
   if (left > MAXVOL)
      return 2;
   if (right > MAXVOL)
      return 3;

   // Send command sequence

   DISABLE
   if (source == EXTREC)
      sendDSP (DSPEXTVOL);
   else
      sendDSP (DSPAUXVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short OperatorOn (UINT op, UINT wpt, UINT wave, UINT amp,
//                                  UINT pitch, UINT size, UINT type);
//
// DESCRIPTION:   Function to turn on a Aria Wave Synthesis operator
//
// PARAMETERS:    op    = operator number to activate (0 to 31)
//                wpt   = wave parameter table number to use (0 to 79)
//                wave  = wave block number (0x0000 to 0xFFFF)
//                amp   = operator maximum amplitude (0 to 0x7FFF)
//                pitch = frequency value xxyy (xx from 0 to 0xFE,
//                                              yy from 0 to 0xFF)
//                size  = number of wave blocks (1 to 256)
//                type  = 0 for multiple cycles of waveform
//                        1 for single play of waveform
//
// RETURNS:       0 = operator turned on
//                1 = operator number out of range
//                2 = wave parameter table number out of range
//                4 = amplitude value out of range
//                5 = pitch value out of range
//                6 = block size out of range
//               -1 = DSP not responding
//
//***************************************************************************

short OperatorOn (UINT op, UINT wpt, UINT wave, UINT amp, UINT pitch,
                  UINT size, UINT type)
   {
   // Validate parameters

   if (op >= wMaxSynthOp)
      return 1;
   if (wpt >= MAXWPTS)
      return 2;
   if (amp > MAXVOL)
      return 4;
   if (pitch == 0xFFFF || pitch == 0)
      return 5;
   if (size > 256)
      return 6;

   // Send command sequence

   if (type)   // Single
      {
      DISABLE
      sendDSP (DSPOPERON2);
      sendDSP (op);
      sendDSP (wpt);
      sendDSP (wave);
      sendDSP (amp);
      sendDSP (pitch);
      sendDSP ((UINT) ((((DWORD) size) << 10) / (DWORD) pitch));
      sendDSP (DSPTERMINATOR);
      ENABLE
      }
   else        // Loop
      {
      DISABLE
      sendDSP (DSPOPERON1);
      sendDSP (op);
      sendDSP (wpt);
      sendDSP (wave);
      sendDSP (amp);
      sendDSP (pitch);
      sendDSP (size);
      sendDSP (DSPTERMINATOR);
      ENABLE
      }

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short OperatorOff (UINT op)
//
// DESCRIPTION:   Function to turn off an active operator.  Will play last
//                two envelope stages.
//
// PARAMETERS:    op = operator number to turn off
//
// RETURNS:       0 = operator turned off
//                1 = operator number out of range
//               -1 = DSP not responding
//
//***************************************************************************

short OperatorOff (UINT op)
   {
   // Validate parameter

   if (op >= DSPMAXOPS)
      return 1;

   // Send command sequence

   DISABLE
   sendDSP (DSPOPEROFF);
   sendDSP (op);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short OperatorCancel (UINT op)
//
// DESCRIPTION:   Function to turn off an active operator.  Will not play
//                last two stages of envelope, but instead terminate
//                immediately.
//
// PARAMETERS:    op = operator number to turn off
//
// RETURNS:       0 = operator canceled
//                1 = operator number out of range
//               -1 = DSP not responding
//
//***************************************************************************

short OperatorCancel (UINT op)
   {
   // Validate parameter

   if (op >= DSPMAXOPS)
      return 1;

   // Send command sequence

   DISABLE
   sendDSP (DSPOPERCANCEL);
   sendDSP (op);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short TableModify (UINT type, UINT table, UINT offset,
//                                   UINT value)
//
// DESCRIPTION:   Function to modify a specific value in the wave parameter
//                table or envelope multiplier table.
//
// PARAMETERS:    type   = table type:  0 wave parameter table
//                                      1 envelope multiplier table
//                table  = table number:
//                           0 to 79 for wave parameter tables
//                           0       for envelope multiplier table
//                offset = word offset in table to modify
//                           for wave parameter tables:
//                             0 to 7  envelope stage parameter
//                           for envelope multiplier table:
//                             0 to 63 multiplier values for envelope
//                                     scaling
//
//                value  = new data value to set (0 to 0xFFFE)
//
// RETURNS:       0 = table modified
//                1 = type value out of range
//                2 = table value out of range
//                3 = offset value out of range
//                4 = data value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short TableModify (UINT type, UINT table, UINT offset, UINT value)
   {
   // Validate parameters

   switch (type)
      {
      case 0:
         if (table >= MAXWPTS)
            return 2;
         if (offset > 15)
            return 3;
         break;

      case 1:
         if (table != 0)
            return 2;
         if (offset > 63)
            return 3;
         break;

      default:
         return 1;
      }
   if (value == 0xFFFF)
      return 4;

   // Send command sequence

   DISABLE
   sendDSP (DSPTABLEMOD);
   sendDSP (type);
   sendDSP (table);
   sendDSP (offset);
   sendDSP (value);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short LFOactivate (UINT LFOnum, UINT type, UINT pitch)
//
// DESCRIPTION:   Function to activate a LFO generator
//
// PARAMETERS:    LFOnum = LFO generator number (0 to 15)
//                type   = type or shape of LFO:
//                           0  ramp
//                           1  reverse ramp
//                           2  triangle
//                           3  square
//                           4  pulse
//                           5  staircase
//                           6  sine wave
//                pitch  = LFO frequency value xxyy (xx from 0 to 0x3F,
//                                                   yy from 0 to 0xFF)
//
// RETURNS:       0 = LFO activated
//                1 = LFO number out of range
//                2 = LFO type value out of range
//                3 = LFO pitch value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short LFOactivate (UINT LFOnum, UINT type, UINT pitch)
   {
   // Validate parameters

   if (LFOnum >= MAXLFOS)
      return 1;
   if (type > 6)
      return 2;
   if ((pitch >> 8) > 0x3F)
      return 3;

   // Send command sequence

   DISABLE
   sendDSP (DSPLFO_ON);
   sendDSP (LFOnum);
   sendDSP (type);
   sendDSP (pitch);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short LFOdeactivate (UINT LFOnum)
//
// DESCRIPTION:   Function to deactivate an LFO generator
//
// PARAMETERS:    LFOnum = LFO generator number to deactivate (0 to 15)
//
// RETURNS:       0 = LFO deactivated
//                1 = LFO number out of range
//               -1 = DSP not responding
//
//***************************************************************************

short LFOdeactivate (UINT LFOnum)
   {
   // Validate parameter

   if (LFOnum >= MAXLFOS)
      return 1;

   // Send command sequence

   DISABLE
   sendDSP (DSPLFO_OFF);
   sendDSP (LFOnum);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetLFOrate (UINT LFOnum, UINT rate)
//
// DESCRIPTION:   Function to change the rate of an LFO generator
//
// PARAMETERS:    LFOnum = LFO generator number (0 to 15)
//                rate   = LFO frequency value xxyy (xx from 0 to 0x3F,
//                                                   yy from 0 to 0xFF)
//
// RETURNS:       0 = LFO rate set
//                1 = LFO number out of range
//                2 = LFO rate value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetLFOrate (UINT LFOnum, UINT rate)
   {
   // Validate parameters

   if (LFOnum >= MAXLFOS)
      return 1;
   if ((rate >> 8) > 0x3F)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPLFORATE);
   sendDSP (LFOnum);
   sendDSP (rate);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetLFOdepth (UINT wpt, UINT type, UINT depth)
//
// DESCRIPTION:   Function to change the depth of an LFO
//
// PARAMETERS:    wpt   = wave parameter table number (0 to 79)
//                type  = 0  Pitch LFO (vibrato)
//                        1  Amplitude LFO (tremolo)
//                        2  Pan LFO
//                depth = LFO depth value (0 to 255)
//
// RETURNS:       0 = LFO depth set
//                1 = wave parameter table number out of range
//                2 = type value out of range
//                3 = depth value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetLFOdepth (UINT wpt, UINT type, UINT depth)
   {
   // Validate parameters

   if (wpt >= MAXWPTS)
      return 1;
   if (type > 2)
      return 2;
   if (depth > 255)
      return 3;

   // Send command sequence

   DISABLE
   sendDSP (DSPLFODEPTH);
   sendDSP (wpt);
   sendDSP (type);
   sendDSP (depth);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetOperatorVolume (UINT op, UINT vol)
//
// DESCRIPTION:   Function to set the volume level of an operator
//
// PARAMETERS:    op  = operator number (0 to 31)
//                vol = new volume level (0 to 0x7FFF)
//
// RETURNS:       0 = operator volume set
//                1 = operator number out of range
//                2 = volume level out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetOperatorVolume (UINT op, UINT vol)
   {
   // Validate parameters

   if (op >= DSPMAXOPS)
      return 1;
   if (vol > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPAMP);
   sendDSP (op);
   sendDSP (vol);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetOperatorPan (UINT op, UINT pan)
//
// DESCRIPTION:   Function to set the stereo pan of an operator
//
// PARAMETERS:    op  = operator number (0 to 31)
//                pan = new stereo pan (0 to 127)
//
// RETURNS:       0 = operator pan set
//                1 = operator number out of range
//                2 = stereo pan out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetOperatorPan (UINT op, UINT pan)
   {
   // Validate parameters

   if (op >= DSPMAXOPS)
      return 1;
   if (pan > MAXPAN)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPPAN);
   sendDSP (op);
   sendDSP (pan);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetOperatorPitch (UINT op, UINT pitch)
//
// DESCRIPTION:   Function to change the pitch of an operator
//
// PARAMETERS:    op    = operator number (0 to 31)
//                pitch = new pitch value xxyy (xx from 0 to 0xFE,
//                                              yy from 0 to 0xFF)
//
// RETURNS:       0 = operator pitch set
//                1 = operator number out of range
//                2 = pitch value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetOperatorPitch (UINT op, UINT pitch)
   {
   // Validate parameters

   if (op >= DSPMAXOPS)
      return 1;
   if (pitch == 0xFFFF)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPPITCH);
   sendDSP (op);
   sendDSP (pitch);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetSynthVolume (UINT left, UINT right)
//
// DESCRIPTION:   Function to set the volume level of the synthesis output
//
// PARAMETERS:    left  = left channel synth volume (0 to 0x7FFF)
//                right = right channel synth volume (0 to 0x7FFF)
//
// RETURNS:       0 = synthesis volume set
//                1 = left channel volume out of range
//                2 = right channel volume out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SetSynthVolume (UINT left, UINT right)
   {
   // Validate parameters

   if (left > MAXVOL)
      return 1;
   if (right > MAXVOL)
      return 2;

   // Send command sequence

   DISABLE
   sendDSP (DSPSYNTHVOL);
   sendDSP (left);
   sendDSP (right);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SelectADC (UINT mode)
//
// DESCRIPTION:   Function to turn ADC on or off
//
// PARAMETERS:    mode = 0 ADC off
//                       1 ADC on
//
// RETURNS:       0 = ADC mode switched
//                1 = mode value out of range
//               -1 = DSP not responding
//
//***************************************************************************

short SelectADC (UINT mode)
   {
   // Validate parameter

   if (mode > 1)
      return 1;

   // Send command sequence

   return SetADC (mode);
   }


//***************************************************************************
//
// FUNCTION:      short SetMonitorMode (UINT source, UINT mode)
//
// DESCRIPTION:   Function to set the mixer monitor and recording modes
//
// PARAMETERS:    source = One of the following input sources:
//                           0001h  Microphone
//                           0002h  CD-ROM audio (AUX2)
//                           0008h  External
//                           0010h  Auxiliary 1 (Mixed output)
//                           0020h  PC Speaker/Telephone input
//                mode   = 0 for stereo record and monitoring
//                         1 for mono left channel record and monitoring
//                         2 for mono right channel record and monitoring
//                         3 Monitor off
//
// RETURNS:       0 = monitor mode set
//                1 = source out of range
//                2 = mode out of range
//               -1 = DSP not responding
//               -2 = Command not available (Mixer chip only)
//
//***************************************************************************

short SetMonitorMode (UINT source, UINT mode)
   {
   // Validate parameters

   if (!MIXERCHIP)
      return -2;
   if (source != MICROPHONE && source != CDROMAUDIO && source != EXTERNAL &&
       source != AUX1AUDIO  && source != PHONEAUDIO)
      return 1;
   if (mode > 3)
      return 2;

   // Send command sequences

   DISABLE
   sendDSP (DSPMONITORMODE);
   sendDSP (source);
   sendDSP (mode);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (mode < 2 && (source == AUX1AUDIO || !(recSource & AUX1AUDIO)))
      {
      // Also, set recording mode for input channel

      DISABLE
      sendDSP (DSPMONITORMODE);
      sendDSP (source | 0x8000);
      sendDSP (mode);
      sendDSP (DSPTERMINATOR);
      ENABLE
      }

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetMasterMode (UINT mode)
//
// DESCRIPTION:   Function to set the master output mode (mono or stereo)
//                and output record connection
//
// PARAMETERS:    mode = LOBYTE:
//                       0 for stereo monitoring
//                       1 for mono monitoring
//
//                       HIBYTE:
//                       0 for disconnecting output record signal
//                       1 for connecting output record signal
//
// RETURNS:       0 = master mode set
//                1 = mode out of range
//               -1 = DSP not responding
//               -2 = Command not available (Mixer chip only)
//
//***************************************************************************

short SetMasterMode (UINT mode)
   {
   // Validate parameter

   if (!MIXERCHIP)
      return -2;
   if ((mode & 0xFF) > 1 || (mode >> 8) > 1)
      return 1;

   // Send command sequence

   DISABLE
   sendDSP (DSPMIXERMODE);
   sendDSP (mode);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetMixerVolume (UINT source, DWORD volume)
//
// DESCRIPTION:   Function to set individual mixer volume controls
//
// PARAMETERS:    source = Any of the following sources (ORed together):
//                           0001h  Microphone
//                           0002h  CD-ROM audio (AUX2)
//                           0004h  Master volume
//                           0008h  External
//                           0010h  Auxiliary 1 (Mixed output)
//                           0020h  PC Speaker/Telephone input
//                left   = Left channel volume (0 to 0x7FFF)
//                right  = Right channel volume (0 to 0x7FFF)
//
// RETURNS:       0 = volume(s) set
//                1 = source out of range
//                2 = left volume out of range
//                3 = right volume out of range
//               -1 = DSP not responding
//               -2 = Command not available (Mixer chip only)
//
//***************************************************************************

short SetMixerVolume (UINT source, UINT left, UINT right)
   {
   UINT mask;

   // Validate parameters

   if (!MIXERCHIP)
      return -2;
   if (source > 63)
      return 1;
   if (left > 0x7FFF)
      return 2;
   if (right > 0x7FFF)
      return 3;

   for (mask = 1; mask < 0x40; mask <<= 1)
      {
      if (mask & source)
         {
         // Send command sequence

         DISABLE
         sendDSP (DSPMIXERVOL);
         sendDSP (mask);
         sendDSP (left);
         sendDSP (right);
         sendDSP (DSPTERMINATOR);
         ENABLE

         if (!DSPstatus)
            return -1;
         }
      }

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      short SetToneControl (UINT left, UINT right)
//
// DESCRIPTION:   Function to set the mixer tone controls
//
// PARAMETERS:    left  = Left channel tone.  Treble must be placed in the
//                        high-order byte, bass must be placed in the low-
//                        order byte:
//                          0        flat response
//                          1 to 7   max. cut to min. cut freq.
//                          8        flat response
//                          9 to 15  min. boost to max. boost freq.
//                right = Right channel tone.  Parameter is same as above.
//
// RETURNS:       0 = tone control set
//                1 = left parameter out of range
//                2 = right parameter out of range
//               -1 = DSP not responding
//               -2 = Command not available (Mixer chip only)
//
//***************************************************************************

short SetToneControl (UINT left, UINT right)
   {
   UINT leftval, rightval;
   static WORD toneval[16] = {0,7,6,5,4,3,2,1,8,9,10,11,12,13,14,15};

   // Validate parameters

   if (!MIXERCHIP)
      return -2;
   if (left & 0xF0F0)
      return 1;
   if (right & 0xF0F0)
      return 2;

   // Send command sequence

   leftval  = (UINT) ((toneval[left >>8] << 8) | toneval[left &0x0F]);
   rightval = (UINT) ((toneval[right>>8] << 8) | toneval[right&0x0F]);

   DISABLE
   sendDSP (DSPTONECTRL);
   sendDSP (leftval);
   sendDSP (rightval);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//
// FUNCTION:      DWORD GetAudioPosition (UINT ch)
//
// DESCRIPTION:   Function to return the number of bytes remaining to be
//                played in a sample
//
// PARAMETERS:    ch = audio channel (0 to 3)
//
// RETURNS:       Number of bytes from end of sample
//                0xFFFFFFFF, if ch out of range
//
//***************************************************************************

DWORD GetAudioPosition (UINT ch)
   {
   // Validate parameter

   if (ch > MODE0_CHANNELS)
      return 0xFFFFFFFFL;

   // Check to see if last packet is pending

   if (waveCh[ch].audioSize == 0L && waveCh[ch].audioPending == OFF)
      return (DWORD) packetSize;

   return waveCh[ch].audioSize;
   }


//***************************************************************************
//
// FUNCTION:      DWORD GetBufferPosition (UINT ch)
//
// DESCRIPTION:   Function to return the number of bytes remaining to be
//                played in a sample buffer
//
// PARAMETERS:    ch = audio channel (0 to 3)
//
// RETURNS:       Number of bytes from end of buffer
//                0xFFFFFFFF, if ch out of range
//
//***************************************************************************

DWORD GetBufferPosition (UINT ch)
   {
   // Validate parameter

   if (ch > MODE0_CHANNELS)
      return 0xFFFFFFFFL;

   // Check to see if last packet is pending

   if (waveCh[ch].bufferOffset == 0L && waveCh[ch].audioPending == OFF)
      return (DWORD) packetSize;

   return waveCh[ch].bufferOffset;
   }


//***************************************************************************
//
// FUNCTION:      VOID putMem16 (UINT dmaAddr, UINT data)
//
// DESCRIPTION:   Function to write data into external DSP RAM
//
// PARAMETERS:    dmaAddr = DSP address (0x6000 to 0x7FFF)
//                data    = data value (0 to 0xFFFF)
//
// RETURNS:       none
//
//***************************************************************************

VOID putMem16 (UINT dmaAddr, UINT data)
   {
   DISABLE
   OUTPW (DMAADDRESS, (WORD) dmaAddr);
   OUTPW (DMADATA, (WORD) data);
   ENABLE
   }


//***************************************************************************
//
// FUNCTION:      UINT getMem16 (UINT dmaAddr)
//
// DESCRIPTION:   Function to read data from external DSP RAM
//
// PARAMETERS:    dmaAddr = DSP address (0x6000 to 0x7FFF)
//
// RETURNS:       Data word found at specified location
//
//***************************************************************************

UINT getMem16 (UINT dmaAddr)
   {
   UINT i;

   DISABLE
   OUTPW (DMAADDRESS, (WORD) dmaAddr);
   i = (UINT) INPW (DMADATA);
   ENABLE

   return i;
   }


//***************************************************************************
//
// FUNCTION:      short sendDSP (UINT data)
//
// DESCRIPTION:   Function to send a command word to the DSP
//
// PARAMETERS:    data = Command to send (0 to 0xFFFE = data,
//                                        0xFFFF = command terminator)
//
// RETURNS:       0 = DSP not responding
//                1 = data sent to DSP
//
//***************************************************************************

short sendDSP (UINT data)
   {
   static short i;   // Time out value

   // Wait until DSP DATA Port is ready

   i = 30000;
   while ((INPW (DSPSTATUS) & PORT0_BUSY) && i)
      --i;

   OUTPW (DSPDATA, (WORD) data);

   if (!i)
      DSPstatus = 0;
   else
      DSPstatus = 1;

   return DSPstatus;
   }


static VOID Delay (UINT wait)
   {
   while (wait--)
      INPW (DSPCONTROL);
   }


#ifdef TURBOC
VOID disableintr (VOID) 
   {
   asm   pushf
   asm   pop   ax
   asm   test  ax, 2
   asm   jz    next1
   asm   cmp   disabled, 0
   asm   ja    next1
   asm   cli
next1:
   asm   inc   disabled
   return;
   }

VOID enableintr (VOID)
   {
   asm   dec   disabled
   asm   cmp   disabled, 0
   asm   ja    next2
   asm   sti
next2:
   return;
   }
#endif


static VOID interrupt FAR IntTest1 (VOID)
   {
   dspint = ARIAINT1;
   gwIntCount++;

   // End Of Interrupt

   OUTP (INTCTRLR1-1, 0x20);
   OUTP (INTCTRLR0-1, 0x20);
   }


static VOID interrupt FAR IntTest2 (VOID)
   {
   dspint = ARIAINT2;
   gwIntCount++;

   // End Of Interrupt

   OUTP (INTCTRLR1-1, 0x20);
   OUTP (INTCTRLR0-1, 0x20);
   }


static VOID interrupt FAR IntTest3 (VOID)
   {
   dspint = ARIAINT3;
   gwIntCount++;

   // End Of Interrupt

   OUTP (INTCTRLR1-1, 0x20);
   OUTP (INTCTRLR0-1, 0x20);
   }

