//***************************************************************************
//
// Copyright (c) 1993 Sierra Semiconductor Corp.
//
// FILE:    ariadsp.c
//
// LANGUAGES:
//          Microsoft C Versions 5.0, 6.0, and 7.0
//          Borland Turbo C Versions 1.5 and 2.0
//
// DESCRIPTION:
//
//          Aria DSP Application Function Library
//
//          For Microsoft C 5.0, 6.0, and 7.0
//          =================================
//          compile:  cl /Zlp /Od /c ariadsp.c    (for small model)
//
//                    for medium  model include /AM option
//                    for compact model include /AC option
//                    for large   model include /AL option
//
//          For Borland Turbo C
//          ===================
//          compile:  bcc -DTURBOC -a- -O- -c ariadsp.c    (for small model)
//
//                    for medium  model include -mm option
//                    for compact model include -mc option
//                    for large   model include -ml option
//
// AdLib is a trademark of AdLib Inc.
// Aria, Aria Synthesizer and Aria Listener are trademarks of Sierra
//  Semiconductor Corp.
// QSound is a trademark of Archer Communications
// Sound Blaster is a trademark of Creative Labs Inc.
//
//***************************************************************************

// $Header:   F:\projects\ariai\dos\archives\ariadsp.c_v   2.5   08 Dec 1993 13:51:20   golds  $
// $Log:   F:\projects\ariai\dos\archives\ariadsp.c_v  $
// 
//    Rev 2.5   08 Dec 1993 13:51:20   golds
// Fixed bug in ADPCM play/record for older ROMs (version 1.6)
// 
//    Rev 2.2   03 Sep 1993 09:59:06   golds
// Fixed recording from external input source
// 
//    Rev 2.1   13 Aug 1993 09:12:18   golds
// New reverb commands, 16-channel MIDI, other optimizations
// 
//    Rev 2.0   24 Jun 1993 12:50:36   golds
// Initial revision.

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>

#include "aria.h"
#include "ariadsp.h"

#define LOAD   0     // Load DSP app flag
#define UNLOAD 1     // Unload DSP app flag

#define STFL   'K'   // Program ID
#define DATAFL 'B'   // Absolute Data Value
#define ADDRFL '9'   // Absolute Address
#define EOL    '7'   // Checksum Data
#define EOL1   '8'   // Checksum Data
#define ENDLN  '\n'  // End of text line
#define ENDFL  ':'   // Final Record

// Macros to return low or high word of long value

#define LOWORD(l) ((WORD)(l))
#define HIWORD(l) ((WORD)(((DWORD)(l) >> 16) & 0xFFFF))

// Macros to return low or high byte of word value

#define LOBYTE(w) ((BYTE)(w))
#define HIBYTE(w) ((BYTE)(((WORD)(w) >> 8) & 0xFF))

static WORD DSPusage = 0;  // DSP utilization (in percent)

                           // DSP application pointers
static LPBYTE hApp[MAXAPPS] = {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
                               NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};

                           // DSP application maximum utilization,
                           // (MAXDSPUSAGE = 100%)
static WORD wMaxUsage[MAXAPPS]={ 625,  DSPMAXUSAGE, 1416, 7420,  
                                   0,         3100,    0, 2500,
                                2500,            0,    0,    0,  
                                   0,            0,    0,    0};

                           // DSP application current utilization
static WORD wCurrentUsage[MAXAPPS]={0,DSPMAXUSAGE,0,0,
                                    0,0,0,0,
                                    0,0,0,0,
                                    0,0,0,0};

                           // DSP application load flag
static BOOL appLoaded[MAXAPPS]={FALSE,TRUE, FALSE,FALSE,
                                FALSE,FALSE,FALSE,FALSE,
                                FALSE,FALSE,FALSE,FALSE,
                                FALSE,FALSE,FALSE,FALSE};

static BOOL spareLoaded=FALSE;   // Downloadable application flag -
                                 //  Some systems only allow one app ID
                                 //  to be downloaded from the host

static UINT reverbLevelL=REVERBLEVELDEF;  // Reverb overall level, left ch
static UINT reverbLevelR=REVERBLEVELDEF;  // Reverb overall level, right ch
static UINT waveReverb = WREVERB_DEFAULT; // Reverb level, wave channels

BOOL maxLoaded=FALSE;      // System allocated flag

//*** External variable declarations ***

extern ARIACAPS AriaCap;
extern WORD wMaxSynthOp;
extern WORD wMaxPercOp;
extern WORD recSource;
extern WORD version;
extern WORD playMode;
extern WORD DSPstatus;
extern WORD speechMode;
extern WORD qsoundMode;
extern WORD reverbMode;

//*** Local function declarations ***

static short SwitchApp (WORD, WORD, WORD);
static short loadobjfile (LPBYTE);
static WORD  CalculateDSPUsage (VOID);
static short SystemWait (VOID);
static short DSPLoadApp (WORD, WORD);
static short DSPUnloadApp (WORD);
static short DSPLoadDiskApp (LPBYTE);

//*** External function declaration ***

extern VOID SetPanControls (VOID);
extern VOID SetReverbControls (VOID);

#ifdef TURBOC
extern VOID disableintr (VOID);
extern VOID enableintr (VOID);
#endif


//***************************************************************************
//  When synthesis operator count changes, this function recomputes DSPusage
//***************************************************************************

VOID SetDSPUsage (UINT mode, UINT opcount)
   {
   register WORD wavecount;

   switch (mode)
      {
      case 0:
         wavecount = 4;
         opcount = 0;
         break;
      case 1:
         wavecount = 2;
         if (!opcount)
            opcount = DSPMINOPS;
         break;
      case 2:
         wavecount = 2;
         if (!opcount)
            opcount = DSPMAXOPS;
         break;
      }
   wCurrentUsage[2] = wavecount * DSPWAVEUNIT;
   wCurrentUsage[3] = (opcount? DSPSYNTHBASIC: 0) + (opcount * DSPSYNTHUNIT);

   DSPusage = CalculateDSPUsage ();
   }


//***************************************************************************
//  Function to return DSP usage (0 to 100%)
//***************************************************************************

UINT GetDSPUsage (VOID)
   {
   DSPusage = CalculateDSPUsage ();

   return ((UINT) DSPusage / (DSPMAXUSAGE/100));
   }


//***************************************************************************
//  This function sets the DSP playback mode
//***************************************************************************

short SetMode (UINT mode, UINT opcount)
   {
   if ((playMode != (WORD) mode && mode <= 2) || opcount)
      {
      DISABLE
      sendDSP (DSPPLAYMODE);
      sendDSP ((UINT) ((opcount << 8) | mode));
      sendDSP (DSPTERMINATOR);
      playMode = (WORD) mode;
      ENABLE
   
      SetDSPUsage (mode, opcount);
      }
   if (!DSPstatus)
      return (-1);

   return (0);
   }


//***************************************************************************
//  This function loads a ROM-based DSP application
//***************************************************************************

static short DSPLoadApp (WORD app, WORD taddr)
   {
   // Zero the DSP application ready flag

   putMem16 (DSP_INIT, 0);

   // Load the application

   if (SwitchApp (app, LOAD, taddr) != 0)
      return -1;

   return SystemWait ();
   }


//***************************************************************************
//  This function unloads a DSP application
//***************************************************************************

static short DSPUnloadApp (WORD app)
   {
   if (hApp[app] != NULL && spareLoaded)   // Check for app 6 loaded
      {
      app = SPARE_ID;
      spareLoaded = FALSE;
      }

   return SwitchApp (app, UNLOAD, 0);
   }


//***************************************************************************
//  This function sends the DSP application load message
//***************************************************************************

static short SwitchApp (WORD app, WORD val, WORD taddr)
   {
   DISABLE
   sendDSP (DSPSYSTEM);
   sendDSP (val);
   sendDSP (app);
   sendDSP (taddr);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//  This function turns the ADC on or off
//***************************************************************************

short SetADC (UINT mode)
   {
   static WORD ADCstate=1;

   if (recSource & DSPOUTPUT)    // If recording from synth/wave, ADC is
      return 0;                  // already off.

   if ((WORD) mode == ADCstate)  // Is current state already correct?
      return 0;

   DISABLE
   sendDSP (DSPADCSWITCH);
   sendDSP (mode);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   ADCstate = (WORD) mode;
   return 0;
   }


//***************************************************************************
//  This function is called when Aria Listener is turned on or off
//***************************************************************************

short SetListenerMode (UINT mode)
   {
   if (!appLoaded[SPEECH_ID])
      return -2;
   if (mode > 1)
      return 1;

   speechMode = (WORD) mode;

   return SetPlaybackMode (mode? MINOPERATORS: MAXOPERATORS);
   }


//***************************************************************************
//  This function sets the reverb mode
//***************************************************************************

short SetReverbMode (UINT mode)
   {
   register short err=0;

   if (!appLoaded[REVERB_ID])
      return -2;
   if (mode == 0 || mode > 7)
      return 1;

   DISABLE
   sendDSP (DSPREVERBMODE);
   sendDSP (mode-1);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   if ((err = SetReverbLevel (reverbLevelL, reverbLevelR)) != 0)
      return err;

   if (!reverbMode)
      {
      SetReverbControls ();
      if ((err = SetReverbWave (waveReverb)) != 0)
         return err;
      }
   reverbMode = (WORD) mode;
   return 0;
   }


//**************************************************************************
//  This function sets the reverb level for all wave output channels
//**************************************************************************

short SetReverbWave (UINT level)
   {
   if (!appLoaded[REVERB_ID])
      return -2;
   if (level > 0x7FFF)
      return 1;

   DISABLE
   sendDSP (DSPREVERBWAVE);
   sendDSP (level);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   waveReverb = level;

   return 0;
   }


//**************************************************************************
//  This functions sets the reverb level for a Wave Parameter Table
//**************************************************************************

short SetReverbChannel (UINT wpt, UINT level)
   {
   if (!appLoaded[REVERB_ID])
      return -2;
   if (wpt >= MAXWPTS)
      return 1;
   if (level > 0x7FFF)
      return 2;

   DISABLE
   sendDSP (DSPREVERBCHAN);
   sendDSP (wpt);
   sendDSP (level);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   return 0;
   }


//***************************************************************************
//  This function sets the overall left and right channel reverb levels
//***************************************************************************

short SetReverbLevel (UINT left, UINT right)
   {
   if (!appLoaded[REVERB_ID])
      return -2;
   if (left > 127)
      return 1;
   if (right > 127)
      return 2;

   DISABLE
   sendDSP (DSPREVERBLEVEL);
   sendDSP (left << 8);
   sendDSP (right << 8);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   reverbLevelL = left;
   reverbLevelR = right;

   return 0;
   }


//***************************************************************************
//  This function sets the QSound effects mode
//***************************************************************************

short SetQSoundMode (UINT mode)
   {
   if (!appLoaded[QSOUND_ID])
      return -2;
   if (mode > 3)
      return 1;

   // select QSound mode

   DISABLE
   sendDSP (DSPQSOUNDMODE);
   sendDSP (mode);
   sendDSP (DSPTERMINATOR);
   ENABLE

   if (!DSPstatus)
      return -1;

   qsoundMode = (WORD) mode;
   SetPanControls ();

   return 0;
   }


//***************************************************************************
//  This function loads a disk-based DSP application
//***************************************************************************

static short DSPLoadDiskApp (LPBYTE hBuf)
   {
   register WORD app;
   register short ret=3;
   LPBYTE bufptr;
   DSP_APP_HEADER FAR *headerptr;
   DSP_APP_DATA   FAR *dataptr;

   if ((bufptr = hBuf) == NULL)
      return ret;

   headerptr = (DSP_APP_HEADER FAR *) bufptr;
   if (headerptr->ckID != FOURCC_RDSP)
      return ret;

   bufptr += 8L;
   headerptr = (DSP_APP_HEADER FAR *) bufptr;
   if (((headerptr->ckID != FOURCC_DSPH) &&
        (headerptr->ckID != FOURCC_DSPH_OLD)) ||
        (headerptr->wMid != 0) ||
       (AriaCap.RAMsize < (headerptr->dwRAMMap << 3)) ||
       (!(headerptr->dwDSPMap & ((WORD) 1 << wDSPpart))) ||
       ((headerptr->bMaxUsage + DSPusage) > DSPMAXUSAGE))
      return ret;

   bufptr += (headerptr->ckSize + 8L);
   dataptr = (DSP_APP_DATA FAR *) bufptr;
   if (dataptr->ckID != FOURCC_DSPD && dataptr->ckID != FOURCC_DSPD_OLD)
      return ret;

   bufptr += sizeof (DSP_APP_DATA);

   DISABLE
   ret = loadobjfile (bufptr);
   ENABLE

   if (ret == 0)
      {
      if (!version)
         GetVersion ();
      if (wROMpart <= SC18054 && version <= 0x015E)
         {                             // If SD8025 (SC18051/2/3/4 ROM) and
         spareLoaded = TRUE;           // ROM version 1.94 or lower,
         app = SPARE_ID;               //   Set load flag (app 6)
         }
      else
         app = headerptr->wAppID;

      ret = DSPLoadApp (0x0100 | app, LOWORD (dataptr->dwAddr));
      if (ret != 0 && spareLoaded)
         spareLoaded = FALSE;
      }

   return ret;
   }


//***************************************************************************
//  This function loads or unloads a DSP application
//***************************************************************************

short SelectDSPApp (UINT app, BOOL load, LPBYTE appAddr)
   {
   register WORD i;
   register short ret=-1;
   BOOL minOps=FALSE;

   if (app >= MAXAPPS)
      return 1;

   if (app == REVERB_ID || app == QSOUND_ID || app == SPEECH_ID)
      minOps = TRUE;

   if (load)   // Load a DSP application
      {
      if (app > DEFAULT_SYNTH)
         {
         if (appLoaded[app])
            return 2;

         if (appLoaded[DEFAULT_SYNTH])
            return -2;  // Device disabled
         }

      // Check for overallocation of DSP resources and
      // check for older SD8025 ROM version (1.94 and below) that do not
      // allow more than one disk-based application to be loaded.

      if (app > DEFAULT_SYNTH)
         {
         if (!version)
            GetVersion ();

         if (minOps)
            SetPlaybackMode (MINOPERATORS);
            
         if ( maxLoaded ||
              ((wMaxUsage[app] + DSPusage) > DSPMAXUSAGE) ||
              ((wROMpart <= SC18054) && (version <= 0x015E) &&
               (appAddr != NULL) && spareLoaded) )
            return -3;  // Overallocation
         }

      // Check available application flag

      if ((AriaCap.DSPapps & ((WORD) 1 << app)) ||
           app == ARIA_SYNTH ||
           appAddr != NULL)
         {
         if (app > DEFAULT_SYNTH)
            SetADC (0);          // Turn off ADC to avoid pop in output
         if (appAddr == NULL)    // Must be a ROM app
            ret = DSPLoadApp ((WORD) app, 0);
         else                    // Load disk-based DSP app
            {
            if ((ret = DSPLoadDiskApp (appAddr)) == 0)
               AriaCap.DSPapps |= (1 << app);
            }
         if (app > DEFAULT_SYNTH)
            SetADC (1);          // Turn ADC back on

         if (minOps)
            {
            if (ret == 0)
               maxLoaded = TRUE;
            else
               SetPlaybackMode (MAXOPERATORS);
            }

         switch (app)
            {
            case ARIA_SYNTH:     // Aria application
               appLoaded[ARIA_SYNTH]    = (ret == 0);
               appLoaded[DEFAULT_SYNTH] = (ret != 0);
               if (ret == 0)
                  {
                  appLoaded[2] = TRUE;
                  appLoaded[3] = TRUE;
                  wCurrentUsage[ARIA_SYNTH] = wMaxUsage[ARIA_SYNTH];
                  wCurrentUsage[DEFAULT_SYNTH] = 0;
                  wCurrentUsage[2] = wMaxUsage[2];
                  wCurrentUsage[3] = wMaxUsage[3];
                  for (i = 4; i < MAXAPPS; ++i)
                     {
                     appLoaded[i] = FALSE;
                     wCurrentUsage[i] = 0;
                     }
                  putMem16 (WAVEBUFADDR, BUFADDRFIX);
                  }
               break;

            case DEFAULT_SYNTH: // Sound Blaster application
               appLoaded[ARIA_SYNTH] = (ret != 0);
               appLoaded[DEFAULT_SYNTH] = (ret == 0);
               if (ret == 0)
                  {
                  wCurrentUsage[0] = 0;
                  wCurrentUsage[1] = wMaxUsage[1];
                  for (i = 2; i < MAXAPPS; ++i)
                     {
                     appLoaded[i] = FALSE;
                     wCurrentUsage[i] = 0;
                     }
                  }
               break;

            default:
               if (ret == 0)
                  {
                  appLoaded[app] = TRUE;
                  wCurrentUsage[app] = wMaxUsage[app];
                  if (app == SPEECH_ID)
                     SetListenerMode (0);
                  else if (app == REVERB_ID)
                     SetReverbMode (REVERBMODEDEF);
                  else if (app == QSOUND_ID)
                     SetQSoundMode (0);
                  }
               else
                  appLoaded[app] = FALSE;
               break;
            }

         if (ret == 0 && appAddr)
            hApp[app] = appAddr;
         }
      else
         {
         if (minOps)
            SetPlaybackMode (MAXOPERATORS);
         ret = -4;
         }
      }
   else        // Unload a DSP application
      {
      ret = 0;
      if (appLoaded[app])
         {
         if ((ret = DSPUnloadApp ((WORD) app)) != 0)
            return ret;

         appLoaded[app] = FALSE;
         if (app == ARIA_SYNTH)  // Aria application
            {
            appLoaded[DEFAULT_SYNTH] = TRUE;
            wCurrentUsage[ARIA_SYNTH] = 0;
            wCurrentUsage[DEFAULT_SYNTH] = wMaxUsage[DEFAULT_SYNTH];
            for (i = 2; i < MAXAPPS; ++i)
               {
               appLoaded[i] = FALSE;
               wCurrentUsage[i] = 0;
               }
            }
         else
            wCurrentUsage[app] = 0;

         if (app == QSOUND_ID)
            {
            qsoundMode = 0;
            SetPanControls ();
            }
         else if (app == REVERB_ID)
            reverbMode = 0;

         if (hApp[app])
            {
            hApp[app] = NULL;
            AriaCap.DSPapps &= ~((WORD) 1 << app);
            }

         if (minOps)
            {
            SetPlaybackMode (MAXOPERATORS);
            maxLoaded = FALSE;
            }
         }
      }

   DSPusage = CalculateDSPUsage ();
   return ret;
   }


//***************************************************************************
//  This function downloads a disk-based DSP application to the Aria DSP
//***************************************************************************

static short loadobjfile (LPBYTE inptr)
   {
   register WORD i;
   register short ch;
   WORD data=0;
   WORD loadaddr=0;

   if ((ch = *inptr++) != STFL)
      // Set the file pointer to the begining of meaningful data
      for (ch = ' '; ((ch != STFL) & (ch != EOF)); ch = *inptr++)
         ;

   if (ch == EOF)
      return 4;

   // Set the file pointer to the load address
   for (ch = *inptr++; ((ch != ADDRFL) & (ch != EOF)); ch = *inptr++)
      ;

   if (ch == EOF)
      return 4;

   while (TRUE)
      {
      switch (ch)
         {
         case ADDRFL:
            loadaddr = 0;
            for (i = 0; i < 4; ++i)
               {
	            loadaddr <<= 4;
	            loadaddr += (ch = *inptr++) - '0';
	            if (ch > ADDRFL)
                  loadaddr -= 7;
               }
            OUTPW (DMAADDRESS, loadaddr);       // Set address
            break;

         case DATAFL:
            data = 0;
            for (i = 0; i < 4; ++i)
               {
               data <<= 4;
	            data += (ch = *inptr++) - '0';
	            if (ch > ADDRFL)
                  data -= 7;
               }
            OUTPW (DMADATA, data);              // Write data
            if (!(++loadaddr & 0x01FF))         // Has address exceeded 
               OUTPW (DMAADDRESS, loadaddr);    //  9 bit limit?
            break;

         default:
         case EOL:
         case EOL1:
            // Set the file pointer to the end of the current line
            for (ch = *inptr++; (ch != ENDLN) & (ch != EOF); ch = *inptr++)
  	            ;

         case EOF:
	         if (ch == EOF)
               return 3;
            break;

         case ENDFL:
            return 0;
         }   // End of switch statement

      ch = *inptr++;

      }  // End of while statement
   }


//***************************************************************************
//  This function tallies the current DSP utilization
//***************************************************************************

static WORD CalculateDSPUsage (VOID)
   {
   register WORD i, sum;

   sum = 0;
   for (i = 0; i < MAXAPPS; ++i)
      sum += wCurrentUsage[i];

   return sum;
   }


//***************************************************************************
//  This function waits for the system to initialize a new DSP application
//***************************************************************************

static short SystemWait (VOID)
   {
   register WORD j=800, i;

   // Wait for system to initialize

   while (j-- > 0)
      {
      for (i = 0; i < 200; ++i)
         INPW (DSPSTATUS);          // Dummy read for time delay

      if (getMem16 (DSP_INIT) == 1) // Init flag was set
         return 0;                  // Task is initialized
      }
   return -1;  // Timed out
   }

