{****************************************************************************}
{                                                                            }
{ MODULE:        ModUnit                                                     }
{                                                                            }
{ DESCRIPTION:   Gives the necessary support for handling the different      }
{                data types and different file formats of the Sound/Noise/   }
{                Pro-Tracker music files. Also, it implements the routines   }
{                for loading them into main memory and (future) saving them  }
{                to disk.                                                    }
{                                                                            }
{ AUTHOR:        Juan Carlos Arvalo Baeza                                   }
{                                                                            }
{ MODIFICATIONS: Nobody (yet).                                               }
{                                                                            }
{ HISTORY:       xx-May-1992 First implementations (lost in the memory of    }
{                            time O:-).                                      }
{                xx-Jun-1992 Lots of improvements (ditto O;-).               }
{                11-Jul-1992 Started first documented version.               }
{                21-Oct-1992 Rechecking.                                     }
{                                                                            }
{ (C) 1992 VangeliSTeam                                                      }
{____________________________________________________________________________}

UNIT ModUnit;

INTERFACE

USES Dos;




{----------------------------------------------------------------------------}
{ Definitions for handling the format of individual notes.                   }
{ Notes are composed of four fields:                                         }
{                                                                            }
{   Period:     A number in the range 0..1023 which states the period of     }
{               the note in units of 1/3584000 per sample. (this is a        }
{               somewhat empyric number. If anyone knows the exact Amiga     }
{               number, please, tell us). A zero means to keep using the     }
{               same period used before.                                     }
{   Instrument: A number in range 0..31 meaning the number of the instrument }
{               which will be used for the note. A zero means use the same.  }
{   Command:    A number (no real range) of the way the note should be       }
{               played (i.e. Vibrato) a change in the playing sequence (i.e. }
{               pattern break) or a change in the general parameters of the  }
{               module player (i.e. set tempo). All the possible values are  }
{               defined in the TModCommand enumerated type below.            }
{   Parameter:  A parameter for the command. Its meaning differs from one    }
{               command to another. Sometimes each nibble is considered as a }
{               different parameter.                                         }
{____________________________________________________________________________}

TYPE
  TModCommand = (mcArpeggio,   { 0 xy } { Rotate through three notes rapidly. }
                 mcTPortUp,    { 1 xx } { Tone Portamento Up:   Gradual change of tone towards high frequencies. }
                 mcTPortDown,  { 2 xx } { Tone Portamento Down: Gradual change of tone towards low  frequencies. }
                 mcNPortamento,{ 3 xy } { Note Portamento:      Gradual change of tone towards a given note.     }
                 mcVibrato,    { 4 xy } { Vibrato: Frequency changes around the note. }
                 mcT_VSlide,   { 5 xy } { Tone Port. Up + Volume slide: Parameter means vol. slide. }
                 mcVib_VSlide, { 6 xy } { Vibrato       + Volume slide: Parameter means vol. slide. }
                 mcTremolo,    { 7 xy } { Tremolo: I don't know for sure. Fast volume variations, I think. }
                 mcNPI1,       { 8 xx } { Do Nothing (as far as I know). }
                 mcSampleOffs, { 9 xx } { Start the sample from the middle. }
                 mcVolSlide,   { A xy } { Volume slide: Gradual change in volume. }
                 mcJumpPattern,{ B xx } { End pattern and continue from a different pattern sequence position. }
                 mcSetVolume,  { C xx } { Set the volume of the sound. }
                 mcEndPattern, { D xx } { Continue at the start of the next pattern. }
                 mcExtended,   { E xy } { Extended set of commands (ProTracker). }
                 mcSetTempo,   { F xx } { Set the tempo of the music, in 1/50ths of a second. }

                 mcSetFilter,  { E 0x } { Set the output filter to the on or off value. }
                 mcFinePortaUp,{ E 1x } { Like TPortUp,   but slower. }
                 mcFinePortaDn,{ E 2x } { Like TPortDown, but slower. }
                 mcGlissCtrl,  { E 3x } { ??? }
                 mcVibCtrl,    { E 4x } { Set the vibrato waveform. }
                 mcFineTune,   { E 5x } { Fine tune the frequency of the sound. }
                 mcJumpLoop,   { E 6x } { Make a loop inside a pattern. }
                 mcTremCtrl,   { E 7x } { Set the tremolo waveform (I think). }
                 mcNPI2,       { E 8x } { Do Nothing (as far as I know). }
                 mcRetrigNote, { E 9x } { ??? }
                 mcVolFineUp,  { E Ax } { Like VolSlide, but slower and towards high frequencies. }
                 mcVolFineDown,{ E Bx } { Like VolSlide, but slower and towards low  frequencies. }
                 mcNoteCut,    { E Cx } { ??? }
                 mcNoteDelay,  { E Dx } { Wait a little before starting note. }
                 mcPattDelay,  { E Ex } { ??? }
                 mcFunkIt,     { E Fx } { No idea, but sounds funny. }

                 mcNone        { 0 00 } { Just play the note, without any special option. }
  );

CONST
  CommandSet1   = [mcArpeggio .. mcVibrato,    { A basic command set found in most modules. }
                   mcVolSlide .. mcSetFilter,
                   mcNone];
  CommandSet2   = [mcArpeggio .. mcSetFilter,  { A not so basic command set found in some modules. }
                   mcNone];
  CommandSetAll = [mcArpeggio .. mcNone];      { The full ProTracker command set. }


TYPE
  PModNote = ^TModNote;
  TModNote = RECORD
    sample : BYTE;
    freq   : WORD;
    comm   : TModCommand;
    param  : BYTE;
  END;




{----------------------------------------------------------------------------}
{ Definitions for handling the format of the patterns.                       }
{ Patterns are arrays of 64 lines of four notes each (one note per channel). }
{ A music module can have up to 64 individual patterns.                      }
{____________________________________________________________________________}

CONST
  MaxPatterns     = 64;
  MaxPatternLines = 64;
  MaxChannels     = 4;

TYPE
  PPackedNote = ^TPackedNote;
  TPackedNote = RECORD
    SampleFreq : WORD;
    Command    : TModCommand;
    Parameter  : BYTE;
  END;

  PPatternLine = ^TPatternLine;
  TPatternLine = ARRAY[1..MaxChannels] OF TPackedNote;

  PPattern = ^TPattern;
  TPattern = ARRAY[0..MaxPatternLines-1] OF TPatternLine;

VAR
  Patterns      : ARRAY[0..MaxPatterns-1] OF PPattern;
  NumPatterns   : BYTE;
  PatternTempos : ARRAY[0..MaxPatterns-1] OF BYTE;


PROCEDURE UnpackNote(n: TPackedNote; VAR r: TModNote);




{----------------------------------------------------------------------------}
{ Definitions for handling sequences of patterns.                            }
{____________________________________________________________________________}

CONST
  MaxSequence = 128;

TYPE
  TPatternSequence = ARRAY[0..MaxSequence-1] OF BYTE;

VAR
  PatternSequence  : TPatternSequence;
  SequenceLength,
  SequenceRepStart : BYTE;




{----------------------------------------------------------------------------}
{ Definitions for handling the instruments used in the module.               }
{ Instruments are fragments of sampled sound (long arrays of bytes which     }
{ describe the wave of the sound of the instrument). The samples used in     }
{ music modules have a default volume and also, they can have a loop (for    }
{ sustained instruments) and a fine tuning constant (not yet implemented).   }
{____________________________________________________________________________}

CONST
  MaxSample      = 65520;
  MaxInstruments = 31;

TYPE
  TSample = ARRAY[0..MaxSample-1] OF SHORTINT;

  TIProperties = WORD; { Properties of the instrument. }

CONST                  { Properties constants. }
  ipMonoFreq = $0001;  { Set if the instrument is played always at the same freq (not implemented). }
  ipLong     = $0002;  { Set if the instrument's sample is longer than 65520 bytes.                 }

TYPE
  TDefInstrument = RECORD
    name  : STRING[22];   { String giving the name of the instrument.                           }
    len,                  { Length of the instrument's sampled image.                           }
    reps,                 { Starting offset of the repeated portion.                            }
    repl  : LONGINT;      { Size of the repeated portion.                                       }
    vol   : BYTE;         { Default volume of the instrument (0..64)                            }
    ftune : BYTE;         { Fine tuning value for the instrument (not yet implemented).         }
    data  : ^TSample;     { Pointer to the first  65520 bytes of the sample.                    }
    xtra  : ^TSample;     { Pointer to the second 65520 bytes of the sample (if there is such). }
    prop  : TIProperties; { Bit mapped properties value.                                        }
  END;

VAR
  Instruments : ARRAY[1..MaxInstruments] OF TDefInstrument;




{----------------------------------------------------------------------------}
{ Definitions for accelerating the use of note periods.                      }
{____________________________________________________________________________}

CONST
  NumberOctaves = 7;
  NumberNotes   = 12;
  NumberPeriods = NumberOctaves * NumberNotes;

TYPE
  TPeriodSet = ARRAY[0..NumberOctaves-1] OF         { Octave }
               ARRAY[0..NumberNotes  -1] OF WORD;   { Note   }

  TPeriodArray = ARRAY[0..NumberPeriods - 1] OF WORD;

CONST
  { The different note values. }

  PeriodSet : TPeriodSet = (
    {  C     C#    D     D#    E     F     F#    G     G#    A     A#    B  }
    ($06B0,$0650,$05F5,$05A0,$054F,$0503,$04BB,$0477,$0436,$03FA,$03C1,$038B),
    ($0358,$0328,$02FB,$02D0,$02A7,$0281,$025D,$023B,$021B,$01FD,$01E0,$01C5),
    ($01AC,$0194,$017D,$0168,$0154,$0141,$012F,$011E,$010E,$00FE,$00F0,$00E3),
    ($00D6,$00CA,$00BF,$00B4,$00AA,$00A0,$0097,$008F,$0087,$007F,$0078,$0071),
    ($006B,$0065,$005F,$005A,$0055,$0050,$004C,$0047,$0043,$0040,$003C,$0039),
    ($0035,$0032,$0030,$002D,$002A,$0028,$0026,$0024,$0022,$0020,$001E,$001C),
    ($001B,$0019,$0018,$0016,$0015,$0014,$0013,$0012,$0011,$0010,$000F,$000E)
  );

  { The different inter-note values. }

  PeriodDiff : TPeriodSet = (
    ($0680,$0622,$05CA,$0577,$0529,$04DF,$0499,$0456,$0418,$03DD,$03A6,$0371),
    ($0340,$0311,$02E5,$02BB,$0294,$026F,$024C,$022B,$020C,$01EE,$01D2,$01B8),
    ($01A0,$0188,$0172,$015E,$014A,$0138,$0126,$0116,$0106,$00F7,$00E9,$00DC),
    ($00D0,$00C4,$00B9,$00AF,$00A5,$009B,$0093,$008B,$0083,$007B,$0074,$006E),
    ($0068,$0062,$005C,$0057,$0052,$004E,$0049,$0045,$0041,$003E,$003A,$0037),
    ($0033,$0031,$002E,$002B,$0029,$0027,$0025,$0023,$0021,$001F,$001D,$001B),
    ($001A,$0018,$0017,$0015,$0014,$0013,$0012,$0011,$0010,$000F,$000E,$000E)
  );

VAR
  PeriodArray : TPeriodArray ABSOLUTE PeriodSet;

TYPE
  TNoteString    = STRING[3];

  TNoteSet       = ARRAY[0..1023] OF WORD;
  TNoteStringSet = ARRAY[0..NumberPeriods] OF TNoteString;

VAR
  NoteIdx : TNoteSet;       { For each period, specifies it's closest note, in two ways:     }
                            {   Hi byte: octave in the hi nibble and note in the low nibble. }
                            {  Low byte: sequential note for indexing.                       }
  NoteStr : TNoteStringSet; { The strings for each note (e.g. 'A#2'). }


PROCEDURE NoteFreq(f: WORD; VAR s: TNoteString);




{----------------------------------------------------------------------------}
{ General definitions for the music module,                                  }
{____________________________________________________________________________}

VAR
  SongName     : STRING[20]; { The name of the loaded song.                  }
  SongFileName : STRING[12]; { The filename from where the song was loaded.  }
  SongFileDir  : DirStr;     { The directory where the filename was located. }

  IOError   : WORD;
  ErrorCode : WORD; { Mod operation result code. }

CONST                            { Result codes. }
  mecOK                     = 0; { Everything was Ok.                                     }
  mecFileOpenError          = 1; { Could not open the .MOD file.                          }
  mecOutOfMemory            = 2; { There is not enough memory left. :-( Shouldn't happen. }
  mecFileReadError          = 3; { Could not read the file.                               }
  mecFileDamaged            = 4; { Syntax checking error on module file.                  }
  mecFileTooShort           = 5; { End of file premature (lot's of modules have this).    }
  mecFileFormatNotSupported = 6; { JMPlayer or ScreamTracker, for example.                }

VAR
  ModFileFormat : WORD; { Identifier of the file format of the loaded module. }

CONST               { Known file formats. }
  mffUnknown   = 0; { Unknown format O:-)                 }
  mffMod31M_K_ = 1; { "M.K." magic field, 31 instruments. }
  mffMod31FLT4 = 2; { "FLT4" magic field, 31 instruments. }
  mffMod15     = 3; { Prehistoric 15 instruments module.  }
  mffJMPlayer  = 4; { JMPlayer module.                    }
  mffOktalizer = 5; { 8 voices Oktalizer MOD.             }

  ModLoaded : BOOLEAN = FALSE; { TRUE when there is a module in memory. }




PROCEDURE InitModUnit;
FUNCTION  GetErrorString(i: WORD)    : STRING;
FUNCTION  LoadMod       (s: PathStr) : BOOLEAN;
PROCEDURE FreeMod;




IMPLEMENTATION

USES StrConst, AsciiZ,
     DOSMem, Filters;




{----------------------------------------------------------------------------}
{ Internal definitions. Format of the files.                                 }
{____________________________________________________________________________}

TYPE
  { Instrument in a MOD file. 30 bytes. }

  TModFileInstrument = RECORD
    Name       : ARRAY [1..22] OF CHAR; { AsciiZ string, name of the instrument. }
    Len        : WORD;                  { Length of the sample DIV 2.            }
    FineTune,                           { Fine tuning value.                     }
    Vol        : BYTE;                  { Default volume.                        }
    LoopStart,                          { Offset of the loop DIV 2.              }
    LoopLen    : WORD;                  { Length of the loop DIV 2.              }
  END;

  { Note in the file. 4 bytes. }

  PModFileNote = ^TModFileNote;
  TModFileNote = RECORD
    CASE INTEGER OF
      1: (l              : LONGINT);
      2: (w1, w2         : WORD);
      3: (b1, b2, b3, b4 : BYTE);
  END;

  PModFilePattern = ^TModFilePattern;
  TModFilePattern = ARRAY [0..63] OF ARRAY [1..4] OF TModFileNote;

  { 15 samples module header format. 600 bytes. }

  PModFile15 = ^TModFile15;
  TModFile15 = RECORD
    Name        : ARRAY [1..20] OF CHAR;               { AsciiZ song name.                   }
    Samples     : ARRAY [1..15] OF TModFileInstrument; { Instruments.                        }
    SongLen     : BYTE;                                { Length of the sequency of the song. }
    SongRep     : BYTE;                                { Song loop start position.           }
    PatternList : ARRAY [0..127] OF BYTE;              { Pattern sequencies.                 }
  END;

  { 31 samples module header format. 1084 bytes. }

  PModFile31 = ^TModFile31;
  TModFile31 = RECORD
    Name        : ARRAY [1..20] OF CHAR;               { AsciiZ song name.                   }
    Samples     : ARRAY [1..31] OF TModFileInstrument; { Instruments.                        }
    SongLen     : BYTE;                                { Length of the sequency of the song. }
    SongRep     : BYTE;                                { Song loop start position.           }
    PatternList : ARRAY [0..127] OF BYTE;              { Pattern sequencies.                 }
    Magic       : LONGINT;                             { Magic number ("M.K." or "FLT4")     }
  END;

  { JMPlayer module header format. }

  TModJMIdString = ARRAY[1..6] OF CHAR; { JMPlayer Id string (at the start of the file). }

  TModFileJMPlayer = RECORD
    IdString : TModJMIdString;
  END;

  { Oktalizer module header format. }

  TModOktIdString = ARRAY[1..8] OF CHAR; { Oktalizer Id string (at the start of the file). }

  TModFileOktalizer = RECORD
    IdString : TModOktIdString;
  END;

  { Grouping of the different header formats. }

  TModFileHeader = RECORD
    CASE INTEGER OF
      1: (Mod15  : TModFile15);
      2: (Mod31  : TModFile31);
      3: (ModJM  : TModFileJMPlayer);
      4: (ModOkt : TModFileOktalizer);
  END;

CONST
  Mod31MagicM_K_  = $2E4B2E4D; { Magic numbers. }
  Mod31MagicFLT4  = $34544C46;

  ModJMIdString  : TModJMIdString  = ('J', 'M', 'P', 'L', 'A', 'Y');
  ModOktIdString : TModOktIdString = ('O', 'K', 'T', 'A', 'S', 'O', 'N', 'G');




{----------------------------------------------------------------------------}
{ Miscellaneous routines.                                                    }
{____________________________________________________________________________}

PROCEDURE ZeroOutData;
  BEGIN
    FillChar(Patterns,        SIZEOF(Patterns),        0);
    FillChar(PatternTempos,   SIZEOF(PatternTempos),   0);
    FillChar(PatternSequence, SIZEOF(PatternSequence), 0);
    FillChar(Instruments,     SIZEOF(Instruments),     0);
    NumPatterns      := 0;
    SequenceLength   := 0;
    SequenceRepStart := 0;
    SongName         := '';
    SongFileName     := '';
    SongFileDir      := '';
    IOError          := 0;
    ErrorCode        := mecOK;
    ModLoaded        := FALSE;
    ModFileFormat    := mffUnkNown;
  END;


FUNCTION GetErrorString(i: WORD) : STRING;
  BEGIN
    CASE i OF
      mecFileOpenError:          GetErrorString := GetString(StrFileOpenError);
      mecOutOfMemory:            GetErrorString := GetString(StrOutOfMemory);
      mecFileReadError:          GetErrorString := GetString(StrFileReadError);
      mecFileDamaged:            GetErrorString := GetString(StrFileDamaged);
      mecFileTooShort:           GetErrorString := GetString(StrFileTooShort);
      mecFileFormatNotSupported: GetErrorString := GetString(StrFileFormatNotSupported);
      ELSE                       GetErrorString := '';
    END;
  END;




{----------------------------------------------------------------------------}
{ Module loading routines.                                                   }
{____________________________________________________________________________}

FUNCTION LoadMod31(VAR f: FILE; VAR Header: TModFileHeader) : BOOLEAN;
  VAR
    j, k,
    i, w     : WORD;
    IsMod31  : BOOLEAN;
    Tempo    : BYTE;
    ls, ll,
    ljunk, r : LONGINT;
    p        : POINTER;
    pp       : TModFileNote;
    cc       : TPackedNote;
  BEGIN
    LoadMod31 := FALSE;

    ErrorCode := mecFileDamaged;

    FOR i := 0 TO 127 DO
      IF Header.Mod31.PatternList[i] > 63 THEN EXIT;

    FOR i := 1 TO 22 DO
      IF (Header.Mod31.Name[i] < ' ') AND (Header.Mod15.Name[i] <> #0) THEN EXIT;

    ErrorCode := mecOK;

    SongName := StrASCIIZ(Header.Mod31.Name, 20);

    NumPatterns := 0;
    FOR i := 0 TO 127 DO
      IF NumPatterns < Header.Mod31.PatternList[i] THEN NumPatterns := Header.Mod31.PatternList[i];
    INC(NumPatterns);

    Move(Header.Mod31.PatternList, PatternSequence, 128);

    SequenceLength   := Header.Mod31.SongLen;
    SequenceRepStart := Header.Mod31.SongRep;

    FOR i := 0 TO NumPatterns-1 DO BEGIN

      Patterns[i] := DOSAlloc(SIZEOF(TPattern));

      IF Patterns[i] = NIL THEN BEGIN
        ErrorCode := mecOutOfMemory;
        EXIT;
      END;

      BlockRead(f, Patterns[i]^, SIZEOF(TPattern), w);

      IOError := IOResult;
      IF IOError <> 0 THEN BEGIN
        ErrorCode := mecFileReadError;
        EXIT;
      END;

      IF w <> SIZEOF(TPattern) THEN BEGIN
        ErrorCode := mecFileTooShort;
        EXIT;
      END;
    END;

    FOR i := 1 TO 31 DO BEGIN
      r  := LONGINT(SWAP(Header.Mod31.Samples[i].Len)      ) SHL 1;
      ls := LONGINT(SWAP(Header.Mod31.Samples[i].LoopStart)) SHL 1;
      ll := LONGINT(SWAP(Header.Mod31.Samples[i].LoopLen)  ) SHL 1;

      IF ll      > r THEN ll := r;
      IF ls + ll > r THEN ls := r - ll;

      Header.Mod31.Samples[i].LoopStart := SWAP(ls DIV 2);
      Header.Mod31.Samples[i].LoopLen   := SWAP(ll DIV 2);

      IF Header.Mod31.Samples[i].Vol > $40 THEN
        Header.Mod31.Samples[i].Vol := $40;

      Instruments[i].len  := r;
      Instruments[i].reps := ls;
      Instruments[i].repl := ll;
      Instruments[i].vol  := Header.Mod31.Samples[i].Vol;
      Instruments[i].name := StrASCIIZ(Header.Mod31.Samples[i].Name, 22);
      IF r > 0 THEN BEGIN
        IF r <= MaxSample THEN BEGIN
          p := DOSAlloc(r);

          IF p = NIL THEN BEGIN
            ErrorCode := mecOutOfMemory;
            EXIT;
          END;

          Instruments[i].data := p;
          BlockRead(f, p^, r, w);

          IOError := IOResult;
          IF IOError <> 0 THEN BEGIN
            ErrorCode := mecFileReadError;
            EXIT;
          END;


          IF w <> r THEN BEGIN
            ErrorCode := mecFileTooShort;
{            EXIT;}
          END;

          Instruments[i].xtra := NIL;
        END ELSE BEGIN
          p := DOSAlloc(MaxSample);

          IF p = NIL THEN BEGIN
            ErrorCode := mecOutOfMemory;
            EXIT;
          END;

          Instruments[i].data := p;
          BlockRead(f, p^, MaxSample, w);

          IOError := IOResult;
          IF IOError <> 0 THEN BEGIN
            ErrorCode := mecFileReadError;
            EXIT;
          END;

          IF w <> r THEN BEGIN
            ErrorCode := mecFileTooShort;
{            EXIT;}
          END;

          p := DOSAlloc(r-MaxSample);

          IF p = NIL THEN BEGIN
            ErrorCode := mecOutOfMemory;
            EXIT;
          END;

          Instruments[i].xtra := p;
          BlockRead(f, p^, r-MaxSample, w);

          IOError := IOResult;
          IF IOError <> 0 THEN BEGIN
            ErrorCode := mecFileReadError;
            EXIT;
          END;

          IF w <> r THEN BEGIN
            ErrorCode := mecFileTooShort;
{            EXIT;}
          END;

        END;
      END;
    END;

    Tempo := 6;

    FOR i := 0 TO NumPatterns-1 DO BEGIN

      PatternTempos[i] := Tempo;

      FOR j := 0 TO MaxPatternLines-1 DO
        FOR k := 1 TO MaxChannels DO BEGIN

          pp := TModFileNote(Patterns[i]^[j][k]);
          cc := Patterns[i]^[j][k];

          WITH cc DO BEGIN
            Command     := TModCommand(pp.b3 AND $F);
            IF Command = mcExtended THEN BEGIN
              Parameter := pp.b4 AND $F;
              Command   := TModCommand($10 + (pp.b4 SHR 4));
            END ELSE IF (Command = mcArpeggio) AND (pp.b4 = 0) THEN BEGIN
              Parameter := 0;
              Command   := mcNone;
            END ELSE
              Parameter := pp.b4;
            SampleFreq  := (pp.b2 + (WORD(pp.b1 AND $7) SHL 8)) +
                           (WORD((pp.b3 SHR 4) + (pp.b1 AND 16)) SHL 11);

            IF (Command = mcSetVolume)   AND (Parameter > $40) THEN Parameter := $40;
            IF (Command = mcJumpPattern) AND (Parameter >  63) THEN Parameter := Parameter AND 63;
            IF (Command = mcEndPattern)  AND (Parameter >  63) THEN Parameter := Parameter AND 63;
            IF (Command = mcSetTempo)    AND (Parameter > $1F) THEN Parameter := $1F;
            IF (Command = mcSetTempo)                          THEN Tempo     := Parameter;

          END;

          Patterns[i]^[j][k] := cc;

        END;
    END;

    LoadMod31 := TRUE;

  END;


FUNCTION LoadMod15(VAR f: FILE; VAR Header: TModFileHeader) : BOOLEAN;
  VAR
    i : WORD;
  BEGIN
    LoadMod15 := FALSE;

    Move(Header.Mod15.SongLen, Header.Mod31.SongLen, 130);
    FOR i := 16 TO 31 DO BEGIN
      Header.Mod31.Samples[i].Len     := 0;
      Header.Mod31.Samples[i].Name[1] := #0;
    END;

    Seek(f, SIZEOF(Header.Mod15));

    LoadMod15 := LoadMod31(f, Header);

  END;


FUNCTION LoadModJM(VAR f: FILE; VAR Header: TModFileHeader) : BOOLEAN;
  BEGIN
    ErrorCode := mecFileFormatNotSupported;
    LoadModJM := FALSE;
  END;


FUNCTION LoadModOkt(VAR f: FILE; VAR Header: TModFileHeader) : BOOLEAN;
  BEGIN
    ErrorCode  := mecFileFormatNotSupported;
    LoadModOkt := FALSE;
  END;


FUNCTION LoadMod(s: PathStr) : BOOLEAN;
  VAR
    f      : FILE;
    Header : TModFileHeader;
    w      : WORD;
    Dir    : DirStr;
    Nam    : NameStr;
    Ext    : ExtStr;
  BEGIN
    LoadMod   := FALSE;

    ErrorCode := mecOK;

    FreeMod;

    ZeroOutData;

    s := FExpand(s); FSplit(s, Dir, Nam, Ext);
    IF Ext = '' THEN Ext := '.MOD';
    SongFileDir  := Dir;
    SongFileName := Nam + Ext;
    s := SongFileDir + SongFileName;

    Assign(f, s);
    Reset(f, 1);

    IOError := IOResult;
    IF IOError <> 0 THEN BEGIN
      ErrorCode := mecFileOpenError;
      EXIT;
    END;

    BlockRead(f, Header, SIZEOF(Header), w);

    IOError := IOResult;
    IF IOError <> 0 THEN BEGIN
      ErrorCode := mecFileReadError;
      Close(f);
      EXIT;
    END;

    IF w <> SIZEOF(Header) THEN BEGIN
      ErrorCode := mecFileTooShort;
      Close(f);
      EXIT;
    END;

    IF Header.ModOkt.IdString = ModOktIdString THEN BEGIN
      ModFileFormat := mffOktalizer;

      ModLoaded := LoadModOkt(f, Header);

    END ELSE IF Header.ModJM.IdString = ModJMIdString THEN BEGIN
      ModFileFormat := mffJMPlayer;

      ModLoaded := LoadModJM(f, Header);

    END ELSE IF Header.Mod31.Magic = Mod31MagicM_K_ THEN BEGIN
      ModFileFormat := mffMod31M_K_;

      ModLoaded := LoadMod31(f, Header);

    END ELSE IF Header.Mod31.Magic = Mod31MagicFLT4 THEN BEGIN
      ModFileFormat := mffMod31FLT4;

      ModLoaded := LoadMod31(f, Header);

    END ELSE BEGIN
      ModFileFormat := mffMod15;

      ModLoaded := LoadMod15(f, Header);

    END;

    Close(f);

    LoadMod := ModLoaded;
  END;


PROCEDURE FreeMod;
  VAR
    i : WORD;
  BEGIN
    DOSWipeMem;
    ASM CLI END;
    IF ModLoaded THEN
      FOR i := 1 TO 31 DO
        Instruments[i].len := 0;
    ModLoaded := FALSE;
    ASM STI END;
  END;




{----------------------------------------------------------------------------}
{ Routines for data format conversions.                                      }
{____________________________________________________________________________}

PROCEDURE UnpackNote(n: TPackedNote; VAR r: TModNote);
  VAR
    src : LONGINT ABSOLUTE n;
    dst : RECORD
      b : BYTE;
      l : LONGINT;
    END           ABSOLUTE r;
  BEGIN
    dst.l := src;

    r.sample := r.freq SHR 11;
    r.freq   := r.freq AND $07FF;
  END;


PROCEDURE NoteFreq(f: WORD; VAR s: TNoteString);
  BEGIN
    IF f > 1023 THEN
      f := 1023;

    s := NoteStr[NoteIdx[f] AND $FF];
  END;




{----------------------------------------------------------------------------}
{ Initialization routine.                                                    }
{____________________________________________________________________________}

PROCEDURE InitModUnit;
  CONST
     NoteLet : STRING[12] = 'CCDDEFFGGAAB';
     NoteSus : STRING[12] = ' # #  # # # ';
  VAR
     l    : LONGINT;
     f,
     o, i : WORD;
     s    : STRING[3];
  LABEL
     Octava, NextFreq;
  BEGIN

     ModLoaded := FALSE;

     ZeroOutData;

     FOR f := 0 TO 1023 DO BEGIN

       FOR o := 0 TO 6 DO
         IF f > PeriodDiff[o][11] THEN GOTO Octava;
       i := 0; o := 0;
       GOTO NextFreq;

Octava:
       FOR i := 0 TO 11 DO
         IF f > PeriodDiff[o][i]  THEN GOTO NextFreq;
       i := 0; o := 0;

NextFreq:
       NoteIdx[f] := (o*16+i)*256 + (o*12+i)

     END;

     FOR i := 0 TO 12*7-1 DO BEGIN

       s[0] := CHR(3);
       o    := i DIV 12;
       s[3] := CHR(o + ORD('0'));
       o    := i MOD 12 + 1;
       s[1] := NoteLet[o];
       s[2] := NoteSus[o];

       NoteStr[i] := s;

     END;

     NoteStr[12*7] := '---';
  END;




END.
