{****************************************************************************}
{                                                                            }
{ MODULE:         PlayMod                                                    }
{                                                                            }
{ DESCRIPTION:    This UNIT allows to play a music module (*.MOD) in any     }
{                 device supported in the SoundDevices sound system.         }
{                                                                            }
{                 Entrys:   PlayMod To begin playing the MOD.                }
{                           StopMod To stop playing the MOD.                 }
{                                                                            }
{ AUTHOR:         Juan Carlos Arvalo                                        }
{                 Luis Crespo (parts extracted from the JAMP 1.5 MOD Player) }
{                                                                            }
{ MODIFICATIONS:  Nobody (yet... ;-)                                         }
{                                                                            }
{ HISTORY:        22-Jun-1992 Begins to use the SoundDevices sound system.   }
{                             Internal cleaning, which was quite needed.     }
{                             UnCanal routine made even faster.              }
{                 11-Nov-1992 Redocumentation. There have been really many   }
{                             enhancements since June, but they weren't      }
{                             documented. Mainly more speed-ups.             }
{                                                                            }
{ (C) 1992 VangeliSTeam                                                      }
{____________________________________________________________________________}

UNIT PlayMod;

INTERFACE

USES ModUnit, SoundDevices, Filters;




TYPE
  TVolumes  = ARRAY[0..3] OF BYTE;      { Volume set (4 channels).         }
  TTickProc = PROCEDURE(note: BOOLEAN); { Procedure to execute every tick. }

                                                             
{ General definitions about the way of playing the music. }
{ Music player configuration.                             }

CONST
  LoopMod            : BOOLEAN       = FALSE;  { TRUE if music can be played forever.                  }
  ForceLoopMod       : BOOLEAN       = FALSE;  { TRUE if music must be played forever.                 }
  PermitFilterChange : BOOLEAN       = FALSE;  { TRUE if the partiture is allowed to change the filter.}
  CanFallBack        : BOOLEAN       = TRUE;   { TRUE if fall-back is allowed.                         }
  FilterOn           : TFilterMethod = fmNone; { Initial value of the ON  filter.                      }
  FilterOff          : TFilterMethod = fmNone; { Initial value of the OFF filter.                      }
  FilterIsOn         : BOOLEAN       = FALSE;  { Initial position of the filter (FALSE = OFF).         }
  TicksPerSecond     : WORD          = 50;     { Number of ticks per second, 50 = Europe, 60 = USA.    }

                                                             
{ Exported variables. }

CONST
  Playing          : BOOLEAN = FALSE;       { (Read only) TRUE if the music is sounding right now.     }
  ModTickProcValid : BOOLEAN = FALSE;       { TRUE if the module tick procedure has been initialised.  }

VAR
  MyLoopMod       : BOOLEAN;                { If TRUE, then the MOD will loop if it was so defined.    }
  NextNote,                                 { Next note in the pattern.                                }
  NextSeq         : WORD;                   { Next pattern index (for the next note).                  }
  ActualHz        : WORD;                   { Desired freq. of the sound.                              }
  NoteHz          : WORD;                   { Freq. to be used in the current tick.                    }
  UserVols        : TVolumes;               { Channel volumes.                                         }
  Permisos        : ARRAY[0..3] OF BOOLEAN; { Permissions for playing the 4 channels.                  }
  TempoCt         : BYTE;                   { Number of ticks in the current note.                     }
  TickCount       : WORD;                   { Ticks counter. Increments each tick.                     }
  ModTickProc     : TTickProc;              { Tick procedure. A procedure to be executed every tick.   }
  MyCanFallBack   : BOOLEAN;                { Actual permission to fall-back.                          }
  FilterVal       : TFilterMethod;          { Method of the filter to be used.                         }
                                                             

{ Definition of the local stack. }

CONST
  PlayModStackSize = 500;   { Size of the stack. }

VAR
  PlayModStack     : ARRAY[1..PlayModStackSize] OF BYTE;     


{ Definitions concerning a note. Buffer of the last N notes. }

TYPE
  PPlayingNote = ^TPlayingNote;
  TPlayingNote = RECORD
    EoMod       : BOOLEAN;                 { TRUE if it is the note following the last.                }
    Tempo       : BYTE;                    { Number of ticks the note will last.                       }
    NotePlaying : BYTE;                    { Index of the note inside the pattern.                     }
    SeqPlaying  : BYTE;                    { Sequence number of the pattern to which the note belongs. }
    Volume      : TVolumes;                { Volumes of the channels.                                  }
    Note        : ARRAY[0..3] OF TModNote; { Notes of the channels.                                    }
    NMuestras   : WORD;                    { Number of samples processed for each note.                }
    fill        : BYTE;                    { To make it a 32-byte record.                              }
  END;

CONST
  NoteBuffSize = 1;   { Number of note buffers. }

VAR
  NoteBuff      : ARRAY[0..NoteBuffSize-1] OF TPlayingNote;

CONST
  NoteTl        : WORD = 0;
  NoteHd        : WORD = 0;

  NoteSound     : PPlayingNote = NIL;
  NoteProcessed : PPlayingNote = NIL;




{----------------------------------------------------------------------------}
{ Definition of the buffers where the samples are placed.                    }
{____________________________________________________________________________}

CONST
  MaxSplPerTick = 700; { Maximum samples in the buffer. Means maximum samples per tick. }
  NumBuffers    = 3;   { Number of buffers.                                             }
VAR
  BuffIdx,         { Tail of the buffer. }
  BuffGive : WORD; { Head of the buffer. }

  Buffers  : ARRAY[1..NumBuffers] OF TSampleBuffer;
  BuffData : ARRAY[1..NumBuffers] OF ARRAY[1..MaxSplPerTick*4] OF INTEGER;



{----------------------------------------------------------------------------}
{ Exported procedures.                                                       }
{____________________________________________________________________________}

PROCEDURE PlayStart;
PROCEDURE PlayStop;

PROCEDURE ChangeSamplingRate(Hz: WORD);
PROCEDURE ProcessTick;
PROCEDURE FillWithSamples   (VAR Buff; Size: WORD);




IMPLEMENTATION

USES Dos,
     Debugging;




{----------------------------------------------------------------------------}
{ General definitions of the module player. They define its actual state.    }
{____________________________________________________________________________}

TYPE                       { Channel state definition. }
  PCanal = ^TCanal;
  TCanal = RECORD
    nota      : TModNote;  { Note being played in the channel. }
    vol       : BYTE;      { Actual volume.                    }
    freq      : WORD;      { Actual freq.                      }

    nfreq,                 { Note portamento increment.        }
    ffreq     : INTEGER;   { Note portamento destination.      }

    arpct     : BYTE;      { Arpeggio count.                   }
    arp1,                  { Arpeggio 2nd freq.                }
    arp2      : WORD;      { Arpeggio 3rd freq.                }

    vibwave,               { Vibrato wave form.                }
    vibpos,                { Vibrato position.                 }
    vibwidth,              { Vibrato width (period).           }
    vibdepth  : BYTE;      { Vibrato depth (amplitude).        }
    vibnreset : BOOLEAN;   { Vibrato ????.                     }

    tporta    : INTEGER;   { Tone portamento increment.        }
    vols      : SHORTINT;  { Volume increment.                 }
  END;

VAR
  DelaySamples    : BOOLEAN;     { TRUE means it couldn't fill the samples buffer.                }
  MuestrasPerTick : WORD;        { Number of samples that there are in a tick at the actual freq. }

  Canales : ARRAY[0..3] OF TCanal; { State of the channels. }




{----------------------------------------------------------------------------}
{ Raw channel definitions.                                                   }
{____________________________________________________________________________}

TYPE
  PModRawChan = ^TModRawChan;
  TModRawChan = RECORD
    Flags      : BYTE;         { Channel flags (see below).                 }

    SplPosFrac : BYTE;         { Position fraction.                         }
    SplPosInt  : WORD;         { Position offset.                           }
    SplPosSeg  : WORD;         { Position segment.                          }
                               
    SplOfs     : WORD;         { Actual sample part offset.                 }
    SplSeg     : WORD;         { Actual sample part segment.                }
    SplLimit   : WORD;         { Actual sample part size.                   }

    SplOfs1    : WORD;         { First  sample part offset.                 }
    SplSeg1    : WORD;         { First  sample part segment.                }
    SplLimit1  : WORD;         { First  sample part size.                   }

    SplOfs2    : WORD;         { Second sample part offset.                 }
    SplSeg2    : WORD;         { Second sample part segment.                }
    SplLimit2  : WORD;         { Second sample part size.                   }

    StepFrac   : BYTE;         { Sample incement fraction.                  }
    StepInt    : WORD;         { Sample incement integer.                   }

    Volume     : BYTE;         { Volume to be used.                         }

    LoopEnd    : WORD;         { Offset of the end of the loop in its part. }
    LoopLen    : WORD;         { Size of the loop in its part.              }
  END;

CONST                      { TModRawChan.Flags }
  rcfLongSample     = $01; { Set if it's a long (more than 65520 bytes) sample.     }
  rcfActiveChannel  = $02; { Set if the channel is activated (permission to sound). }
  rcfDoesLoop       = $04; { Set of the sample has a loop.                          }
  rcfPlaying2nd     = $08; { Set if playing the 2nd part of the long loop.          }
  rcfLongLoopLen    = $10; { Loop size goes from the 2nd part to the 1st.           }
  rcfLongLoopEnd    = $20; { Loop ends in the 2nd part.                             }
  rcfSampleFinished = $40; { Set if the sample has already finished.                }

VAR                                                 { Raw channels. }
  RawChan0 : ARRAY[1..SIZEOF(TModRawChan)] OF BYTE;
  RawChan1 : ARRAY[1..SIZEOF(TModRawChan)] OF BYTE;
  RawChan2 : ARRAY[1..SIZEOF(TModRawChan)] OF BYTE;
  RawChan3 : ARRAY[1..SIZEOF(TModRawChan)] OF BYTE;

  RawChannels : ARRAY[0..3] OF TModRawChan ABSOLUTE RawChan0;




{----------------------------------------------------------------------------}
{ Sinus table for the vibrato.                                               }
{____________________________________________________________________________}

CONST
  VibTabla : ARRAY[0..31] OF BYTE = (
      0, 24, 49, 74, 97,120,141,161,
    180,197,212,224,235,244,250,253,
    255,253,250,244,235,224,212,197,
    180,161,141,120, 97, 74, 49, 24
  );




{----------------------------------------------------------------------------}
{ Basic, fast assembler routines.                                            }
{____________________________________________________________________________}




{----------------------------------------------------------------------------}
{                                                                            }
{ ROUTINE: UnCanal                                                           }
{                                                                            }
{ Fills a buffer with 8 bit samples, calculated from a sample, a freq. and   }
{ a volume (a RawChannel).                                                   }
{ Implemented as several specialised routines, for speed's sake.             }
{ It doesn't play long samples yet.                                          }
{ This routine self-modifies, for speed's sake.                              }
{                                                                            }
{ IN:       CX    = Number of samples.                                       }
{           BX    = Offset of the channel data (TModRawChan).                }
{           DI    = Offset of the buffer to be filled.                       }
{                                                                            }
{ OUT:      The buffer will have been filled.                                }
{                                                                            }
{ MODIFIES: Every register except DS.                                        }
{                                                                            }
{............................................................................}

PROCEDURE UnCanal; ASSEMBLER;
  ASM
        TEST    [TModRawChan(DS:BX).Flags],rcfActiveChannel  { Active channel? }
        JZ      @@Desactivado                                { If not -> do the silent loop }

        TEST    [TModRawChan(DS:BX).Flags],rcfSampleFinished { Already finished? }
        JNZ     @@Desactivado                                { If it is -> do the silent loop }

        PUSH    BX                                           { BX is saved for restoring data at the end }

        TEST    [TModRawChan(DS:BX).Flags],rcfDoesLoop       { Does the sample have a loop? }
        JZ      @@NoDoesLoop                                 { If not -> do the loop-less routine }

{

  Sample with a loop (it doesn't check the end of the sample).

}

        MOV     AX,[TModRawChan(DS:BX).LoopEnd]              
        MOV     WORD PTR [CS:@@dlData2-2],AX                 { Puts the loop-end OFFSET in its instruction }

        MOV     AX,[TModRawChan(DS:BX).LoopLen]              
        MOV     WORD PTR [CS:@@dlData3-2],AX                 { Puts the loop-size in its instruction }

        LES     BP,DWORD PTR [TModRawChan(DS:BX).SplPosInt]  { Pointer to the next sample to be read }
        MOV     DL,BYTE PTR [TModRawChan(DS:BX).Volume]      { Volume }
        MOV     AL,[TModRawChan(DS:BX).StepFrac]             { Increment fraction }
        MOV     SI,[TModRawChan(DS:BX).StepInt]              { Increment integer  }

        MOV     AH,[TModRawChan(DS:BX).SplPosFrac]           { Position OFFSET }

        MOV     BX,AX   { R ********************************** }

{

      Bucle. Se entra con:
        DL = Volumen
        BL = Parte fraccionaria del incremento.
        SI = Parte entera del incremento.
        BH = Parte fraccionaria de la posicin en el sample.
        BP = Parte entera de la posicin en el sample.
        ES = Segmento del sample.
        DI = Buffer donde se almacenan las muestras.
        CX = Nmero total de muestras a generar.

}

@@dlLoop:
        MOV     AL,[ES:BP]                                   { Leo la muestra correspondiente }
        IMUL    DL                                           { Multiplico por el volumen }
        MOV     [DI],AX                                      { Lo meto en el buffer (Instruccin automodificada) }
        ADD     DI,8

        ADD     BH,BL                                        { Aade el incremento fraccionario }
        ADC     BP,SI                                        { Aade el incremento entero }
        JC      @@dlSplLoop                                  { Carry -> Ha pasado el lmite, seguro }
                                                             { (mximo n de muestras = 65520) }
@@dlChkLoop:
        CMP     BP,$1234                                     { CMP BP,[TModRawChan(DS:BX).LoopEnd] }
@@dlData2:                                                   { He llegado al pto. de retorno del loop? }
        JB      @@dlNoLoop

@@dlSplLoop:
        SUB     BP,$1234                                     { SUB BP,[TModRawChan(DS:BX).LoopLen] }
@@dlData3:                                                   { Si es as, vuelvo para atrs. Esto es muy importante hacerlo }
                                                             { restando el tamao del bucle, y conservando la parte frac. }

@@dlNoLoop:
        LOOP    @@dlLoop                                     { Y fin del bucle }

        JMP     @@Finish                                     { Salta al final, donde se almacenan los valores de por donde }
                                                             { han quedado los punteros y dems }

{

  Sample sin loop (no comprueba el fin de loop).
  Filosofa igual al anterior.

}

@@NoDoesLoop:
        MOV     AX,[TModRawChan(DS:BX).SplLimit]             { Pone el OFFSET del fin del sample en la instruccin }
        MOV     WORD PTR [CS:@@nlData2-2],AX

        LES     BP,DWORD PTR [TModRawChan(DS:BX).SplPosInt]  { Puntero a la prxima muestra a leer }
        MOV     DL,BYTE PTR [TModRawChan(DS:BX).Volume]      { Volumen }
        MOV     AL,[TModRawChan(DS:BX).StepFrac]             { Parte fraccionaria del incremento }
        MOV     AH,[TModRawChan(DS:BX).SplPosFrac]           { Parte fraccionaria del OFFSET del puntero a la muestra }

        MOV     SI,[TModRawChan(DS:BX).StepInt]              { Parte entera del incremento }

        MOV     BX,AX

{

      Bucle. Se entra con:
        DL = Volumen
        BL = Parte fraccionaria del incremento.
        SI = Parte entera del incremento.
        BH = Parte fraccionaria de la posicin en el sample.
        BP = Parte entera de la posicin en el sample.
        ES = Segmento del sample.
        DI = Buffer donde se almacenan las muestras.
        CX = Nmero total de muestras a generar.

}

@@nlLoop:
        MOV     AL,[ES:BP]                                   { Leo la muestra correspondiente }
        IMUL    DL                                           { Multiplico por el volumen }
        MOV     [DI],AX                                      { Lo meto en el buffer }
        ADD     DI,8

        ADD     BH,BL                                        { Aade el incremento fraccionario }
        ADC     BP,SI                                        { Aade el incremento entero }
        JC      @@nlSeguroFin                                { Carry -> Ha pasado el lmite del sample, seguro }
                                                             { (mximo n de muestras = 65520) }

        CMP     BP,$1234                                     { CMP BP,[TModRawChan(DS:BX).SplLimit] }
@@nlData2:                                                   { He llegado al final del sample? }
        JNB     @@nlSeguroFin                                { Si es as, dejo de calcular }

@@nlNoLoop:
        LOOP    @@nlLoop                                     { Y fin del bucle }

        JMP     @@Finish                                     { Salta al final, donde se almacenan los valores de por donde }
                                                             { han quedado los punteros y dems }

@@nlSeguroFin:                                               { Se ha terminado el sample }
        POP     BX                                           { Recupera el TModRawChan en BX }
        OR      BYTE PTR [TModRawChan(DS:BX).Flags],rcfSampleFinished                { Desactivo el canal }
        DEC     CX                                           { Decrementa el nmero de muestras, no se ha podido hacer antes }
        JCXZ    @@Fin                                        { Si ya no hay ms -> bye }

{

  Bucle correspondiente a un sample vaco. No se puede eliminar
  porque tiene que, por lo menos, poner el buffer a cero.

}

@@Desactivado:
        XOR     AX,AX                                        { Todas las muestras a cero }
@@Data2:
        MOV     [DI],AX                                      { Le meto el cero en el buffer }
        ADD     DI,8
        LOOP    @@Data2                                      { Fin del bucle }

        JMP     @@Fin                                        { Y me vuelvo sin restaurar nada }







@@Finish:
        POP     SI                                           { Recupero el TModRawChan }
        MOV     [TModRawChan(DS:SI).SplPosInt],BP            { Y guardo el OFFSET del sample donde se ha quedado }
        MOV     [TModRawChan(DS:SI).SplPosFrac],BH

@@Fin:
  END;




{----------------------------------------------------------------------------}
{ Rutinas que se dedican a interpretar la partitura.                         }
{____________________________________________________________________________}




{----------------------------------------------------------------------------}
{                                                                            }
{ RUTINA: SetNewSample                                                       }
{                                                                            }
{ Inicializa un nuevo sample en uno de los canales.                          }
{                                                                            }
{ ENTRADAS: Raw : TModRawChan correspondiente al canal.                      }
{           Spl : TSample correspondinte al canal.                           }
{                                                                            }
{ SALIDAS:  Ninguna.                                                         }
{                                                                            }
{............................................................................}

PROCEDURE SetNewSample(VAR Raw: TModRawChan; VAR Spl: TDefInstrument);
  CONST
    _or : BYTE    = 0;
    f   : BOOLEAN = FALSE;
  BEGIN
    ASM

        MOV     DI,WORD PTR Raw
        LES     SI,Spl

        MOV     AX,WORD PTR TDefInstrument([ES:SI]).data
        MOV     TModRawChan([DI]).SplOfs1,AX
        MOV     AX,WORD PTR TDefInstrument([ES:SI+2]).data
        MOV     TModRawChan([DI]).SplSeg1,AX                              { Inicializa los valores mnimos }
        MOV     _or,rcfActiveChannel

        MOV     AX,WORD PTR TDefInstrument([ES:SI+1]).repl
        AND     AX,AX
        JNZ     @@1
         MOV    AL,BYTE PTR TDefInstrument([ES:SI]).repl
         CMP    AL,4
         JNB    @@1
         MOV    f,1
         JMP    @@2
@@1:    MOV     f,0                                     { Si tiene loop (no s si es buena la comprobacin }
        OR      _or,rcfDoesLoop
@@2:

      END;

(*
    Raw.SplOfs1   := OFS(Spl.data^);
    Raw.SplSeg1   := SEG(Spl.data^);                                      { Inicializa los valores mnimos }
    _or           := rcfActiveChannel;

    f := Spl.repl <= 4;
    IF NOT f THEN INC(_or, rcfDoesLoop);                { Si tiene loop (no s si es buena la comprobacin }
*)

    IF Spl.len > MaxSample THEN BEGIN

      ASM

        MOV     DI,WORD PTR Raw                  { Entra aqu si es un sample largo (mayor de 65520 bytes) }
        LES     SI,Spl

        OR      _or,rcfLongSample
        MOV     TModRawChan([DI]).SplLimit1,MaxSample

      END;

(*
      INC(_or, rcfLongSample);                   { Entra aqu si es un sample largo (mayor de 65520 bytes) }

      Raw.SplLimit1 := MaxSample;
*)
      Raw.SplLimit2 := Spl.len - MaxSample;                      { Inicializa valores para el sample largo }
      Raw.SplOfs2   := OFS(Spl.xtra^);
      Raw.SplSeg2   := SEG(Spl.xtra^);

      IF NOT f THEN BEGIN                                                   { Si hay loop, pequeo lo :-) }
        IF (Spl.reps > MaxSample) OR (Spl.reps+Spl.repl <= MaxSample) THEN
          Raw.LoopLen := Spl.repl
        ELSE BEGIN
          Raw.LoopLen := Spl.repl - MaxSample;
          INC(_or, rcfLongLoopLen);
        END;
        IF Spl.reps+Spl.repl <= MaxSample THEN
          Raw.LoopEnd := Spl.reps + Spl.repl
        ELSE BEGIN
          Raw.LoopEnd := Spl.reps + Spl.repl - MaxSample;
          INC(_or, rcfLongLoopEnd);
        END;
      END;
    END ELSE BEGIN

      ASM

        MOV     DI,WORD PTR Raw                { Entra aqu si es un sample pequeo (menor de 65520 bytes) }
        LES     SI,Spl

        MOV     AX,WORD PTR TDefInstrument([ES:SI]).len
        MOV     TModRawChan([DI]).SplLimit1,AX

        MOV     AL,f
        AND     AL,AL
        JNZ     @@1
         MOV    AX,WORD PTR TDefInstrument([ES:SI]).repl
         MOV    TModRawChan([DI]).LoopLen,AX
         ADD    AX,WORD PTR TDefInstrument([ES:SI]).reps
         MOV    TModRawChan([DI]).LoopEnd,AX
@@1:

      END;

(*
      Raw.SplLimit1 := Spl.len;                { Entra aqu si es un sample pequeo (menor de 65520 bytes) }

      IF NOT f THEN BEGIN                                                                    { Si hay loop }
        Raw.LoopEnd := Spl.reps + Spl.repl;
        Raw.LoopLen := Spl.repl;
      END;
*)
    END;

    ASM

        MOV     DI,WORD PTR Raw                

        MOV     TModRawChan([DI]).SplPosFrac,0

        MOV     AX,TModRawChan([DI]).SplOfs1
        MOV     TModRawChan([DI]).SplPosInt,AX
        MOV     TModRawChan([DI]).SplOfs,AX

        MOV     AX,TModRawChan([DI]).SplSeg1
        MOV     TModRawChan([DI]).SplPosSeg,AX
        MOV     TModRawChan([DI]).SplSeg,AX

        MOV     AX,TModRawChan([DI]).SplLimit1
        MOV     TModRawChan([DI]).SplLimit,AX

        MOV     AL,_or
        MOV     TModRawChan([DI]).Flags,AL

    END;
(*
    Raw.SplPosFrac := 0;
    Raw.SplPosInt  := Raw.SplOfs1;
    Raw.SplPosSeg  := Raw.SplSeg1;

    Raw.SplOfs   := Raw.SplOfs1;
    Raw.SplSeg   := Raw.SplSeg1;
    Raw.SplLimit := Raw.SplLimit1;

    Raw.Flags  := _or;
*)
  END; 


                      



PROCEDURE MyMove(VAR Src, Dest; Bytes: WORD); ASSEMBLER;
  ASM
                PUSH    DS

                LDS     SI,[Src]
                LES     DI,[Dest]
                MOV     CX,[Bytes]

                CLD

                AND     CX,CX
                JZ      @@Fin

                TEST    SI,1
                JZ      @@nobeg
                 MOVSB
                 DEC    CX
                 JZ     @@Fin

@@nobeg:        MOV     BX,CX
                SHR     CX,1
                REP MOVSW
                MOV     CX,BX
                AND     CL,1
                JZ      @@Fin

                MOVSB
@@Fin:
                POP     DS
  END;






{----------------------------------------------------------------------------}
{                                                                            }
{ PROCEDIMIENTO: ProcessNewNote                                              }
{                                                                            }
{ Calcula y procesa la siguiente nota de la partitura.                       }
{                                                                            }
{ ENTRADAS: Ninguna.                                                         }
{                                                                            }
{ SALIDAS:  Ninguna.                                                         }
{                                                                            }
{............................................................................}

PROCEDURE ProcessNewNote;
  CONST
    f   : INTEGER      = 0;
    g   : INTEGER      = 0;
    i   : WORD         = 0;
    j   : WORD         = 0;
    n   : TModNote     = (sample:0);
    p   : PPattern     = NIL;
    pp  : PPatternLine = NIL;
    can : ^TCanal      = NIL;
  BEGIN

{    SetBorder($FF, 0, 0);}

    i := (NoteHd + 1) AND (NoteBuffSize - 1);
    NoteProcessed := @NoteBuff[i];
    MyMove(NoteBuff[NoteHd], NoteProcessed^, SIZEOF(NoteBuff[0]));
    NoteHd := i;
    WITH NoteProcessed^ DO BEGIN

      EoMod       := NextNote = $FF;

      IF EoMod THEN 
        IF MyLoopMod THEN BEGIN
          IF SequenceRepStart >= SequenceLength THEN
            NextSeq := 0
          ELSE
            NextSeq := SequenceRepStart;
          NextNote  := 0;
          EoMod     := FALSE;
        END ELSE BEGIN
          Playing   := FALSE;
          EXIT;
        END;

      NotePlaying := NextNote;
      SeqPlaying  := NextSeq;
      Volume      := UserVols;

      IF NextNote < 63 THEN
        INC(NextNote)
      ELSE BEGIN
        INC(NextSeq);
        IF NextSeq >= SequenceLength THEN NextNote := $FF
                                     ELSE NextNote := 0;
      END;

      p  := Patterns[PatternSequence[SeqPlaying]];
      pp := @p^[NotePlaying];

      FOR j := 0 TO 3 DO BEGIN

        can := @Canales[j];

        UnpackNote(pp^[j+1], n);

        IF (can^.nota.comm    = mcArpeggio)  OR
           (((can^.nota.comm  = mcVibrato) OR
             (can^.nota.comm  = mcVib_VSlide)) AND
            ((        n.comm <> mcVibrato) AND
             (        n.comm <> mcVib_VSlide))) THEN can^.freq := can^.nota.freq;

        MyMove(n, Note[j], SIZEOF(n));

        IF ((n.sample <> 0)                      AND
           (can^.nota.sample <> n.sample)) OR
           ((0 <> n.freq)                        AND
            (n.comm <> mcNPortamento)            AND
            (n.comm <> mcT_VSlide))              THEN BEGIN
          IF n.sample <> 0 THEN
            can^.nota.sample := n.sample;
          SetNewSample(RawChannels[j], Instruments[can^.nota.sample]);
        END;

        IF (n.freq <> 0) THEN
          can^.nota.freq := n.freq;

        IF (n.sample <> 0) THEN BEGIN
          can^.vol       := Instruments[can^.nota.sample].vol;
          IF can^.vol > 64 THEN can^.vol := 64;
        END;

        can^.nota.comm  := n.comm;
        can^.nota.param := n.param;

        CASE n.comm OF
          MCArpeggio:    BEGIN
                           can^.arpct := 0;

                           f := n.freq;
                           IF f = 0 THEN f := can^.nota.freq;

                           g := WORD((n.param SHR  4));
                           can^.arp1 := PeriodArray[(NoteIdx[f] AND $FF) + g];
                           g := WORD((n.param AND $F));
                           can^.arp2 := PeriodArray[(NoteIdx[f] AND $FF) + g];
                         END;
          mcTPortUp:     BEGIN
                           can^.tporta := -INTEGER(n.param);
                         END;
          mcTPortDown:   BEGIN
                           can^.tporta :=  INTEGER(n.param);
                         END;
          mcNPortamento: BEGIN
                           IF n.freq <> 0 THEN
                             can^.ffreq := n.freq
                           ELSE IF can^.ffreq = 0 THEN
                             can^.ffreq := can^.freq;

                           IF n.param = 0 THEN BEGIN
                             IF can^.nfreq > 0 THEN n.param := BYTE( can^.nfreq)
                                               ELSE n.param := BYTE(-can^.nfreq);
                           END;

                           IF INTEGER(can^.ffreq - can^.freq) >= 0 THEN
                             can^.nfreq :=  INTEGER(n.param)
                           ELSE
                             can^.nfreq := -INTEGER(n.param);

                           can^.nota.freq := can^.freq;
                         END;
          mcVibrato:     BEGIN
                           f := n.param AND $F;
                           IF f <> 0 THEN 
                             CASE can^.vibwave OF
                               0: can^.vibwidth := f;
                               1: can^.vibwidth := f;
                               2: can^.vibwidth := (f*255) SHR 7;
                             END;

                           f := n.param SHR  4;
                           IF f <> 0 THEN can^.vibdepth := f SHL 2;
{
                           IF (n.freq <> 0) AND NOT can^.vibnreset THEN
                             can^.vibpos := 0;
}
                         END;
          mcJumpPattern: BEGIN
                           NextSeq := n.param;
                           IF NextSeq >= SequenceLength THEN
                             NextNote := $FF
                           ELSE
                             NextNote := 0;
                         END;
          mcT_VSlide,
          mcVib_VSlide,
          mcVolSlide:    BEGIN
                           IF n.param > $F THEN can^.vols := n.param SHR 4
                                           ELSE can^.vols := -SHORTINT(n.param AND $F);
                         END;
          mcSetVolume:   BEGIN
                           IF n.param > 64 THEN n.param := 64;
                           can^.vol        := n.param;
                         END;
          mcEndPattern:  BEGIN
                           IF NextNote <> 0 THEN INC(NextSeq);
                           IF NextSeq >= SequenceLength THEN
                             NextNote := $FF
                           ELSE BEGIN
                             NextNote := (n.param AND $0F) +
                                         (n.param SHR   4)*10;
                             IF NextNote > 63 THEN NextNote := 63;
                           END;
                           IF NextNote <> $FF THEN NextNote := 0;
                         END;
          mcSetTempo:    BEGIN
                           IF n.param <> 0 THEN
                             Tempo := n.param;
                           IF Tempo > $1F THEN Tempo := $1F;
                         END;
          mcSetFilter:   BEGIN
                           IF PermitFilterChange THEN
                             FilterIsOn := n.param <> 0;
                         END;
          mcVolFineUp:   BEGIN
                           can^.vols :=           n.param AND $F;
                         END;
          mcVolFineDown: BEGIN
                           can^.vols := -SHORTINT(n.param AND $F);
                         END;
        END;

        IF (n.freq <> 0) AND (n.comm <> mcNPortamento) THEN
          can^.freq := n.freq;

      END;

      MuestrasPerTick := ActualHz DIV TicksPerSecond;

      IF MuestrasPerTick > MaxSplPerTick THEN
        MuestrasPerTick := MaxSplPerTick;

      NMuestras       := MuestrasPerTick * Tempo;
      NoteHz          := ActualHz;

    END;

    NoteTl    := NoteHd;
    NoteSound := NoteProcessed;

{    SetBorder(0, 0, 0);}

  END;




{----------------------------------------------------------------------------}
{ Rutinas de comandos, para cada Tick.                                       }
{                                                                            }
{ Se entra con SI apuntando al TCanal correspondiente.                       }
{____________________________________________________________________________}

PROCEDURE TickArpeggio;    ASSEMBLER; { 0 xx }
  ASM

        INC     TCanal([SI]).arpct
        MOV     AL,TCanal([SI]).arpct
        DEC     AL
        JNZ     @@2
         MOV    AX,TCanal([SI]).nota.freq
         MOV    TCanal([SI]).freq,AX
        JMP    @@Fin
@@2:    DEC     AL
        JNZ     @@3
         MOV    AX,TCanal([SI]).arp1
         MOV    TCanal([SI]).freq,AX
        JMP    @@Fin
@@3:     MOV    AX,TCanal([SI]).arp2
         MOV    TCanal([SI]).freq,AX
         MOV    TCanal([SI]).arpct,0
@@Fin:
  END;

PROCEDURE TickTPortUpDown; ASSEMBLER; { 1 xx y 2 xx }
  ASM

        MOV     AX,TCanal([SI]).tporta
        ADD     AX,TCanal([SI]).nota.freq
        MOV     TCanal([SI]).nota.freq,AX
        MOV     TCanal([SI]).freq,AX

  END;

PROCEDURE TickNPortamento; ASSEMBLER; { 3 xy }
  ASM

        MOV     AX,TCanal([SI]).ffreq
        AND     AX,AX
        JZ      @@Fin
        MOV     AX,TCanal([SI]).nfreq
        AND     AX,AX
        JZ      @@Fin
        MOV     BX,TCanal([SI]).nota.freq
        ADD     BX,AX
        TEST    AH,80h
        JNZ      @@neg
         CMP    BX,TCanal([SI]).ffreq
         JA     @@pneg
        JMP     @@cnt
@@neg:   TEST   BH,80h
         JNZ    @@pneg
         CMP    BX,TCanal([SI]).ffreq
         JAE    @@cnt
@@pneg:  MOV    BX,TCanal([SI]).ffreq
@@cnt:  MOV     TCanal([SI]).nota.freq,BX
        MOV     TCanal([SI]).freq,BX
@@Fin:

  END;

PROCEDURE TickVibrato;     ASSEMBLER; { 4 xy }
  ASM

        MOV     AH,TCanal([SI]).vibwave
        MOV     DH,TCanal([SI]).vibwidth
        MOV     DL,TCanal([SI]).vibpos
        MOV     AL,DL
        AND     AH,AH
        JNZ     @@nosn
         SHR    AL,2
         AND    AL,1Fh
         MOV    BX,OFFSET VibTabla
         XLAT
         JMP    @@set
@@nosn: DEC     AH
        JNZ     @@notr
         SHL    AL,1
         JNC    @@set
          NOT   AL
@@set:   MUL    DH
         SHL    AH,1
         MOV    AL,AH
         XOR    AH,AH
        JMP     @@calc
@@notr:  MOV    AL,DH
         XOR    AH,AH
@@calc: MOV     BX,TCanal([SI]).nota.freq
        TEST    DL,80h
        JZ      @@mas
         NEG    AX
@@mas:  ADD     BX,AX
        MOV     TCanal([SI]).freq,BX
        ADD     DL,TCanal([SI]).vibdepth
        MOV     TCanal([SI]).vibpos,DL

  END;

PROCEDURE TickVolSlide;    ASSEMBLER; { A xy }
  ASM

        MOV     AL,TCanal([SI]).vol
        MOV     AH,TCanal([SI]).vols
        ADD     AL,AH
        CMP     AL,64
        JBE     @@Fin
         XOR    AL,AL
         TEST   AH,80h
         JNZ    @@Fin
          MOV   AL,64
@@Fin:  MOV     TCanal([SI]).vol,AL

  END;

PROCEDURE TickT_VSlide;    ASSEMBLER; { 5 xy }
  ASM

        CALL    TickNPortamento
        JMP     TickVolSlide

  END;

PROCEDURE TickVib_VSlide;  ASSEMBLER; { 6 xy }
  ASM

        CALL    TickVibrato
        JMP     TickVolSlide

  END;



PROCEDURE TickTremolo;     ASSEMBLER; { 7 xy }
  ASM
  END;

PROCEDURE TickNPI1;        ASSEMBLER; { 8 xx }
  ASM
  END;

PROCEDURE TickSampleOffs;  ASSEMBLER; { 9 xx }
  ASM
  END;



PROCEDURE TickSetFilter;   ASSEMBLER; { E 0x }
  ASM
  END;

PROCEDURE TickFinePortaUp; ASSEMBLER; { E 1x }
  ASM
  END;

PROCEDURE TickFinePortaDn; ASSEMBLER; { E 2x }
  ASM
  END;

PROCEDURE TickGlissCtrl;   ASSEMBLER; { E 3x }
  ASM
  END;

PROCEDURE TickVibCtrl;     ASSEMBLER; { E 4x }
  ASM
  END;

PROCEDURE TickFineTune;    ASSEMBLER; { E 5x }
  ASM
  END;

PROCEDURE TickJumpLoop;    ASSEMBLER; { E 6x }
  ASM
  END;

PROCEDURE TickTremCtrl;    ASSEMBLER; { E 7x }
  ASM
  END;

PROCEDURE TickNPI2;        ASSEMBLER; { E 8x }
  ASM
  END;

PROCEDURE TickRetrigNote;  ASSEMBLER; { E 9x }
  ASM
  END;

PROCEDURE TickVolFineUpDn; ASSEMBLER; { E Ax y E Bx }
  ASM

        MOV     AL,TempoCt
        DEC     AL
        JNZ     @@Fin
         JMP    TickVolSlide
@@Fin:

{
            mcVolFineUp,
            mcVolFineDown: BEGIN
                             IF TempoCt = 1 THEN BEGIN
                             INC(vol, vols);
                             IF vol > 64 THEN
                               IF vols >= 0 THEN vol := 64
                                            ELSE vol := 0;
                             END;
                           END;
}
  END;

PROCEDURE TickNoteCut;     ASSEMBLER; { E Cx }
  ASM
  END;

PROCEDURE TickNoteDelay;   ASSEMBLER; { E Dx }
  ASM
  END;

PROCEDURE TickPattDelay;   ASSEMBLER; { E Ex }
  ASM
  END;

PROCEDURE TickFunkIt;      ASSEMBLER; { E Fx }
  ASM
  END;


PROCEDURE TickNone;        ASSEMBLER; { 0 00 }
  ASM
  END;

CONST
  TickCommOfs : ARRAY[mcArpeggio..mcNone] OF WORD = (
    OFS(TickArpeggio),   { 0 xx }
    OFS(TickTPortUpDown),{ 1 xx }
    OFS(TickTPortUpDown),{ 2 xx }
    OFS(TickNPortamento),{ 3 xy }
    OFS(TickVibrato),    { 4 xy }
    OFS(TickT_VSlide),   { 5 xy }
    OFS(TickVib_VSlide), { 6 xy }
    OFS(TickTremolo),    { 7 xy }
    OFS(TickNPI1),       { 8 xx }
    OFS(TickSampleOffs), { 9 xx }
    OFS(TickVolSlide),   { A xy }
    OFS(TickNone),       { B xx }
    OFS(TickNone),       { C xx }
    OFS(TickNone),       { D xx }
    OFS(TickNone),       { E xy }
    OFS(TickNone),       { F xx }

    OFS(TickSetFilter),  { E 0x }
    OFS(TickFinePortaUp),{ E 1x }
    OFS(TickFinePortaDn),{ E 2x }
    OFS(TickGlissCtrl),  { E 3x }
    OFS(TickVibCtrl),    { E 4x }
    OFS(TickFineTune),   { E 5x }
    OFS(TickJumpLoop),   { E 6x }
    OFS(TickTremCtrl),   { E 7x }
    OFS(TickNPI2),       { E 8x }
    OFS(TickRetrigNote), { E 9x }
    OFS(TickVolFineUpDn),{ E Ax }
    OFS(TickVolFineUpDn),{ E Bx }
    OFS(TickNoteCut),    { E Cx }
    OFS(TickNoteDelay),  { E Dx }
    OFS(TickPattDelay),  { E Ex }
    OFS(TickFunkIt),     { E Fx }

    OFS(TickNone)        { 0 00 }

  );

PROCEDURE FillChannels;
  CONST
    FirstTick : BOOLEAN      = TRUE;
    i         : WORD         = 0;
    p         : ^TModRawChan = NIL;
    q         : POINTER      = NIL;
    SplBuf    : ARRAY[0..3] OF WORD = ( 0, 0, 0, 0 );
  BEGIN
{
    SetBorder($FF, $FF, 0);
}
    DelaySamples := Buffers[BuffIdx].InUse;
    IF DelaySamples THEN
      BEGIN
        EXIT;
      END;


    FOR i := 0 TO 3 DO BEGIN
      p := @RawChannels[i];
      q := @BuffData[BuffIdx][i+1];
      ASM
        PUSH    BP
        PUSH    DI
        PUSH    SI
        MOV     CX,MuestrasPerTick
        MOV     BX,WORD PTR p
        MOV     DI,WORD PTR q
        CALL    UnCanal
        POP     SI
        POP     DI
        POP     BP
      END;

      SplBuf[i] := FilterChunkWord(BuffData[BuffIdx][i+1], MuestrasPerTick, 4, FilterVal, SplBuf[i]);
    END;


    WITH Buffers[BuffIdx] DO BEGIN
      InUse    := TRUE;
      NSamples := MuestrasPerTick;
      RateHz   := NoteHz;
      Channels := 4;
    END;

    INC(BuffIdx);
    IF BuffIdx > NumBuffers THEN BuffIdx := 1;

  END; { PROCEDURE FillChannels }

{----------------------------------------------------------------------------}
{                                                                            }
{ PROCEDIMIENTO: ProcessTick                                                 }
{                                                                            }
{ Procesa un tick de la msica. Normalmente, se usan 50 ticks por segundo,   }
{ pero puede cambiarse.                                                      }
{                                                                            }
{ ENTRADAS: Ninguna.                                                         }
{                                                                            }
{ SALIDAS:  Ninguna.                                                         }
{                                                                            }
{............................................................................}

PROCEDURE ProcessTick;
  CONST
    SOTCanal = SIZEOF(TCanal);
    Semaphor : BYTE        = 0;
    incr     : INTEGER     = 0;
    Can      : PCanal      = NIL;
    Raw      : PModRawChan = NIL;
    NoteHzFreq : LONGINT   = 0;
    i        : WORD        = 0;
    j        : WORD        = 0;
    step     : LONGINT     = 0;
    _SS      : WORD        = 0;
    _SP      : WORD        = 0;
    FBCount  : WORD        = 0;
  LABEL
    Fin, Fin1, Fin2;
  BEGIN 

    IF NOT Playing THEN
      BEGIN
        TempoCT := 1;
        GOTO Fin1;
      END;

    IF Semaphor <> 0 THEN
      GOTO Fin2;

    INC(Semaphor);

    ASM
        MOV     [_SS],SS
        MOV     [_SP],SP
        MOV     AX,DS
        MOV     SS,AX
        MOV     SP,OFFSET PlayModStack + PlayModStackSize
    END;

    IF DelaySamples THEN BEGIN
      FillChannels;

      IF DelaySamples THEN GOTO Fin;
    END;

    INC(TickCount);

    INC(TempoCt);
    IF TempoCt >= NoteProcessed^.Tempo THEN BEGIN
{
      SetBorder($FF, $FF, $FF);
}
      ProcessNewNote;

      IF NOT Playing THEN GOTO Fin;
      TempoCt := 0;
    END;
{
    SetBorder($FF, $00, $00);
}
    IF NOT MyCanFallBack THEN
      PleaseFallBack := 0;

    IF PleaseFallBack > 0 THEN BEGIN
      PleaseFallBack := 0;
      i := ActualHz;
      WHILE (i = ActualHz) AND (i <> ActiveDevice^.GetRealFreqProc(0)) DO
        BEGIN
          DEC(DesiredHz, 100);
          i := ActiveDevice^.GetRealFreqProc(DesiredHz);
        END;
      ChangeSamplingRate(DesiredHz);
    END;

    ASM

                MOV     AL,TempoCt
                AND     AL,AL
                JZ      @@nofirst

                MOV     CX,4
@@lp:            PUSH   CX
                 MOV    AL,CL
                 DEC    AL
                 MOV    BL,SOTCanal
                 MUL    BL
                 MOV    SI,OFFSET Canales
                 ADD    SI,AX
                 MOV    BL,TCanal([SI]).nota.comm
                 ADD    BL,BL
                 XOR    BH,BH
                 ADD    BX,OFFSET TickCommOfs
                 CALL   WORD PTR [BX]
                 POP    CX
                 LOOP   @@lp
@@nofirst:
    END;


    FOR i := 0 TO 3 DO
      BEGIN
        Can := @Canales[i];
        Raw := @RawChannels[i];

        IF NOT Permisos[i] THEN Raw^.Flags := Raw^.Flags AND NOT rcfActiveChannel
                           ELSE Raw^.Flags := Raw^.Flags OR      rcfActiveChannel;

        Raw^.Volume := WORD(Can^.vol*UserVols[i]) DIV 255;

        IF Can^.freq   = 0 THEN Can^.freq   := 1;
        IF NoteHz      = 0 THEN NoteHz      := 1;

        ASM

                LES     DI,[Can]                    { LONGINT(NoteHzFreq) := }
                MOV     DX,TCanal([ES:DI]).freq     {   WORD(Can^.freq) *    }
                MOV     AX,[NoteHz]                 {   WORD(NoteHz)         }
                MUL     DX
                MOV     WORD PTR [NoteHzFreq],AX
                MOV     WORD PTR [NoteHzFreq+2],DX

        END;

        step := (65536 * 14000) DIV NoteHzFreq;

        Raw^.StepFrac := LO(step);
        Raw^.StepInt  := step SHR 8;

        IF FilterIsOn THEN FilterVal := FilterOn
                      ELSE FilterVal := FilterOff;
      END;

    FillChannels;

Fin:
{
    SetBorder(0, 0, 0);
}

    ASM
        MOV     SS,[_SS]
        MOV     SP,[_SP]
    END;

    DEC(Semaphor);

Fin1:
    IF ModTickProcValid THEN 
      ModTickProc(TempoCt = 0);
Fin2:
  END;




FUNCTION IdleGiver : PSampleBuffer; FAR;
  BEGIN
    IdleGiver := NIL;
  END;


FUNCTION BufferGiver : PSampleBuffer; FAR;
  BEGIN
    BufferGiver := NIL;
    IF NOT Buffers[BuffGive].InUse THEN EXIT;
    BufferGiver := @Buffers[BuffGive];
    INC(BuffGive);
    IF BuffGive > NumBuffers THEN BuffGive := 1;
  END;




PROCEDURE FillWithSamples(VAR Buff; Size: WORD);
  CONST
    mBuff : PIntBuff = NIL;
  BEGIN

    mBuff := Buffers[BuffGive].IData;

    ASM
        PUSH    DS

        XOR     SI,SI

        MOV     CX,[Size]
        MOV     AX,[MuestrasPerTick]
        AND     AX,AX
        JZ      @@bien
        CMP     AX,CX
        JNC     @@bien

        SUB     CX,AX
        MOV     SI,CX
        MOV     CX,AX

@@bien: CLD
        MOV     DX,8
        LDS     BX,[mBuff]
        LES     DI,[Buff]

@@lp:    MOV    AX,[BX]
         ADD    AX,[BX+2]
         ADD    AX,[BX+4]
         ADD    AX,[BX+6]
         ADD    BX,DX
         STOSW
         LOOP   @@lp

        AND     SI,SI
        JZ      @@Fin

        MOV     CX,SI
        XOR     AX,AX
        REP STOSW

@@Fin:  POP     DS
    END;

  END;




PROCEDURE PlayStart;
  VAR
    i, j : WORD;
  BEGIN

    ASM CLI END;

    MyLoopMod       := (LoopMod AND (SequenceRepStart < SequenceLength)) OR ForceLoopMod;
    TempoCt         := 254;
    TickCount       := 0;
    NextNote        := 0;
    NextSeq         := 0;
    DelaySamples    := FALSE;
    MuestrasPerTick := 1;

    WITH NoteBuff[0] DO BEGIN
      EoMod       := FALSE;
      Tempo       := 6;
      NotePlaying := 0;
      SeqPlaying  := 0;
      Volume      := UserVols;
      NMuestras   := 0;
    END;

    NoteHd        := 0;
    NoteTl        := 0;
    NoteSound     := @NoteBuff[0];
    NoteProcessed := @NoteBuff[0];

    FillChar(Canales, SIZEOF(Canales), 0);

    FOR i := 0 TO 3 DO
      WITH Canales[i] DO BEGIN
        nota.freq   := 800;
        nota.sample := 1;
        nota.comm   := mcNone;
        freq        := 800;
      END;

    FillChar(Buffers,  SIZEOF(Buffers),  0);
    FillChar(BuffData, SIZEOF(BuffData), 0);
    FOR i := 1 TO 4 DO Buffers[i].IData := @BuffData[i];
    BuffIdx  := 1;
    BuffGive := 1;

    FillChar(RawChannels, SIZEOF(RawChannels), 0);

    ASM STI END;

    SetBufferAsker(IdleGiver);

    StartSampling;

    MyCanFallBack  := FALSE;
    Playing        := TRUE;

    FOR i := 1 TO NumBuffers DO
      ProcessTick;

    SetBufferAsker(BufferGiver);

    WHILE DeviceIdling DO;

    PleaseFallBack := 0;
    MyCanFallBack  := CanFallBack;

  END;




PROCEDURE ChangeSamplingRate(Hz: WORD);
  VAR
    MyHz : WORD;
  LABEL
    Otra;
  BEGIN
Otra:
    DesiredHz := Hz;
    MyHz      := ActiveDevice^.GetRealFreqProc(Hz);

    IF MyHz >  MaxSplPerTick * TicksPerSecond THEN
      BEGIN
        DEC(Hz, 100);
        GOTO Otra;
      END;

    IF MyHz < 1000 THEN
      BEGIN
        INC(Hz, 100);
        GOTO Otra;
      END;

    IF MyHz <> ActualHz THEN
      BEGIN
        ActualHz := MyHz;
        SetPeriodicProc(ProcessTick, TicksPerSecond * 3 {DIV 2});
      END;

  END;




PROCEDURE PlayStop;
  BEGIN

    Playing := FALSE;

    SetBufferAsker(IdleGiver);

  END;




BEGIN
  Playing        := FALSE;
  LoopMod        := FALSE;
  ActualHz       := 0;

  IF FilterIsOn THEN FilterVal := FilterOn
                ELSE FilterVal := FilterOff;

  FillChar(UserVols, SIZEOF(UserVols), 255);
  FillChar(Permisos, SIZEOF(Permisos), TRUE);
END.
