//*****************************************************************
//
// Module Name: AvkAlg.h
//
// Description: AVK Video and Audio Microcode Definitions
//
// Status: Version 1 Release 2
//
// Copyright Intel Corp. 1990, 1993
//
// Copyright IBM Corp. 1990, 1993
//
// All Rights Reserved
//
//****************************************************************

#ifndef AVKALG_H_INCLUDED          // Ignore multiple includes
#define AVKALG_H_INCLUDED          // Flag file as included

// **********************************************************************
// AVSS Frame Format
// **********************************************************************

#define AVK_SMALLEST_FRAME_SIZE 16

// **************************************************
// Encoded Bitstream Header format
// **************************************************

// Hdr: ver | flag | bitslo | bitshi | reserved (2) | height | width
typedef struct {
	U16	CodeAlgVer;	// [ 0 ]	// Algorithm number
	U16	CodeFlag;	// [ 1 ]	// Flag (see below)
	U16	CodeSizeLow;	// [ 2 ]	// Size (see below)
	U16	CodeSizeHigh;	// [ 3 ]
	U32	CodeReserved;	// [ 4 ]	// Write as zero
	U16	CodeYLen;	// [ 6 ]	// Resolution: Y / X order
	U16	CodeXLen;	// [ 7 ]
} AVK_CODE_HDR;

#define AVK_CODE_S_LSB  0x02	// LSB always set to 1
#define AVK_CODE_STILL  0x04	// Still: makes no ref to previous image
#define AVK_CODE_PLANE0 0x10	// Color Plane: 2-bit field
#define AVK_CODE_PLANE1 0x20	//    00, 01, 10 <=> Y, v, U
#define AVK_CODE_RESERVED   0xC000	// reserved

// CodeSize: Bits, except old RTV
#define AVK_CODE_BYTES(alg, ph)	\
	(( ((((U32) pHdr->CodeSizeHigh) << 16) | pHdr->CodeSizeLow) + 7) / 8)

// **********************************************************************
// Audio Microcode
// **********************************************************************

// **************************************************
// Special frames: Null (Ignore) & Dummy (from RTV: Time, but no data)

// Frame size (bytes) in AVSS frame header
#define AVK_AUD_NULL_FRM_SIZE	0	// Null frame: AVSS Frame header
#define AVK_AUD_DUMMY_FRM_SIZE	8	// Dummy frame: AVSS Frame header

// Dummy Frame data: 4-word bitstream header -- INTERNAL AlgNum
#define AVK_AUD_DUMMY_FRM_INIT(pData, AlgNum, SampRate)	{  \
	((U16 FAR *) pData)[0] = 0;	\
    ((U16 FAR *) pData)[1] = ((AlgNum) << 8) | 0xff;  \
	((U16 FAR *) pData)[2] = (SampRate);	\
	((U16 FAR *) pData)[3] = 0;	}

// **************************************************
// Audio Algorithms

//The following symbols define FAMILIES of audio algorithm microcode
//logic, according to the encoding method:
#define AVK_AUD_ALG_ADPCM4	0	// ADPCM, 4-bit
#define AVK_AUD_ALG_PCM8	1	// PCM, 8-bit


// ADPCM4 Sample Rates: Samples / Sec
#define AVK_AUD_ALG_ADPCM4_SAMP_HIGH	 33075
#define AVK_AUD_ALG_ADPCM4_SAMP_MED	 31129	// 124519
#define AVK_AUD_ALG_ADPCM4_SAMP_LOW	  8268

// PCM8 Sample Rates: Samples / Sec
#define AVK_AUD_ALG_PCM8_SAMP_HIGH	 44100
#define AVK_AUD_ALG_PCM8_SAMP_MED	 22050
#define AVK_AUD_ALG_PCM8_SAMP_LOW	 11025

#define AVK_AUD_ALG_MONO	0x1000	// Mono (Flags, can OR)
#define AVK_AUD_ALG_STEREO	0x2000	// Stereo

#define AVK_AUD_ALG_PLAYBACK	0x4000	// Playback (Flags, can OR)
#define AVK_AUD_ALG_CAPTURE	0x8000	// Capture

#define AVK_AUD_ALG_NUM(alg)    ((alg) & 0x0ff)

// macro to derive bits per sample from known algorithm numbers...
#define AVK_AUD_ALG_SAMP_BITS(alg)	\
	(((alg) == AVK_AUD_ALG_ADPCM4) ? 4 :		\
	(((alg) == AVK_AUD_ALG_PCM8)   ? 8 : -1))

// audio sample rates BELOW this value permit two audio streams at once...
// The number of SAMPLES PER SECOND maximum, at which two audio streams
// at once can be allowed.
#define AVK_MAX_AUDIO_DUAL  12000

// **********************************************************************
// Video Microcode
// **********************************************************************

// **************************************************
// Special frames: Null (Ignore) & Dummy (from RTV: Time, but no data)

// Frame size (bytes) in AVSS frame header
#define AVK_VID_NULL_FRM_SIZE	0	// Null frame: AVSS Frame header
#define AVK_VID_DUMMY_FRM_SIZE	16	// Dummy frame: AVSS Frame header

#define AVK_VID_DUMMY_FRM_CNT	2	// Number of Dummy frames to write

// Dummy Frame data: 8-word, 16-byte, 16*8-bit bitstream header
#define AVK_VID_DUMMY_FRM_INIT(pData, AlgNum, xLen, yLen)	{  \
	((U16 FAR *) pData)[0] = (AlgNum);	\
	((U16 FAR *) pData)[1] = 0;	\
	((U16 FAR *) pData)[2] = (AVK_VID_DUMMY_FRM_SIZE * 8);	\
	((U16 FAR *) pData)[3] = 0;	\
	((U16 FAR *) pData)[4] = 0;	\
	((U16 FAR *) pData)[5] = 0;	\
	((U16 FAR *) pData)[6] = (yLen);	\
	((U16 FAR *) pData)[7] = (xLen);	}

// **************************************************
// Compression / Decompression Algorithms
//   Algorithm Numbers
// Some of these have alternate spellings
// **************************************************

// PLV 9-bit Motion Video Algorithms, Decode only
//   Algorithm 5 is backward compatible with version 6 (i.e., use 6 for 5 too)

#define AVK_VID_ALG_PLV_10    5    // PLV 1.0,
#define AVK_VID_ALG_PLV_15 	  6    // PLV 1.5
#define AVK_VID_ALG_PLV_20	 20    // PLV 2.0
#define AVK_VID_ALG_PLV_201	 21    // PLV 2.0.1

// alternates
#define   AVK_PLV_1_0 AVK_VID_ALG_PLV_10
#define   AVK_PLV_1_5 AVK_VID_ALG_PLV_15
#define   AVK_PLV_2_0 AVK_VID_ALG_PLV_20
#define   AVK_PLV_2_01 AVK_VID_ALG_PLV_201

// RTV 9-bit Motion Video Algorithms, Decode & Encode

#define AVK_VID_ALG_RTV_10	192    // RTV 1.0,  10 fps, 1 Mbit/sec
#define AVK_VID_ALG_RTV_15H	194    // RTV 1.5H, 30 fps, 3 Mbit/sec
#define AVK_VID_ALG_RTV_15	195    // RTV 1.5,  30 fps, 1.6 Mbit/sec
#define AVK_VID_ALG_RTV_20	200    // RTV 2.0, flex data rate
#define AVK_VID_ALG_RTV_21	201    // RTV 2.1, sw decode
#define AVK_VID_ALG_RTV_22	202    // RTV 2.2, data rate arg

// alternates
#define   AVK_RTV_1_5 AVK_VID_ALG_RTV_15
#define   AVK_RTV_2_0 AVK_VID_ALG_RTV_20
#define   AVK_RTV_2_1 AVK_VID_ALG_RTV_21
#define   AVK_RTV_2_2 AVK_VID_ALG_RTV_22

// 8-bit (9-bit) Planar Lossy Stills Algorithms

// PIC 1.0 (9-bit), Lossy YUV-9 stills
#define AVK_VID_ALG_PIC9_10A	128	// Old PIC 1.0, decode only
#define AVK_VID_ALG_PIC9_10	130	// PIC 1.0 for AVK, encode & decode

// alternates
#define   AVK_PIC_YUV_9OLD	AVK_VID_ALG_PIC9_10A
#define   AVK_PIC_YUV_9		AVK_VID_ALG_PIC9_10

// JPEG Baseline, 9R-7
#define AVK_VID_ALG_JPEG_10	129    // JPEG baseline; 9-bit only for now
// alternates
#define   AVK_JPEG_YUV_9	AVK_VID_ALG_JPEG_10

// 16-bit Lossless Stills Algorithms

// PIC 1.0 (16-bit), 16-bit lossless
#define AVK_VID_ALG_PIC16_10A	  1    // Original PIC 1.0, decode only
#define AVK_VID_ALG_PIC16_10	  2    // Improved PIC 1.0, encode & decode

// alternates
#define   AVK_PIC_YUV_16OLD	AVK_VID_ALG_PIC16_10A
#define   AVK_PIC_YUV_16	AVK_VID_ALG_PIC16_10

// **************************************************
// Compression / Decompression Algorithms
//   Algorithm Arguments
// **************************************************

// **************************************************
// PLV 1.0, 9-bit (Alg 5): Decode arguments

#define AVK_PLV_10_DECODE_ARGS_SIZE	0
#define AVK_PLV_10_DECODE_WORK_SIZE	(64 * 1024L)

// **************************************************
// PLV 1.5, 9-bit (Alg 6): Decode arguments

#define AVK_PLV_15_DECODE_ARGS_SIZE	0
#define AVK_PLV_15_DECODE_WORK_SIZE	(64 * 1024L)

// **************************************************
// PLV 2.0, 9-bit (Alg 20): Decode arguments

#define AVK_PLV_20_DECODE_ARGS_SIZE	0
#define AVK_PLV_20_DECODE_WORK_SIZE	(64 * 1024L)

// **************************************************
// RTV 1.5, 9-bit (Alg 195): Decode arguments

#define AVK_RTV_15_DECODE_ARGS_SIZE	0
#define AVK_RTV_15_DECODE_WORK_SIZE	0L

// **************************************************
// RTV 2.0, 9-bit (Alg 200): Encode arguments

#define AVK_RTV_20_ENCODE_ARGS_SIZE	sizeof(AVK_RTV_20_ENCODE_ARGS)
#define AVK_RTV_20_ENCODE_WORK_SIZE	(64 * 1024L)	// full page, aligned
#define AVK_RTV_20_ENCODE_BITS_RATIO	50	// approx % of original

#define AVK_RTV_20_ENCODE_ALIGN_ORIG_X	8	// Image alignment, origin
#define AVK_RTV_20_ENCODE_ALIGN_ORIG_Y	8	//   align to multiple
#define AVK_RTV_20_ENCODE_ALIGN_X	16	// Image alignment, size
#define AVK_RTV_20_ENCODE_ALIGN_Y	8	//   align to multiple
#define AVK_RTV_20_ENCODE_MAX_X 	256	// Image resolution
#define AVK_RTV_20_ENCODE_MAX_Y 	288	//   maximum size

#define AVK_RTV_20_ENCODE_DATA_NAME	"ke080200.vsh"	// load in work page

typedef struct  AVK_RTV_20_ARGS {
	// Standard Encode Args
	U16	ArgCnt;		// Word count, size of structure (12)
	U16	AlgNum;		// Algorithm number (200)
	U16	xPix, yPix; 	// location and size of rectangle to compress
	U16	xLen, yLen;

	// Algorithm-specific Encode Args
	U16	StillPeriod;	// how often to force a still
			// (1 = all stills; N = force a still every Nth)
	U16	BytesPerFrame;	// Max bytes/frame (2.2 only)
	U16	LinesPerFrame;	// amount of time available for compression,
			// in (DB) horiz line times (ignored for now)
	U16	Flags;	// See below.
	U16	RelQuant;	// quant value for rel frames (0-7)
				// (increasing quant increases compression)
	U16	StillQuant;	// quantization for stills (0-7; 0-3 best)
} AVK_RTV_20_ENCODE_ARGS;

// alternates
typedef AVK_RTV_20_ENCODE_ARGS	rtv20args;

// RTV 2.0 Encode flag bits

#define AVK_RTV_20_FLAG_PREFILTER	0x0008	// pre-filter
		// Blurs image to greatly increases compression (about 2X).
		// Essential to achieve CD-ROM rate at 128x240.
#define AVK_RTV_20_FLAG_ASPECT_45	0x0000	// 4:5 aspect ratio
#define AVK_RTV_20_FLAG_ASPECT_25	0x0100	// 2:5 aspect ratio
#define AVK_RTV_20_FLAG_EZ_DECODE	0x0010	// Easier decode

// alternates
#define AVK_RTV_20_PREFILTER	AVK_RTV_20_FLAG_PREFILTER
#define AVK_RTV_20_ASPECT_45	AVK_RTV_20_FLAG_ASPECT_45
#define AVK_RTV_20_ASPECT_25	AVK_RTV_20_FLAG_ASPECT_25
#define AVK_RTV_20_EZ_DECODE	AVK_RTV_20_FLAG_EZ_DECODE

// **************************************************
// RTV 2.0, 9-bit (Alg 200): Decode arguments

#define AVK_RTV_20_DECODE_ARGS_SIZE	0
#define AVK_RTV_20_DECODE_WORK_SIZE	0L

// **************************************************
// PIC 1.0, 9-bit Original (Alg 128): Decode arguments

#define AVK_PIC9_10A_DECODE_ARGS_SIZE	0
#define AVK_PIC9_10A_DECODE_WORK_SIZE	(32 * 1024L)	// 1/2 page, aligned

// **************************************************
// PIC 1.0, 9-bit (Alg 130): Encode arguments

#define AVK_PIC9_10_ENCODE_ARGS_SIZE	sizeof(AVK_PIC9_10_ENCODE_ARGS)
#define AVK_PIC9_10_ENCODE_WORK_SIZE	(64 * 1024L)	// full page, aligned
#define AVK_PIC9_10_ENCODE_BITS_RATIO	50	// approx % of original

typedef struct {
	// Standard Encode Args
	U16	ArgCnt;		// Word count, size of structure (8)
	U16	AlgNum;		// Algorithm number (130)
	U16	xPix, yPix; 	// location and size of rectangle to compress
	U16	xLen, yLen;

	// Algorithm-specific Encode Args
	U16	Filtering;	// 0-255, typically 15-20, good 11
	U16	Quantization;	// 0-11, typically 5, good 3
} AVK_PIC9_10_ENCODE_ARGS;

typedef AVK_PIC9_10_ENCODE_ARGS	epic9args;

// **************************************************
// PIC 1.0, 9-bit (Alg 130): Decode arguments

#define AVK_PIC9_10_DECODE_ARGS_SIZE	0
#define AVK_PIC9_10_DECODE_WORK_SIZE	(32 * 1024L)	// 1/2 page, aligned

// **************************************************
// JPEG, 9-bit (Alg 129): Encode arguments

#define AVK_JPEG_10_ENCODE_ARGS_SIZE	sizeof(AVK_JPEG_10_ENCODE_ARGS)
#define AVK_JPEG_10_ENCODE_QUAN_SIZE	192	// Subtract ArgSize no Quant
#define AVK_JPEG_10_ENCODE_WORK_SIZE	(64 * 1024L)	// full page, aligned
#define AVK_JPEG_10_ENCODE_BITS_RATIO	50	// approx % of original (1/8)

typedef struct {
	// Standard Encode Args
	U16	ArgCnt;		// Word count, size of struct (7 or 103)
	U16	AlgNum;		// Algorithm number (129)
	U16	xPix, yPix; 	// location and size of rectangle to compress
	U16	xLen, yLen;

	// Algorithm-specific Encode Args
	U16	QuanOffset;	// Quantization table offset, bytes
				// 0 or 14 for (3*64=192 byte table follows)
	char	Quant[192];	// Quantization table (if Offset not 0)
} AVK_JPEG_10_ENCODE_ARGS;

typedef AVK_JPEG_10_ENCODE_ARGS	ejpegargs;

// **************************************************
// JPEG, 9-bit (Alg 129): Decode arguments

#define AVK_JPEG_10_DECODE_ARGS_SIZE	0	// None
#define AVK_JPEG_10_DECODE_WORK_SIZE	(64 * 1024L)	// full page, aligned

// **************************************************
// PIC 1.0 A, 16-bit (Alg 1): Decode arguments

#define AVK_PIC16_10A_DECODE_ARGS_SIZE	0
#define AVK_PIC16_10A_DECODE_WORK_SIZE	0L

// **************************************************
// PIC 1.0, 16-bit (Alg 2): Encode arguments

#define AVK_PIC16_10_ENCODE_ARGS_SIZE	sizeof(AVK_PIC16_10_ENCODE_ARGS)
#define AVK_PIC16_10_ENCODE_WORK_SIZE	(10 * 1024L)	// 8K + pad
#define AVK_PIC16_10_ENCODE_BITS_RATIO	50	// approx % of original

typedef struct {
	// Standard Encode Args
	U16	ArgCnt;		// Word count, size of structure (6)
	U16	AlgNum;		// Algorithm number (2)
	U16	xPix, yPix; 	// location and size of rectangle to compress
	U16	xLen, yLen;

	// Algorithm-specific Encode Args
	// (None)
} AVK_PIC16_10_ENCODE_ARGS;

typedef AVK_PIC16_10_ENCODE_ARGS	epic16args;

// **************************************************
// PIC 1.0, 16-bit (Alg 2): Decode arguments

#define AVK_PIC16_10_DECODE_ARGS_SIZE	0
#define AVK_PIC16_10_DECODE_WORK_SIZE	0L

#endif       // End of #ifndef to control multiple includes
