/**MOD+***********************************************************************/
/* Module:    asample1.c                                                     */
/*                                                                           */
/* Purpose:   TP1, APPC program browser                                      */
/*                                                                           */
/* The transaction programs TP1 and TP2 enable a user to browse through a    */
/* file on another system.  The user of TP1 is presented with a single data  */
/* block at a time, in hex and character format.  After each block a user    */
/* can request the next block, request the previous block, or quit.          */
/* TP1 (the invoking TP) sends a filename to TP2 (the invoked TP).  If TP2   */
/* locates the file, it returns the first block to TP1, otherwise it issues  */
/* deallocate, and terminates.  If TP1 receives a block it displays it on    */
/* the screen and waits for a user prompt; if it receives an indication      */
/* that TP2 has terminated, it too terminates.                               */
/* If the user asks for the next block when TP2 has sent the last one, TP2   */
/* wraps to the beginning of file.  Similarly, TP2 wraps to send the last    */
/* block if the previous one is requested when the first block is displayed. */
/* Neither program attempts to recover from errors.  A 'bad' return code     */
/* from APPC causes the program to terminate with an explanatory message.    */
/*                                                                           */
/* (C) COPYRIGHT DATA CONNECTION LIMITED 1989-2006                           */
/*                                                                           */
/**MOD-***********************************************************************/

#include <appc_c.h>
#include <acssvcc.h>
#define  INCL_SUB
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <unistd.h>

/*****************************************************************************/
/* remove comments from following line to implement debugging to screen      */
/*****************************************************************************/
/* #define DEBUGGING */

typedef int boolean;
#define TRUE 1
#define FALSE 0


/*****************************************************************************/
/* Flipping macros.                                                          */
/* These are required to pass control information between TP1 and TP2 even   */
/* if the TPs are running on systems with different byte ordering.           */
/* Include the file svconfig.h to get the correct value for PFLIPRQD.        */
/* If this file is not available, set PFLIPRQD to 1 for OS's with Intel      */
/* byte ordering - otherwise, set to 0.                                      */
/*****************************************************************************/
#include <svconfig.h>
#if PFLIPRQD==1
#ifdef _LP64
#define PFLIPI(X) (unsigned short)(((X >> 8) & 0x00ff) | ((X << 8) & 0xff00))

#define PLFXIPI(X) (PFLIPI(((unsigned long)X & 0x0000ffff)) << 16 | \
                    PFLIPI( (unsigned long)X >> 16))

#define PLFLIPI(X) ((((unsigned long)PLFXIPI(X)) << 32)) | PLFXIPI((X >> 32))
#else
#define PFLIPI(X) (unsigned short)(((X >> 8) & 0x00ff) | ((X << 8) & 0xff00))

#define PLFLIPI(X) (PFLIPI(((unsigned long)X & 0x0000ffff)) << 16 | \
                    PFLIPI( (unsigned long)X >> 16))
#endif
#else

#define PFLIPI(X) (X)
#define PLFLIPI(X) (X) 

#endif

/*****************************************************************************/
/* structures required by APPC                                               */
/*****************************************************************************/
struct tp_started               tpstart;
struct mc_allocate              mcalloc;
struct mc_send_data             mcsend;
struct mc_receive_and_wait      mcrec;
struct mc_deallocate            mcdealloc;
struct tp_ended                 tpended;
struct convert                  conv_block;              /* for convert verb */

/*****************************************************************************/
/* Data area                                                                 */
/*****************************************************************************/
unsigned char   sharebufptr[512];

unsigned char   tp_id[ 8 ];                  /* TP id returned by tp_started */
AP_UINT32       conv_id;          /* conversation id returned by mc_allocate */

typedef struct
{
   size_t block_number;                 /* block no in range 1 to num_blocks */
   size_t num_blocks;                   /* size of file in 256 byte blocks */
   size_t block_size;
   unsigned char      data_block[ 256 ];
} BLOCK_RECEIVED;

void  initialize();
void  allocate();
void  send();
void  display_block();
boolean receive_wait();
char  get_prompt();
void  deallocate();
void  exit_error();
void  terminate();


/**PROC+**********************************************************************/
/* Name:      main                                                           */
/*                                                                           */
/* Purpose:   Handles protocols for operator end of file browser             */
/*                                                                           */
/* Returns:   0  if user terminates the program                              */
/*            1  if any errors occur                                         */
/*                                                                           */
/* Params:    IN     argc     -  Argument count                              */
/*            IN     argv     -  Array of arguments                          */
/*                                                                           */
/* Operation: Initializes and allocates the conversation. Sends the file name*/
/*            to the partner tp. Then sits on a receive_and_wait. When this  */
/*            returns, it switches on the what_rcvd parameter. If it is data */
/*            it is displayed. If it is send permission, the user is prompted*/
/*            for the next block of file to be viewed, and this is sent to   */
/*            the partner tp. This extracts the required portion of the file */
/*            and sends it back here, to be picked up on the next receive_   */
/*            and_wait verb.                                                 */
/*                                                                           */
/**PROC-**********************************************************************/
int main( argc, argv )
   int  argc;
   char *argv[];
{
   char      prompt = ' ';
   boolean   good_receive;
   char     *fname;

   if ( argc < 2 )
   {
      printf( "Usage %s filename\n", argv[ 0 ] );
      exit( -1 );
   }
   else
   {
      fname = argv[ 1 ];
   }
   /**************************************************************************/
   /* Issue tp_started, allocate, and send file name                         */
   /**************************************************************************/
   initialize();
   allocate();
   send( fname, strlen( fname ) );
   do
   {
      /***********************************************************************/
      /* Issue receive_and_wait verb                                         */
      /***********************************************************************/
      good_receive = receive_wait();
      if( good_receive )
      {

         switch( mcrec.what_rcvd )
         {
            case AP_DATA_COMPLETE :

              /***************************************************************/
              /* Convert all ints from line format                           */
              /***************************************************************/
              ((BLOCK_RECEIVED*)sharebufptr)->block_number =
                PLFLIPI(((BLOCK_RECEIVED*)sharebufptr)->block_number);
              ((BLOCK_RECEIVED*)sharebufptr)->num_blocks =
                PLFLIPI(((BLOCK_RECEIVED*)sharebufptr)->num_blocks);
              ((BLOCK_RECEIVED*)sharebufptr)->block_size =
                PLFLIPI(((BLOCK_RECEIVED*)sharebufptr)->block_size);

               display_block( fname, ( BLOCK_RECEIVED * ) sharebufptr );
               break;

            case AP_DATA_INCOMPLETE :
               printf( "Incomplete block received\n" );
               break;

            case AP_SEND :
               prompt = get_prompt();

               /**************************************************************/
               /* do not send Q prompt                                       */
               /**************************************************************/
               if( prompt == 'F' || prompt == 'B' )
                  send( &prompt, 1 );
               break;

            default :
               printf("Unknown 'What_Rcvd' code rcvd %04x\n", mcrec.what_rcvd);
               break;
         }
      }
   } while( prompt != 'Q' && good_receive );

   if ( !good_receive )
   {
      printf( "File name %s reported invalid or not found\n", argv[ 1 ] );
      exit( -1 );
   }
   deallocate();
   terminate();
   return( 0 );
}


/**PROC+**********************************************************************/
/* Name:      get_from_env                                                   */
/*                                                                           */
/* Purpose:   Get the value of an environment variable (or default if none)  */
/*            present                                                        */
/*                                                                           */
/* Returns:   Pointer to string                                              */
/*                                                                           */
/* Params:    Name of environment variable                                   */
/*            Default value to use if variable empty or no variable present  */
/*                                                                           */
/**PROC-**********************************************************************/
unsigned char * get_from_env (pname, pdef)
char * pname;
char * pdef;
{
  char * temp;

  temp = (char *) getenv (pname);

  if (temp == NULL || *temp == (char) NULL)
  {
    return ((unsigned char *)pdef);
  }

  return ((unsigned char *)temp);
}




/**PROC+**********************************************************************/
/* Name:      initialize                                                     */
/*                                                                           */
/* Purpose:   Issue a tp_started verb                                        */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void initialize()
{
   unsigned char * envptr;
   unsigned char  tpstart_name[ 64 ];

#ifdef DEBUGGING
   printf( "initialize: top\n" );
#endif

   memset( &tpstart, (int ) '\0', sizeof( tpstart ) );
   tpstart.opcode = AP_TP_STARTED;

   /**************************************************************************/
   /* Set up the local TP name.                                              */
   /**************************************************************************/
#define LOCTPNAME "TPNAME1"
   memset( tpstart_name, ( int ) ' ', sizeof( tpstart_name ) ); /* 64 blanks */
   envptr = get_from_env ("LOCTPNAME", LOCTPNAME);
   memcpy( tpstart_name, envptr, strlen((char *)envptr) );

   /**************************************************************************/
   /* Set up the LU alias.                                                   */
   /**************************************************************************/
#define LOCLUALIAS "TPLU1"
   memset( tpstart.lu_alias, ( int ) ' ', 8 );
   envptr = get_from_env ("LOCLUALIAS", LOCLUALIAS);
   memcpy( tpstart.lu_alias, envptr, strlen((char *)envptr) );

   /**************************************************************************/
   /* convert ASCII tpstart_name to EBCDIC tpstart.tp_name                   */
   /**************************************************************************/
   memset(&conv_block, 0, sizeof(conv_block));
   conv_block.opcode    = SV_CONVERT;
   conv_block.opext     = 0;
   conv_block.direction = SV_ASCII_TO_EBCDIC;
   conv_block.char_set  = SV_AE;
   conv_block.len       = sizeof( tpstart_name );
   conv_block.source    = tpstart_name;
   conv_block.target    = tpstart.tp_name;
   ACSSVC_C( (char *) (&conv_block) );

   APPC( &tpstart );

   if ( tpstart.primary_rc  == AP_OK )
   {
      /***********************************************************************/
      /* save tp id                                                          */
      /***********************************************************************/
      memcpy( tp_id, tpstart.tp_id, sizeof( tp_id ) );
   }
   else
   {
      exit_error( tpstart.primary_rc );
   }
#ifdef DEBUGGING
   printf( "initialize: end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      allocate                                                       */
/*                                                                           */
/* Purpose:   Issue an mc_allocate call                                      */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void allocate()
{
   unsigned char * envptr;
   unsigned char  partner_tpname[ 64 ];
   unsigned char  mode_name[ 8 ];

#ifdef DEBUGGING
   printf( "mc_allocate\n" );
#endif

   memset(&mcalloc, 0, sizeof(mcalloc));
   mcalloc.opcode     = AP_M_ALLOCATE;
   mcalloc.opext      = AP_MAPPED_CONVERSATION;
   mcalloc.format     = 1;
   memcpy( mcalloc.tp_id, tp_id, sizeof( tp_id ) );
   mcalloc.sync_level = AP_NONE;
   mcalloc.rtn_ctl    = AP_WHEN_SESSION_ALLOCATED;
   mcalloc.security   = AP_NONE;

   /**************************************************************************/
   /* Set up the remote TP name for the allocate.                            */
   /**************************************************************************/
#define REMTPNAME "TPNAME2"
   memset( partner_tpname, ( int ) ' ', sizeof( partner_tpname ) );
   envptr = get_from_env ("REMTPNAME", REMTPNAME);
   memcpy( partner_tpname, envptr, strlen((char *)envptr) );

   /**************************************************************************/
   /* convert ASCII partner_name to EBCDIC mcalloc.tp_name                   */
   /**************************************************************************/
   memset(&conv_block, 0, sizeof(conv_block));
   conv_block.opcode    = SV_CONVERT;
   conv_block.opext     = 0;
   conv_block.direction = SV_ASCII_TO_EBCDIC;
   conv_block.char_set  = SV_AE;
   conv_block.len       = sizeof( partner_tpname );
   conv_block.source    = partner_tpname;
   conv_block.target    = mcalloc.tp_name;
   ACSSVC_C( (char *) (&conv_block) );

   /**************************************************************************/
   /* Set up the mode name for the allocate.                                 */
   /**************************************************************************/
#define MODENAME "LOCMODE"
   memset( mode_name, ( int ) ' ', sizeof( mode_name ) );
   envptr = get_from_env ("MODENAME", MODENAME);
   memcpy( mode_name, envptr, strlen((char *)envptr) );

   /**************************************************************************/
   /* convert ASCII mode_name to EBCDIC mcalloc.mode_name                    */
   /**************************************************************************/
   memset(&conv_block, 0, sizeof(conv_block));
   conv_block.opcode    = SV_CONVERT;
   conv_block.opext     = 0;
   conv_block.direction = SV_ASCII_TO_EBCDIC;
   conv_block.char_set  = SV_A;
   conv_block.len       = 8;
   conv_block.source    = (unsigned char *) mode_name;
   conv_block.target    = mcalloc.mode_name;
   ACSSVC_C( (char *) (&conv_block) );

   /**************************************************************************/
   /* Set up the partner LU alias for the allocate.                          */
   /**************************************************************************/
#define REMLUALIAS "TPLU2"
   memset( mcalloc.plu_alias, ( int ) ' ', sizeof( mcalloc.plu_alias ) );
   envptr = get_from_env ("REMLUALIAS", REMLUALIAS);
   memcpy( mcalloc.plu_alias, envptr, strlen((char *)envptr) );

   APPC( &mcalloc );

   if( mcalloc.primary_rc != AP_OK )
   {
      exit_error( mcalloc.primary_rc );
   }

   conv_id = mcalloc.conv_id;

#ifdef DEBUGGING
   printf( "mc_allocate: end\n" );
#endif
}

/**PROC+**********************************************************************/
/* Name:      send                                                           */
/*                                                                           */
/* Purpose:   Issue an mc_send call                                          */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    IN   datalen      length of data to be sent                    */
/*            IN   *datastring  data to be sent                              */
/*                                                                           */
/* Operation: NOTE: This procedure uses the extended function of the send    */
/*                  verb whereby multiple verbs may be issued with a single  */
/*                  call. In this case a PREPARE_TO_RECEIVE_FLUSH is issued  */
/*                  after the SEND.                                          */
/*                                                                           */
/**PROC-**********************************************************************/
void send( datastring, datalen )
   unsigned char *      datastring;
   unsigned    datalen;
{
   int i;

   /**************************************************************************/
   /* call appc (send_data) to send datastring for a length of datalen.      */
   /* If return code other than AP_OK, exit with error message.              */
   /**************************************************************************/

#ifdef DEBUGGING
   printf( "send: top with datalen %d\n", datalen );
#endif

   memset(&mcsend, 0, sizeof(mcsend));
   mcsend.opcode  = AP_M_SEND_DATA;
   mcsend.opext   = AP_MAPPED_CONVERSATION;
   mcsend.format   = 1;
   for( i = 0; i < datalen; i++ )
   {
      sharebufptr[ i ] = datastring[ i ];
   }
   memcpy( mcsend.tp_id, tp_id, sizeof( tp_id ) );
   mcsend.conv_id = conv_id;
   mcsend.dlen    = datalen;
   mcsend.dptr    = sharebufptr;
   mcsend.type    = AP_SEND_DATA_P_TO_R_FLUSH;

   APPC( &mcsend );

   if( mcsend.primary_rc != AP_OK )
   {
      exit_error( mcsend.primary_rc );
   }

#ifdef DEBUGGING
   printf( "send: end\n" );
#endif

}


/**PROC+**********************************************************************/
/* Name:      receive_wait                                                   */
/*                                                                           */
/* Purpose:   Issue an mc_receive_and_wait                                   */
/*                                                                           */
/* Returns:   TRUE   - data received                                         */
/*            FALSE  - an error has occured                                  */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/* Operation: Issues a receive and wait verb. If the return code is dealloc  */
/*            abend, the other tp could not open the file to be browsed.     */
/*            NOTE: This is the standard usage of receive verbs where status */
/*                  e.g. AP_SEND and data had to be received on separate     */
/*                  calls of the verb. See the other TP's receive for the    */
/*                  extended usage.                                          */
/*                                                                           */
/**PROC-**********************************************************************/
boolean receive_wait()
{

#ifdef DEBUGGING
   printf( "mc_receive_and_wait\n" );
#endif

#ifdef DEBUGGING
   printf( "receive_wait: top\n" );
#endif
   memset(&mcrec, 0, sizeof(mcrec));
   mcrec.opcode  = AP_M_RECEIVE_AND_WAIT;
   mcrec.opext   = AP_MAPPED_CONVERSATION;
   memcpy( mcrec.tp_id, tp_id, sizeof( tp_id ) );
   mcrec.conv_id = conv_id;
   mcrec.max_len = sizeof( BLOCK_RECEIVED );
   mcrec.dptr    = sharebufptr;
   APPC( &mcrec );

   if ( mcrec.primary_rc != AP_OK )
   {
      if ( mcrec.primary_rc == AP_DEALLOC_ABEND )
      {
         return( FALSE );
      }
      else
      {
         exit_error( mcrec.primary_rc );
      }
   }

#ifdef DEBUGGING
   printf( "receive_wait: end\n" );
#endif
   return( TRUE );
}


/**PROC+**********************************************************************/
/* Name:      display_block                                                  */
/*                                                                           */
/* Purpose:   Displays the newly received block of data                      */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    *fname       - file containing the data                        */
/*            *r_blockptr  - the received data                               */
/*                                                                           */
/* Operation: Displays the data in hex, and characters                       */
/*                                                                           */
/**PROC-**********************************************************************/
void display_block( fname, r_blockptr )
    char               * fname;
    BLOCK_RECEIVED *r_blockptr;
{
   size_t   ii;
   int   ch;
   char  buf[ 17 ];


#ifdef DEBUGGING
   printf( "display_block: top\n" );
#endif
   printf(
  "File %s: Size %u blocks.  Block Number %u, size %d\n\n\n", fname,
          r_blockptr->num_blocks, r_blockptr->block_number,
          r_blockptr->block_size );

   buf[ 16 ] = '\0';
   for( ii = 0; ii < r_blockptr->block_size; )
   {
      ch = r_blockptr->data_block[ ii ];
      printf( "%2.2X ", ch & 0xFF );
      /***********************************************************************/
      /* check if char is printable                                          */
      /***********************************************************************/
      buf[ ii % 16 ] = ( ch >= 0x20 && ch <= 0x7E ) ?  ( char ) ch : '.';
      if( ( ++ii % 16 ) == 0 )
      {
         printf( "      %s\n", buf );
      }
   }
   if( ii = ii % 16 )
   {
      buf[ ii ] = '\0';
      for( ; ii < 16; ii++ )
      {
         printf( "   " );
      }
      printf( "      %s\n", buf );
   }

#ifdef DEBUGGING
   printf( "display_block:end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      get_prompt                                                     */
/*                                                                           */
/* Purpose:   Asks the user for the next block to view, or (Q)uit            */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
char get_prompt()
{
   int    ch;

#ifdef DEBUGGING
   printf( "get_prompt: top\n" );
#endif

   printf( "\nEnter F (forward), B (back), or Q (quit): " );
   do
   {
      ch = getchar();
      ch = ( char ) toupper( ch );
   } while( ch != 'F' && ch != 'B' && ch != 'Q' );
   printf( "%c\n\n", ch );

#ifdef DEBUGGING
   printf( "get_prompt: end\n" );
#endif
   return( ch );
}


/**PROC+**********************************************************************/
/* Name:      deallocate                                                     */
/*                                                                           */
/* Purpose:   Issues a deallocate normal                                     */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void deallocate()
{

#ifdef DEBUGGING
   printf( "deallocate: end\n" );
#endif

   memset(&mcdealloc, 0, sizeof(mcdealloc));
   mcdealloc.opcode       = AP_M_DEALLOCATE;
   mcdealloc.opext        = AP_MAPPED_CONVERSATION;
   memcpy( mcdealloc.tp_id, tp_id, sizeof( tp_id ) );
   mcdealloc.conv_id      = conv_id;
   mcdealloc.dealloc_type = AP_FLUSH;
   APPC( &mcdealloc );

   if( mcdealloc.primary_rc != AP_OK )
   {
      exit_error( mcdealloc.primary_rc );
   }

#ifdef DEBUGGING
   printf( "deallocate: end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      exit_error                                                     */
/*                                                                           */
/* Purpose:   Produces error message for APPC error conditions               */
/*                                                                           */
/* Returns:   -1   Error has occurred                                        */
/*                                                                           */
/* Params:    rc   the APPC return code                                      */
/*                                                                           */
/**PROC-**********************************************************************/
void exit_error( rc )
   int    rc;
{
   switch( rc )
   {
      case AP_ALLOCATION_ERROR :
         printf( "APPC returned allocation error\n" );
         break;

      case AP_CANCELLED :
         printf( "APPC returned cancelled error\n" );
         break;

      case AP_DEALLOC_ABEND :
         printf( "APPC returned Dealloc_abend\n" );
         break;

      case AP_COMM_SUBSYSTEM_NOT_LOADED :
         printf( "APPC returned Comms Subsystem not loaded\n" );
         break;

      case AP_PARAMETER_CHECK :
         printf( "APPC returned Parameter Check\n" );
         break;

      case AP_STATE_CHECK :
         printf( "APPC returned State Check\n" );
         break;

      case AP_STACK_TOO_SMALL :
         printf( "APPC returned Stack too small\n" );
         break;

      default :
         printf( "APPC returned primary_rc %04X\n", rc );
         break;
   }
   exit( -1 );
}


/**PROC+**********************************************************************/
/* Name:      terminate                                                      */
/*                                                                           */
/* Purpose:   Issues a tp_ended verb                                         */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void terminate()
{

#ifdef DEBUGGING
   printf( "terminate: top\n" );
#endif

   memset(&tpended, 0, sizeof(tpended));
   tpended.opcode = AP_TP_ENDED;
   memcpy( tpended.tp_id, tp_id, sizeof( tp_id ) );
   APPC( &tpended );

   if ( tpended.primary_rc != AP_OK )
   {
      exit_error( tpended.primary_rc );
   }


#ifdef DEBUGGING
   printf( "terminate: end\n" );
#endif
}


