#!/usr/bin/ksh
#
# NAME: do-build.ksh driverName [ debug | beta ]
# 
# SAMPLE USAGES:
#   do-build.ksh  driver1
#   do-build.ksh  2001-03-01 debug
#
# PURPOSE:
#  This is the top level script to build the application "myApp".
#
# CUSTOMIZATION NOTES: 
# * See the sections labeled "CUSTOMIZATION REQUIRED".
# * This example has 3 variables that need to be added to the master.list
#   file, because they change often and/or are set at build time. 
#   + They need to be defined in Section "Definition" and 
#   + They need to be added to the master.list in Section "Adding to List". 
#   The variables are:
#    DRIVER_NAME      -> Identifier which might be related to the build date.
#                        This is passed as an input parameter to this script.
#    BUILD_MODE       -> Normal (default), or Debug or Beta
#    OPERATING_SYSTEM -> Using "uname -a" to find out the name.
#                        This is computed at build-time.
# * The file "level.list" has the serviceability strings that do not change
#   often. The contents of this file is copied into the "master.list". 
# * The master.list file is a temporary file that contains all the 
#   serviceability strings. This file is generated every single time 
#   this build script is invoked and it is used inside the main make file 
#   (myApp-x.mak) which in turn, calls do-what.ksh.
#
#******************************************************************

if [ $# -eq 0 ]
then
  print "Error: need to specify driver name."
  print "Usage: do-build.ksh driverName [ debug | beta ]"
  exit 1
fi

print "do-build.ksh: Starting a build of the myApp application"
print "date: `date`"
print ""

# CUSTOMIZATION REQUIRED: Section "Definition"
# * Set up the variables that change frequently:
export DRIVER_NAME="2001-01-16"
export BUILD_MODE="Normal"

# Find out the version and name of the operating system    
                                                           
export OPERATING_SYSTEM=`uname -a`                                
if [ "$OPERATING_SYSTEM" = "" ]                            
then                                                       
   print "*** Error: no output from uname"                 
   exit 1                                                  
fi                                                         

# CUSTOMIZATION REQUIRED:
# * You need to list the files to be removed before doing a build. 

rm myApp             # This is the main application.
rm myAppLevel        # This is the standalone level application.
rm what-info.h       # This is the common include header file.
rm what-info.c       # This is the generic include source file.
rm what-info-level.c # This is the include source file for level application.
rm master.list       # This is a temporary file, with the complete list of strings.
rm *.o


# CUSTOMIZATION REQUIRED:
# * You need to handle the appropriate processing of input arguments.
#   For example, the DEBUG and BETA options are set.

if [ $# -ge 1 ]
then
  typeset -u DRIVER_NAME=$1
  if [ $# -eq 2 ]
  then
    typeset -u OPTION=$2
    case "$OPTION" in
    BETA|beta)
      FLAGS='-DBETA'
      export BUILD_MODE="BETA"
      ;;
    DEBUG|debug)
      FLAGS='-DDEBUG -g'
      export BUILD_MODE="DEBUG"
      ;;
    *)
      print "Unrecognized option: $OPTION"
      ;;
    esac
  fi
fi


# CUSTOMIZATION REQUIRED: Section "Adding to List"
# * It is necessary to create the master.list file with the variables
#   defined in this script. Follow the format below, which is also used
#   in the file with those serviceability strings that do not change often,
#   such as level.list. The format is a variableName (such as driverInfo) 
#   followed by a value which can be several strings.
print "   driverInfo Driver: $DRIVER_NAME"      >>  master.list
print "buildModeInfo Build Mode: $BUILD_MODE"   >>  master.list
print " platformInfo This code was generated in $OPERATING_SYSTEM" >>  master.list


# CUSTOMIZATION REQUIRED:
# * If you use a name other than "level.list", then customize the following
#   statements.
levelListFile="level.list"
while read line                                                                
do                                                                             
   print $line >> master.list
done < $levelListFile                                                          


# CUSTOMIZATION REQUIRED:
# * Invoke your make file

make -f myApp-x.mak all FLAGS="$FLAGS"

# end of file

