package com.ibm.abt.web.servlet;

/**
 * This type was created in VisualAge.
 */
import java.io.* ;
import java.lang.* ;
import java.util.* ;
public class RouterConfiguration implements WsiConstants {
	private static RouterConfiguration current ;  // Hold the current WSI Router configuration
	private String abtWsiBaseName ;   	// Hold the value of the environment variable ABTWSI_BASENAME
	private Vector links ;				// LINK entries found in configuration file
	private Vector transports ;			// TRANSPORT entries found in configuration file
	private boolean verbose = false ;	// VERBOSE value from configuration file
	private boolean debug = false ;     // Indicates whether debug messages should be written
	private boolean log = true;			// LOG value from configuration file
	private String debugFileName ;		// file name where LOG and DEBUG messages are written
	private String errorFileName ;		// file name where formatted error information is written	
	private String codePage ;			// CODEPAGE read from configuration file
	private int socketTimeOut = WSI_DEFAULT_SOCKET_TIMEOUT ;         
										// The time in milliseconds before the socket times out
/**
 * VastWsiRouterConfiguration constructor comment.
 */
public RouterConfiguration() {
	super();
}
/**
 * 
 * @return a String containing the value of the environment variable ABTWSI_BASENAME
 * 
*/
public String getAbtWsiBaseName() {
	
	return abtWsiBaseName ;
}
/**
 * 
 * @return a String containing the 'CODEPAGE' for this configuration.
*/
public String getCodePage() {
	
	return codePage ;

}
/**
 * Return the name of the VAST Web Connection configuration file.  This is
 * determined by querying the environment variable 'ABTWSI_BASENAME'.  If the environment
 * variable is not set, the default name is used.
 */
public String getConfigurationFileName() {

	String baseName = getAbtWsiBaseName() ;
	String fileName = null ;

	if (baseName == null )
		fileName = ( getDefaultDirectory() + DEFAULT_NAME + CONFIG_FILE_EXT ) ;
	else
		fileName = baseName + CONFIG_FILE_EXT ;

	return fileName ;
}
/**
 * 
 * @return the currently active configuration
*/
public static RouterConfiguration getCurrent() {
	
	if ( current == null ) {
		current = new RouterConfiguration() ;
		current.load() ; }

	return current;

}
/**
 * 
 * @return a boolean indicating whether debug messages should be written.
*/
public boolean getDebug() {
	
	return debug ;

}
/**
 * 
 * @return a String containing the name of a file where debug information should be written.
 */
public String getDebugFileName() {
	
	return debugFileName ;

}
/**
 * 
 * @return a String containing the default directory name where VAST Web Server Interface files are stored
*/
public static String getDefaultDirectory() {

	return System.getProperty("user.dir") + System.getProperty("file.separator") ;
}
/**
 * 
 * @return a boolean containing the 'errorFile' file name setting for this configuration.
 * If the name is not set, build a default name.
*/
public String getErrorFileName() {
	

	if ( errorFileName == null ) {
		if ( getAbtWsiBaseName() != null )
			errorFileName = getAbtWsiBaseName() + ERROR_FILE_EXT ;
		else
			errorFileName = getDefaultDirectory() + DEFAULT_NAME + ERROR_FILE_EXT ;
	}
	
		return errorFileName ;
	}
/**
 * 
 * @return the appropriate WsiLink for usage with
 * the part specified by pathInfo.  If multiple links match
 * the pathinfo, the last located link is used.
*/
public WsiLink getLinkForPathInfo(String pathInfo) {
	
	Vector searchLinks = getLinks() ;
	WsiLink targetLink = null ;
	WsiLink currentLink = null ;
	StringTokenizer tokenizer ;
	String token ;

	for (int i = 0; (i < searchLinks.size()); i++ ) {
		currentLink = (WsiLink) searchLinks.elementAt(i) ;
	// Special case. If pathinfo contains only the wildcard string "*"
		if (currentLink.getPathInfo().equals(MSG_STRING_WILDCARD))
			targetLink = currentLink ;
		else {
		tokenizer = new StringTokenizer(currentLink.getPathInfo(),MSG_STRING_WILDCARD);
		if (tokenizer.hasMoreTokens()) {
			token = tokenizer.nextToken();
			if ( pathInfo.startsWith(token) )
				targetLink = currentLink ; } } }
	
	return targetLink;
	}
/**
 * 
 * @return a Vector containing the 'LINK' entries for this configuration
*/
public Vector getLinks() {
	if (links == null) {
		links = new Vector() ;
	}	

return links ;

}
/**
 * 
 * @return a boolean containing the 'LOG' for this configuration.
*/
public boolean getLog() {
	
	return log ;

}
/**
 * 
 * @return a String containing the file name for the log file.
*/
public String getLogFileName() {

	String fileName ;
// Return null if logging is not enabled
	if (!getLog())
		return null ;

	fileName = getDebugFileName();
	if ( fileName == null ) {
		if ( getAbtWsiBaseName() != null )
			fileName = getAbtWsiBaseName() + LOG_FILE_EXT ;
		else
			fileName = getDefaultDirectory() + DEFAULT_NAME + LOG_FILE_EXT ;
	}
	
		return fileName ;
	}
/**
 * 
 * @return an InputStream that references the property file for this class
*/
private InputStream getPropertyFileInputStream() {
	
	String propertyFileName ;
	InputStream propertyFileInputStream = null;

	propertyFileName = getDefaultDirectory() + DEFAULT_NAME + PROPERTY_FILE_EXT;

	try {
		propertyFileInputStream = new FileInputStream(propertyFileName) ; }
	catch (FileNotFoundException e ) {} ;
	
	return propertyFileInputStream ;
	
}
/**
 * 
 * Return the value of the 'SOCKET_TIMEOUT' for this configuration.
*/
public int getSoTimeout() {
	
	return socketTimeOut ;

}
/**
 * 
 * @return the appropriate WsiTransport for usage with
 * the part specified by pathInfo.
*/
public WsiTransport getTransportForPathInfo(String pathInfo) {
	
	WsiLink link = getLinkForPathInfo(pathInfo) ;
	Vector activeTransports = getTransports() ;
	WsiTransport targetTransport = null ;
	boolean found = false;
	

	// For now return null when no appropriate links can be found
	if ( link == null )
		return null ;

	for ( int i = 0 ; ((i < activeTransports.size()) && (!found)); i++ ) {
		targetTransport = (WsiTransport)activeTransports.elementAt(i) ;
		if (link.getTransportName().equals(targetTransport.getTransportName() ))
			found = true ; }	

	if ( found )
		return targetTransport ;
	else
		return null ;
		

}
/**
 * 
 * @return a Vector containing the 'TRANSPORT' entries for this configuration
*/
public Vector getTransports() {
	if (transports == null) {
		transports = new Vector() ;
	}	

return transports ;

}
/**
 * 
 * @return a boolean containing the 'VERBOSE' setting for this configuration
*/
public boolean getVerbose() {
	
	return verbose ;

}
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
public void handleConfigurationLine(String line) {

	if (( line.length() > 0 ) && ( line.charAt(0) == MSG_COMMENT_CHAR )) {
		return ; }
		
	StringTokenizer tokenizer = new StringTokenizer(line) ;
	String firstToken = null;

	try { 
		firstToken = tokenizer.nextToken() ;
		}		
	catch ( NoSuchElementException e) {}

	// No tokens in current line.  It may have been blank
	if ( firstToken == null ) {
		return ; }	
	
	// Comment line ; skip it
	if (firstToken.charAt(0) == MSG_COMMENT_CHAR) {
		return ; }

	firstToken = firstToken.toUpperCase() ;	
	if (firstToken.equals(MSG_STRING_LINK)) { 		
			processLink( tokenizer ) ;
			return ;}
	
	if (firstToken.equals(MSG_STRING_TRANSPORT)) { 		
			processTransport( tokenizer ) ;
			return ;}

	if (firstToken.equals(MSG_STRING_VERBOSE)) { 		
			processVerbose ( tokenizer ) ;
			return ;}

	if (firstToken.equals(MSG_STRING_LOG)) { 		
			processLog( tokenizer ) ;
			return ;}

	if (firstToken.equals(MSG_STRING_CODEPAGE)) { 		
			processCodePage ( tokenizer ) ;
			return ;}

	if (firstToken.equals(MSG_STRING_ERRORFILE)) { 		
			processErrorFile ( tokenizer ) ;
			return ;}

	logMessage(ERR_STRING_INVALIDKEYWORD,firstToken) ;
	
	return ;

}
/**
 * Read the properties file for this configuration (abtwsi.properties).
 * Then read the VAST Web Connection configuration file and initialize the
 * links and transports for this instance.
 */
public int load() {
	
	String line ;
	FileReader fileReader = null;
	BufferedReader bufferedReader ;

	// Store any properties defined in my properties file
	loadProperties() ;
	try {
		fileReader = new FileReader(getConfigurationFileName()) ;			
	} catch (Throwable exception) {
		logMessage(ERR_STRING_NOCONFIGFILE,getConfigurationFileName()) ;	
		}
		
	if (fileReader != null)
		bufferedReader = new BufferedReader(fileReader) ; 
	else
		return -1 ;
		
	try {
		while(null != (line = bufferedReader.readLine())) 
			handleConfigurationLine(line);                                                  
		}		
	catch ( IOException e) {
	   logMessage(ERR_STRING_READCONFIGFILE,getConfigurationFileName()); }
		   
	try {
			bufferedReader.close() ;
	   }
	catch ( IOException e) {}   
		   	
	return 0 ;
}
/**
 * Read the properties file for this instance
 */
private void loadProperties() {

	InputStream propertyStream = getPropertyFileInputStream() ;
	if (propertyStream != null ) {
		java.util.Properties props = new java.util.Properties() ;
		try {
			String propertyValue ;
			props.load(propertyStream) ; 
			setAbtWsiBaseName(props.getProperty(ABTWSI_BASENAME)) ; 
			propertyValue = props.getProperty(SOCKET_TIMEOUT);
			if (propertyValue == null)
				setSoTimeout(WSI_DEFAULT_SOCKET_TIMEOUT ) ;			
			else {
			try  {
				Integer timeOut = new Integer(propertyValue) ;
				setSoTimeout(timeOut.intValue()) ; }
			catch ( NumberFormatException e ) { 
				logMessage(ERR_STRING_INVALIDSOCKETTIMEOUT) ;
				setSoTimeout(WSI_DEFAULT_SOCKET_TIMEOUT) ;} } }
		catch (IOException e ) {} ;
		} 
	}

	
	
/**
 * Add the passed msgText to the error log
 */
public void logMessage (String msgText) {

	logMessage(msgText, (String[]) null ); }  
/**
 * Add the passed msgText to the error log
 */
public void logMessage (String msgText, String[] subs) {

	String newText = msgText ;
	String fileName = getLogFileName() ;
	int stringIndex ;	


	if ( fileName == null )	
		return ;

	if (( subs != null ) && (subs.length > 0) ) {
		StringBuffer stringBuffer = new StringBuffer() ;
	// Substitute the values in 'subs' for the occurences of %s
	// in the passed message string	
		for (int i = 0 ; i < subs.length ; i++ ) {
			stringIndex = newText.indexOf("%s") ;
			if ( stringIndex != -1 ) {
				stringBuffer.append(newText.substring(0,stringIndex)) ;
				stringBuffer.append(subs[i] ) ;
				newText = newText.substring(stringIndex + 2) ; } }
		stringBuffer.append(newText) ;
		newText = new String(stringBuffer) ; }
		
	newText = (new Date()).toString() + "   " + newText ;
	try { 		
		FileOutputStream outputStream = new FileOutputStream(fileName,true) ;
		PrintWriter printWriter = new PrintWriter (outputStream ) ;
		printWriter.println(newText) ;		
		printWriter.close() ; }
	catch
		( IOException e) { 	System.err.println(e.toString()); }  
	
}
/**
 * Add the passed msgText to the error log
 */
public void logMessage (String msgText, String sub) {

	String[] parm2 = new String[1] ;
	parm2[0] = sub ;
	
	logMessage(msgText, parm2 ); }  
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
private void processCodePage(StringTokenizer tokenizer) {

	try {	
		setCodePage(tokenizer.nextToken()); 		
		}		
	catch ( NoSuchElementException e) {
		logMessage(ERR_STRING_NOPARMS,MSG_STRING_CODEPAGE ) ;}
	}
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
private void processErrorFile(StringTokenizer tokenizer) {

	try {
		setErrorFileName(tokenizer.nextToken()); 		
		}		
	catch ( NoSuchElementException e) {
		logMessage(ERR_STRING_NOPARMS,MSG_STRING_ERRORFILE ) ;}
	
	}
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
private void processLink(StringTokenizer tokenizer) {

	WsiLink newLink = new WsiLink() ;

	try {
		newLink.setPathInfo(tokenizer.nextToken() ) ;
		newLink.setTransportName(tokenizer.nextToken()) ;		
		}		
	catch ( NoSuchElementException e) 
	{	logMessage(ERR_STRING_NOTENOUGHPARMS,MSG_STRING_LINK ); }
	
	getLinks().addElement(newLink);
	}
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
private void processLog(StringTokenizer tokenizer) {
	String tempToken;
	try {
		tempToken = tokenizer.nextToken().toUpperCase();
		if (tempToken.equals(MSG_STRING_ON))
			setLog(true); 
		else
			if (tempToken.equals(MSG_STRING_DEBUG)) {
				setLog(true); 
				setDebug(true); } 
			else
				setLog(false); } 
	catch (NoSuchElementException e) {
		logMessage(ERR_STRING_NOTENOUGHPARMS, MSG_STRING_LOG); }

	// See if a file name was specified for the LOG entry
	try {
		setDebugFileName(tokenizer.nextToken()); } catch (NoSuchElementException e) {
	}
}
/**
 * Parse the passed configuration line and save and create a new Transport based upon
 * the parsed input.
 */
private void processTransport(StringTokenizer tokenizer) {

	WsiTransport newTransport = new WsiTransport() ;
	String portNumberString ;

	try {
		newTransport.setTransportName(tokenizer.nextToken()) ;
		newTransport.setLibraryName(tokenizer.nextToken()) ;
		newTransport.setHostName(tokenizer.nextToken()) ;
		portNumberString = tokenizer.nextToken() ;
		if ( portNumberString != null )
			newTransport.setPortNumber(Integer.parseInt(portNumberString)) ;
		}		
	catch ( NoSuchElementException e) 
	{	logMessage(ERR_STRING_INVALIDLINKTRANSPORT, newTransport.getTransportName() ); }
	
	getTransports().addElement(newTransport) ;

}
/**
 * Parse the passed configuration line and save any information relevant to Servlet
 * routing.
 */
private void processVerbose(StringTokenizer tokenizer) {

	String tempToken ;
	try {
		tempToken = tokenizer.nextToken();
		if (tempToken.equalsIgnoreCase(MSG_STRING_ON)) {
			setVerbose(true); }		
		}		
	catch ( NoSuchElementException e) {
		logMessage(ERR_STRING_NOTENOUGHPARMS,MSG_STRING_VERBOSE);}
	
	}
/**
 * 
 * Clear the 'current' configuration so that it is re-initialized next time it is needed
 */
public static void reset() {
	
	current = null ;

}
/**
 * 
 * Set the value of the 'ABTWSI_BASENAME' for this configuration.
*/
public void setAbtWsiBaseName(String s) {
	
	abtWsiBaseName = s  ;

}
/**
 * 
 * Set the value of the 'CODEPAGE' for this configuration.
*/
public void setCodePage(String s) {
	
	codePage = s  ;

}
/**
 * 
 * Set the value of the 'debug' variable which indicates whether debug messages should be written
 * during execution.
*/
public void setDebug(boolean b) {
	
	debug = b;

}
/**
 * 
 * Set the value of the 'DEBUGFILE' for this configuration.  A debug file is specified
 * in the configuration file when the 'Log' option is set to 'Debug'
*/
public void setDebugFileName(String s) {
	
	debugFileName = s  ;

}
/**
 * 
 * Set the value of the 'ERRORFILE' for this configuration.
*/
public void setErrorFileName(String s) {
	
	errorFileName = s  ;

}
/**
 * 
 * Set the value of the 'LOG' for this configuration.
*/
public void setLog(boolean bool) {
	
	log = bool  ;

}
/**
 * 
 * Set the value of the 'SOCKET_TIMEOUT' for this configuration.
*/
public void setSoTimeout(int i) {
	
	socketTimeOut = i  ;

}
/**
 * 
 * Set the value of the 'VERBOSE' for this configuration.
*/
public void setVerbose(boolean bool) {
	
	verbose = bool  ;

}
}
