/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is TransforMiiX XSLT processor.
 *
 * The Initial Developer of the Original Code is The MITRE Corporation.
 * Portions created by MITRE are Copyright (C) 1999 The MITRE Corporation.
 *
 * Portions created by Keith Visco as a Non MITRE employee,
 * (C) 1999 Keith Visco. All Rights Reserved.
 *
 * Contributor(s):
 * Keith Visco, kvisco@ziplink.net
 *   -- original author.
 *
 */

/**
 * StringFunctionCall
 * A representation of the XPath String funtions
**/

#include "ExprResult.h"
#include "FunctionLib.h"
#include "txAtoms.h"
#include "txIXPathContext.h"
#include "XMLDOMUtils.h"
#include "XMLUtils.h"
#include <math.h>
#include "nsReadableUtils.h"
#include "txAXPathWalkCallback.h"

/**
 * Creates a StringFunctionCall of the given type
**/
StringFunctionCall::StringFunctionCall(StringFunctions aType) : mType(aType)
{
}

/**
 * Evaluates this Expr based on the given context node and processor state
 * @param context the context node for evaluation of this Expr
 * @param ps the ContextState containing the stack information needed
 * for evaluation
 * @return the result of the evaluation
**/
nsresult
StringFunctionCall::evaluate(txIEvalContext* aContext, txAExprResult** aResult)
{
    *aResult = nsnull;

    nsresult rv = NS_OK;
    txListIterator iter(&params);
    switch (mType) {
        case CONCAT:
        {
            if (!requireParams(2, -1, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;
                
            nsRefPtr<StringResult> strRes;
            rv = aContext->recycler()->getStringResult(getter_AddRefs(strRes));
            NS_ENSURE_SUCCESS(rv, rv);

            while (iter.hasNext()) {
                evaluateToString((Expr*)iter.next(), aContext, strRes->mValue);
            }
            *aResult = strRes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
        case CONTAINS:
        {
            if (!requireParams(2, 2, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString arg1, arg2;
            Expr* arg1Expr = (Expr*)iter.next();
            evaluateToString((Expr*)iter.next(), aContext, arg2);
            if (arg2.IsEmpty()) {
                aContext->recycler()->getBoolResult(PR_TRUE, aResult);
            }
            else {
                evaluateToString(arg1Expr, aContext, arg1);
                aContext->recycler()->getBoolResult(arg1.Find(arg2) >= 0,
                                                    aResult);
            }

            return NS_OK;
        }
        case NORMALIZE_SPACE:
        {
            if (!requireParams(0, 1, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString resultStr;
            if (iter.hasNext())
                evaluateToString((Expr*)iter.next(), aContext, resultStr);
            else
                XMLDOMUtils::getNodeValue(aContext->getContextNode(),
                                          resultStr);

            nsRefPtr<StringResult> strRes;
            rv = aContext->recycler()->getStringResult(getter_AddRefs(strRes));
            NS_ENSURE_SUCCESS(rv, rv);

            MBool addSpace = MB_FALSE;
            MBool first = MB_TRUE;
            strRes->mValue.SetCapacity(resultStr.Length());
            PRUnichar c;
            PRUint32 src;
            for (src = 0; src < resultStr.Length(); src++) {
                c = resultStr.CharAt(src);
                if (XMLUtils::isWhitespace(c)) {
                    addSpace = MB_TRUE;
                }
                else {
                    if (addSpace && !first)
                        strRes->mValue.Append(PRUnichar(' '));

                    strRes->mValue.Append(c);
                    addSpace = MB_FALSE;
                    first = MB_FALSE;
                }
            }
            *aResult = strRes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
        case STARTS_WITH:
        {
            if (!requireParams(2, 2, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString arg1, arg2;
            Expr* arg1Expr = (Expr*)iter.next();
            evaluateToString((Expr*)iter.next(), aContext, arg2);
            if (arg2.IsEmpty()) {
                aContext->recycler()->getBoolResult(PR_TRUE, aResult);
            }
            else {
                evaluateToString(arg1Expr, aContext, arg1);
                aContext->recycler()->getBoolResult(
                      StringBeginsWith(arg1, arg2), aResult);
            }

            return NS_OK;
        }
        case STRING_LENGTH:
        {
            if (!requireParams(0, 1, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString resultStr;
            if (iter.hasNext())
                evaluateToString((Expr*)iter.next(), aContext, resultStr);
            else
                XMLDOMUtils::getNodeValue(aContext->getContextNode(),
                                          resultStr);
            rv = aContext->recycler()->getNumberResult(resultStr.Length(),
                                                       aResult);
            NS_ENSURE_SUCCESS(rv, rv);

            return NS_OK;
        }
        case SUBSTRING:
        {
            if (!requireParams(2, 3, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString src;
            double start, end;
            evaluateToString((Expr*)iter.next(), aContext, src);
            start = evaluateToNumber((Expr*)iter.next(), aContext);

            // check for NaN or +/-Inf
            if (Double::isNaN(start) ||
                Double::isInfinite(start) ||
                start >= src.Length() + 0.5) {
                aContext->recycler()->getEmptyStringResult(aResult);

                return NS_OK;
            }

            start = floor(start + 0.5) - 1;
            if (iter.hasNext()) {
                end = start + evaluateToNumber((Expr*)iter.next(),
                                               aContext);
                if (Double::isNaN(end) || end < 0) {
                    aContext->recycler()->getEmptyStringResult(aResult);

                    return NS_OK;
                }
                
                if (end > src.Length())
                    end = src.Length();
                else
                    end = floor(end + 0.5);
            }
            else {
                end = src.Length();
            }

            if (start < 0)
                start = 0;
 
            if (start > end) {
                aContext->recycler()->getEmptyStringResult(aResult);
                
                return NS_OK;
            }

            return aContext->recycler()->getStringResult(
                  Substring(src, (PRUint32)start, (PRUint32)(end - start)),
                  aResult);
        }
        case SUBSTRING_AFTER:
        {
            if (!requireParams(2, 2, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString arg1, arg2;
            evaluateToString((Expr*)iter.next(), aContext, arg1);
            evaluateToString((Expr*)iter.next(), aContext, arg2);
            if (arg2.IsEmpty()) {
                return aContext->recycler()->getStringResult(arg1, aResult);
            }

            PRInt32 idx = arg1.Find(arg2);
            if (idx == kNotFound) {
                aContext->recycler()->getEmptyStringResult(aResult);
                
                return NS_OK;
            }

            PRUint32 len = arg2.Length();
            return aContext->recycler()->getStringResult(
                  Substring(arg1, idx + len, arg1.Length() - (idx + len)),
                  aResult);
        }
        case SUBSTRING_BEFORE:
        {
            if (!requireParams(2, 2, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsAutoString arg1, arg2;
            Expr* arg1Expr = (Expr*)iter.next();
            evaluateToString((Expr*)iter.next(), aContext, arg2);
            if (arg2.IsEmpty()) {
                aContext->recycler()->getEmptyStringResult(aResult);

                return NS_OK;
            }

            evaluateToString(arg1Expr, aContext, arg1);

            PRInt32 idx = arg1.Find(arg2);
            if (idx == kNotFound) {
                aContext->recycler()->getEmptyStringResult(aResult);
                
                return NS_OK;
            }

            return aContext->recycler()->
                getStringResult(Substring(arg1, 0, idx), aResult);
        }
        case TRANSLATE:
        {
            if (!requireParams(3, 3, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsRefPtr<StringResult> strRes;
            rv = aContext->recycler()->getStringResult(getter_AddRefs(strRes));
            NS_ENSURE_SUCCESS(rv, rv);

            nsAutoString src;
            evaluateToString((Expr*)iter.next(), aContext, src);
            strRes->mValue.SetCapacity(src.Length());
            nsAutoString oldChars, newChars;
            evaluateToString((Expr*)iter.next(), aContext, oldChars);
            evaluateToString((Expr*)iter.next(), aContext, newChars);
            PRUint32 i;
            PRInt32 newCharsLength = (PRInt32)newChars.Length();
            for (i = 0; i < src.Length(); i++) {
                PRInt32 idx = oldChars.FindChar(src.CharAt(i));
                if (idx != kNotFound) {
                    if (idx < newCharsLength)
                        strRes->mValue.Append(newChars.CharAt((PRUint32)idx));
                }
                else {
                    strRes->mValue.Append(src.CharAt(i));
                }
            }
            *aResult = strRes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
        case STRING:
        {
            if (!requireParams(0, 1, aContext))
                return NS_ERROR_XPATH_BAD_ARGUMENT_COUNT;

            nsRefPtr<StringResult> strRes;
            rv = aContext->recycler()->getStringResult(getter_AddRefs(strRes));
            NS_ENSURE_SUCCESS(rv, rv);

            if (iter.hasNext())
                evaluateToString((Expr*)iter.next(), aContext, strRes->mValue);
            else
                XMLDOMUtils::getNodeValue(aContext->getContextNode(),
                                          strRes->mValue);
            *aResult = strRes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
    }

    aContext->receiveError(NS_LITERAL_STRING("Internal error"),
                           NS_ERROR_UNEXPECTED);
    return NS_ERROR_UNEXPECTED;
}

nsresult StringFunctionCall::getNameAtom(nsIAtom** aAtom)
{
    switch (mType) {
        case CONCAT:
        {
            *aAtom = txXPathAtoms::concat;
            break;
        }
        case CONTAINS:
        {
            *aAtom = txXPathAtoms::contains;
            break;
        }
        case NORMALIZE_SPACE:
        {
            *aAtom = txXPathAtoms::normalizeSpace;
            break;
        }
        case STARTS_WITH:
        {
            *aAtom = txXPathAtoms::startsWith;
            break;
        }
        case STRING:
        {
            *aAtom = txXPathAtoms::string;
            break;
        }
        case STRING_LENGTH:
        {
            *aAtom = txXPathAtoms::stringLength;
            break;
        }
        case SUBSTRING:
        {
            *aAtom = txXPathAtoms::substring;
            break;
        }
        case SUBSTRING_AFTER:
        {
            *aAtom = txXPathAtoms::substringAfter;
            break;
        }
        case SUBSTRING_BEFORE:
        {
            *aAtom = txXPathAtoms::substringBefore;
            break;
        }
        case TRANSLATE:
        {
            *aAtom = txXPathAtoms::translate;
            break;
        }
        default:
        {
            *aAtom = 0;
            return NS_ERROR_FAILURE;
        }
    }
    NS_ADDREF(*aAtom);
    return NS_OK;
}

nsresult
StringFunctionCall::iterateSubItems(txAXPathWalkCallback* aCallback)
{
    static const Expr::ResultType containsArgs[] = { STRING_RESULT, STRING_RESULT, 0 };
    static const Expr::ResultType normalizeSpaceArgs[] = { STRING_RESULT, 0 };
    static const Expr::ResultType startsWithArgs[] = { STRING_RESULT, STRING_RESULT, 0 };
    static const Expr::ResultType stringArgs[] = { STRING_RESULT, 0 };
    static const Expr::ResultType stringLengthArgs[] = { STRING_RESULT, 0 };
    static const Expr::ResultType substringArgs[] = { STRING_RESULT, NUMBER_RESULT, NUMBER_RESULT, 0 };
    static const Expr::ResultType substringAfterArgs[] = { STRING_RESULT, STRING_RESULT, 0 };
    static const Expr::ResultType substringBeforeArgs[] = { STRING_RESULT, STRING_RESULT, 0 };
    static const Expr::ResultType translateArgs[] = { STRING_RESULT, STRING_RESULT, STRING_RESULT, 0 };

    const Expr::ResultType* types;

    switch (mType) {
        case CONCAT:
        {
            txListIterator iter(&params);
            while (iter.hasNext()) {
                nsAutoPtr<Expr> expr(NS_STATIC_CAST(Expr*, iter.next()));
                nsresult rv = aCallback->walkedExpr(expr, STRING_RESULT);
                iter.replace(expr.forget());
                NS_ENSURE_SUCCESS(rv, rv);
            }
            
            return NS_OK;
        }
        case CONTAINS:
        {
            types = containsArgs;
            break;
        }
        case NORMALIZE_SPACE:
        {
            types = normalizeSpaceArgs;
            break;
        }
        case STARTS_WITH:
        {
            types = startsWithArgs;
            break;
        }
        case STRING:
        {
            types = stringArgs;
            break;
        }
        case STRING_LENGTH:
        {
            types = stringLengthArgs;
            break;
        }
        case SUBSTRING:
        {
            types = substringArgs;
            break;
        }
        case SUBSTRING_AFTER:
        {
            types = substringAfterArgs;
            break;
        }
        case SUBSTRING_BEFORE:
        {
            types = substringBeforeArgs;
            break;
        }
        case TRANSLATE:
        {
            types = translateArgs;
            break;
        }
    }

    return iterateArgs(aCallback, types);
}

Expr::ResultType
StringFunctionCall::getReturnType()
{
    switch (mType) {
        case CONCAT:
        case NORMALIZE_SPACE:
        case STRING:
        case SUBSTRING:
        case SUBSTRING_AFTER:
        case SUBSTRING_BEFORE:
        case TRANSLATE:
        {
            return STRING_RESULT;
        }
        case CONTAINS:
        case STARTS_WITH:
        {
            return BOOLEAN_RESULT;
        }
        case STRING_LENGTH:
        {
            return NUMBER_RESULT;
        }
    }

    NS_NOTREACHED("how'd we get here?");
    return ANY_RESULT;
}

Expr::ContextSensitivity
StringFunctionCall::getContextSensitivity()
{
    switch (mType) {
        case CONCAT:
        case CONTAINS:
        case STARTS_WITH:
        case SUBSTRING:
        case SUBSTRING_AFTER:
        case SUBSTRING_BEFORE:
        case TRANSLATE:
        {
            return getArgsContextSensitivity();
        }
        case NORMALIZE_SPACE:
        case STRING:
        case STRING_LENGTH:
        {
            if (params.isEmpty()) {
                return NODE_CONTEXT;
            }
            return getArgsContextSensitivity();
        }
    }

    NS_NOTREACHED("how'd we get here?");
    return UNKNOWN_CONTEXT;
}
