/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2002  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2002  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <libbeep/xentry.h>
#include "mpg123.h"
#include "gdk/gdkkeysyms.h"

#include "mp3.xpm"

static GtkWidget *window = NULL;
static GtkWidget *filename_entry, *id3_frame;
static GtkWidget *title_entry, *artist_entry, *album_entry, *year_entry,
    *tracknum_entry, *comment_entry;
static GtkWidget *genre_combo;
static GtkWidget *mpeg_level, *mpeg_bitrate, *mpeg_samplerate, *mpeg_flags,
    *mpeg_error, *mpeg_copy, *mpeg_orig, *mpeg_emph, *mpeg_frames,
    *mpeg_filesize;
static GtkWidget *mpeg_level_val, *mpeg_bitrate_val, *mpeg_samplerate_val,
    *mpeg_error_val, *mpeg_copy_val, *mpeg_orig_val, *mpeg_emph_val,
    *mpeg_frames_val, *mpeg_filesize_val;


GtkWidget *vbox, *hbox, *left_vbox, *table;
GtkWidget *mpeg_frame, *mpeg_box;
GtkWidget *label, *filename_vbox;
GtkWidget *bbox;
GtkWidget *remove_id3, *cancel, *save;
GtkWidget *boxx;
// GtkWidget *revert;


FILE *fh;
struct id3v1tag_t tag;
const gchar *emphasis[4];
const gchar *bool_label[2];


static GList *genre_list = NULL;
static gchar *current_filename = NULL;

extern gchar *mpg123_filename;
extern gint mpg123_bitrate, mpg123_frequency, mpg123_layer, mpg123_lsf,
    mpg123_mode;
extern gboolean mpg123_stereo, mpg123_mpeg25;

long int info_rate;

void fill_entries(GtkWidget * w, gpointer data);

#define MAX_STR_LEN 100

static void set_entry_tag(GtkEntry * entry, gchar * tag, gint length)
{
    gint stripped_len;
    gchar *text, *tmp;

    stripped_len = mpg123_strip_spaces(tag, length);
    text = g_strdup_printf("%-*.*s", stripped_len, stripped_len, tag);
    tmp = g_locale_to_utf8(text, -1, NULL, NULL, NULL);
    if (tmp == NULL)
	tmp = g_strdup("");
    gtk_entry_set_text(entry, tmp);
    g_free(text);
}

static void get_entry_tag(GtkEntry * entry, gchar * tag, gint length)
{
    gchar *text;

    text = (gchar *) gtk_entry_get_text(entry);
    text = g_locale_from_utf8(text, -1, NULL, NULL, NULL);
    memset(tag, ' ', length);
    memcpy(tag, text, strlen(text) > length ? length : strlen(text));

}

static gint find_genre_id(gchar * text)
{
    gint i;

    for (i = 0; i < GENRE_MAX; i++) {
	if (!strcmp(mpg123_id3_genres[i], text))
	    return i;
    }
    if (text[0] == '\0')
	return 0xff;
    return 0;
}

static gint genre_comp_func(gconstpointer a, gconstpointer b)
{
    return strcasecmp(a, b);
}

static void press_save(GtkWidget * w, gpointer data)
{

    gtk_button_clicked(GTK_BUTTON(save));

}

static gboolean fileinfo_keypress_cb(GtkWidget * widget,
				     GdkEventKey * event,
				     gpointer userdata)
{
    if (!event)
	return FALSE;

    switch (event->keyval) {
    case GDK_Escape:
	gtk_widget_destroy(window);
	break;
    default:
	return FALSE;
    }

    return TRUE;
}


static void save_cb(GtkWidget * w, gpointer data)
{
    gint fd;
    char *msg = NULL;

    if (!strncasecmp(current_filename, "http://", 7))
	return;

    if ((fd = open(current_filename, O_RDWR)) != -1) {
	int tracknum;

	lseek(fd, -128, SEEK_END);
	read(fd, &tag, sizeof(struct id3v1tag_t));

	if (!strncmp(tag.tag, "TAG", 3))
	    lseek(fd, -128, SEEK_END);
	else
	    lseek(fd, 0, SEEK_END);
	tag.tag[0] = 'T';
	tag.tag[1] = 'A';
	tag.tag[2] = 'G';
	get_entry_tag(GTK_ENTRY(title_entry), tag.title, 30);
	get_entry_tag(GTK_ENTRY(artist_entry), tag.artist, 30);
	get_entry_tag(GTK_ENTRY(album_entry), tag.album, 30);
	get_entry_tag(GTK_ENTRY(year_entry), tag.year, 4);
	tracknum = atoi(gtk_entry_get_text(GTK_ENTRY(tracknum_entry)));
	if (tracknum > 0) {
	    get_entry_tag(GTK_ENTRY(comment_entry), tag.u.v1_1.comment,
			  28);
	    tag.u.v1_1.__zero = 0;
	    tag.u.v1_1.track_number = MIN(tracknum, 255);
	} else
	    get_entry_tag(GTK_ENTRY(comment_entry), tag.u.v1_0.comment,
			  30);
	tag.genre = find_genre_id((gchar *)
				  gtk_entry_get_text(GTK_ENTRY
						     (GTK_COMBO
						      (genre_combo)->
						      entry)));
	if (write(fd, &tag, sizeof(tag)) != sizeof(tag))
	    msg = g_strdup_printf(_("%s\nUnable to write to file: %s"),
				  _("Couldn't write tag!"),
				  strerror(errno));
	close(fd);
    } else
	msg = g_strdup_printf(_("%s\nUnable to open file: %s"),
			      _("Couldn't write tag!"), strerror(errno));
    if (msg) {
	GtkWidget *mwin = xmms_show_message(_("File Info"), msg, _("Ok"),
					    FALSE, NULL, NULL);
	gtk_window_set_transient_for(GTK_WINDOW(mwin), GTK_WINDOW(window));
	g_free(msg);
    } else {
	gtk_widget_set_sensitive(GTK_WIDGET(data), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(w), FALSE);
//              gtk_widget_set_sensitive(GTK_WIDGET(revert),FALSE);
    }
}

static void label_set_text(GtkWidget * label, char *str, ...)
{
    va_list args;
    gchar tempstr[MAX_STR_LEN];

    va_start(args, str);
    g_vsnprintf(tempstr, MAX_STR_LEN, str, args);
    va_end(args);

    gtk_label_set_text(GTK_LABEL(label), tempstr);
}

static void remove_id3_cb(GtkWidget * w, gpointer data)
{
    gint fd, len;
    struct id3v1tag_t tag;
    char *msg = NULL;

    if (!strncasecmp(current_filename, "http://", 7))
	return;

    if ((fd = open(current_filename, O_RDWR)) != -1) {
	len = lseek(fd, -128, SEEK_END);
	read(fd, &tag, sizeof(struct id3v1tag_t));

	if (!strncmp(tag.tag, "TAG", 3)) {
	    if (ftruncate(fd, len))
		msg = g_strdup_printf(_("%s\n"
					"Unable to truncate file: %s"),
				      _("Couldn't remove tag!"),
				      strerror(errno));
	} else
	    msg = strdup(_("No tag to remove!"));
	close(fd);
    } else
	msg = g_strdup_printf(_("%s\nUnable to open file: %s"),
			      _("Couldn't remove tag!"), strerror(errno));
    if (msg) {
	GtkWidget *mwin = xmms_show_message(_("File Info"), msg, _("Ok"),
					    FALSE, NULL, NULL);
	gtk_window_set_transient_for(GTK_WINDOW(mwin), GTK_WINDOW(window));
	g_free(msg);
    } else {

	gtk_entry_set_text(GTK_ENTRY(title_entry), "");
	gtk_entry_set_text(GTK_ENTRY(artist_entry), "");
	gtk_entry_set_text(GTK_ENTRY(album_entry), "");
	gtk_entry_set_text(GTK_ENTRY(comment_entry), "");
	gtk_entry_set_text(GTK_ENTRY(year_entry), "");
	gtk_entry_set_text(GTK_ENTRY(album_entry), "");
	gtk_entry_set_text(GTK_ENTRY(tracknum_entry), "");
	gtk_widget_set_sensitive(GTK_WIDGET(w), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(data), FALSE);
//              gtk_widget_set_sensitive(GTK_WIDGET(revert),FALSE);

    }

}

static void set_mpeg_level_label(gboolean mpeg25, gint lsf, gint layer)
{
    if (mpeg25)
	label_set_text(mpeg_level_val, "MPEG-2.5 Layer %d", layer);
    else
	label_set_text(mpeg_level_val, "MPEG-%d Layer %d", lsf + 1, layer);
}

static gchar *channel_mode_name(int mode)
{
    static const gchar *channel_mode[] =
	{ N_("Stereo"), N_("Joint stereo"),
	N_("Dual channel"), N_("Single channel")
    };
    if (mode < 0 || mode > 3)
	return "";
    return gettext(channel_mode[mode]);
}

static void file_info_http(char *filename)
{
    gtk_widget_set_sensitive(id3_frame, FALSE);
    if (mpg123_filename && !strcmp(filename, mpg123_filename) &&
	mpg123_bitrate != 0) {
	set_mpeg_level_label(mpg123_mpeg25, mpg123_lsf, mpg123_layer);
	label_set_text(mpeg_bitrate_val, _("%d KBit/s"), mpg123_bitrate);
	label_set_text(mpeg_samplerate_val, _("%ld Hz"), mpg123_frequency);
	label_set_text(mpeg_flags, "%s", channel_mode_name(mpg123_mode));
    }
}

static void change_buttons(GtkObject * object)
{
    gtk_widget_set_sensitive(GTK_WIDGET(object), TRUE);
//      gtk_widget_set_sensitive(GTK_WIDGET(revert),TRUE);
}

void mpg123_file_info_box(char *filename)
{
    gint i;
    gchar *title;

    emphasis[0] = _("None");
    emphasis[1] = _("50/15 ms");
    emphasis[2] = "";
    emphasis[3] = _("CCIT J.17");
    bool_label[0] = _("No");
    bool_label[1] = _("Yes");

    if (!window) {

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	g_signal_connect(G_OBJECT(window), "destroy",
			 G_CALLBACK(gtk_widget_destroyed), &window);
	gtk_container_set_border_width(GTK_CONTAINER(window), 10);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(window), vbox);


	filename_vbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), filename_vbox, FALSE, TRUE, 0);

	GtkWidget *pixmapwid;
	GdkPixbuf *pixbuf;
	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)
					      gnome_mime_audio_xpm);
	pixmapwid = gtk_image_new_from_pixbuf(pixbuf);
	g_object_unref(pixbuf);
	gtk_misc_set_alignment(GTK_MISC(pixmapwid), 0, 0);
	gtk_box_pack_start(GTK_BOX(filename_vbox), pixmapwid, FALSE, FALSE,
			   0);

	label = gtk_label_new(NULL);

	PangoAttrList *attrs;
	PangoAttribute *attr;

	attrs = pango_attr_list_new();

	attr = pango_attr_weight_new(PANGO_WEIGHT_BOLD);
	attr->start_index = 0;
	attr->end_index = -1;
	pango_attr_list_insert(attrs, attr);

	gtk_label_set_attributes(GTK_LABEL(label), attrs);


	gtk_label_set_text(GTK_LABEL(label), "Name:");



	gtk_box_pack_start(GTK_BOX(filename_vbox), label, FALSE, FALSE, 0);
	filename_entry = xmms_entry_new();
	gtk_editable_set_editable(GTK_EDITABLE(filename_entry), FALSE);
	gtk_misc_set_alignment(GTK_MISC(filename_entry), 0, 0);
	gtk_box_pack_start(GTK_BOX(filename_vbox), filename_entry, TRUE,
			   TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);

	left_vbox = gtk_table_new(2, 4, FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), left_vbox, FALSE, FALSE, 0);




	// MPEG-Info window

	mpeg_frame = gtk_frame_new(_(" MPEG Info "));
	gtk_table_attach(GTK_TABLE(left_vbox), mpeg_frame, 0, 2, 0, 1,
			 GTK_FILL, GTK_FILL, 0, 4);

	mpeg_box = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(mpeg_frame), mpeg_box);
	gtk_container_set_border_width(GTK_CONTAINER(mpeg_box), 10);
	gtk_box_set_spacing(GTK_BOX(mpeg_box), 0);

	//// MPEG Layer Info

	// FIXME: Obvious...
	GtkWidget *test_table = gtk_table_new(2, 10, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(test_table), 0);
	gtk_container_add(GTK_CONTAINER(mpeg_box), test_table);

	mpeg_level = gtk_label_new(_("MPEG Level:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_level), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_level), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_level), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_level, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_level_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_level_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_level_val), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_level_val, 1, 2, 0, 1,
			 GTK_FILL, GTK_FILL, 10, 2);


	mpeg_bitrate = gtk_label_new(_("Bitrate:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_bitrate), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_bitrate), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_bitrate), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_bitrate, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_bitrate_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_bitrate_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_bitrate_val),
			      GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_bitrate_val, 1, 2, 1,
			 2, GTK_FILL, GTK_FILL, 10, 2);

	mpeg_samplerate = gtk_label_new(_("Samplerate:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_samplerate), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_samplerate),
			      GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_samplerate), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_samplerate, 0, 1, 2,
			 3, GTK_FILL, GTK_FILL, 5, 2);

	mpeg_samplerate_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_samplerate_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_samplerate_val),
			      GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_samplerate_val, 1, 2,
			 2, 3, GTK_FILL, GTK_FILL, 10, 2);

	mpeg_frames = gtk_label_new(_("Frames:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_frames), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_frames), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_frames), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_frames, 0, 1, 3, 4,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_frames_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_frames_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_frames_val),
			      GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_frames_val, 1, 2, 3,
			 4, GTK_FILL, GTK_FILL, 10, 2);

	mpeg_filesize = gtk_label_new(_("Filesize:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_filesize), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_filesize), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_filesize), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_filesize, 0, 1, 4, 5,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_filesize_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_filesize_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_filesize_val),
			      GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_filesize_val, 1, 2, 4,
			 5, GTK_FILL, GTK_FILL, 10, 2);

	GtkWidget *urk, *blark;

	urk = gtk_label_new("");
	blark = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(urk), 1, 0.5);
	gtk_misc_set_alignment(GTK_MISC(blark), 0, 0);

	gtk_table_attach(GTK_TABLE(test_table), urk, 0, 1, 5, 6, GTK_FILL,
			 GTK_FILL, 5, 5);
	gtk_table_attach(GTK_TABLE(test_table), blark, 1, 2, 5, 6,
			 GTK_FILL, GTK_FILL, 10, 5);

	mpeg_error = gtk_label_new(_("Error Protection:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_error), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_error), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_error), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_error, 0, 1, 6, 7,
			 GTK_FILL, GTK_FILL, 5, 0);

	mpeg_error_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_error_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_error_val), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_error_val, 1, 2, 6, 7,
			 GTK_FILL, GTK_FILL, 10, 2);

	mpeg_copy = gtk_label_new(_("Copyright:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_copy), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_copy), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_copy), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_copy, 0, 1, 7, 8,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_copy_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_copy_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_copy_val), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_copy_val, 1, 2, 7, 8,
			 GTK_FILL, GTK_FILL, 10, 2);

	mpeg_orig = gtk_label_new(_("Original:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_orig), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_orig), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_orig), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_orig, 0, 1, 8, 9,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_orig_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_orig_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_orig_val), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_orig_val, 1, 2, 8, 9,
			 GTK_FILL, GTK_FILL, 10, 2);

	mpeg_emph = gtk_label_new(_("Emphasis:"));
	gtk_misc_set_alignment(GTK_MISC(mpeg_emph), 1, 0.5);
	gtk_label_set_justify(GTK_LABEL(mpeg_emph), GTK_JUSTIFY_RIGHT);
	gtk_label_set_attributes(GTK_LABEL(mpeg_emph), attrs);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_emph, 0, 1, 9, 10,
			 GTK_FILL, GTK_FILL, 5, 2);

	mpeg_emph_val = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(mpeg_emph_val), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mpeg_emph_val), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(test_table), mpeg_emph_val, 1, 2, 9, 10,
			 GTK_FILL, GTK_FILL, 10, 2);


	id3_frame = gtk_frame_new(_(" ID3 Tag "));
	gtk_table_attach(GTK_TABLE(left_vbox), id3_frame, 2, 4, 0, 1,
			 GTK_FILL, GTK_FILL, 0, 4);

	table = gtk_table_new(7, 5, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_container_add(GTK_CONTAINER(id3_frame), table);

	label = gtk_label_new(_("Title:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL,
			 GTK_FILL, 5, 5);

	title_entry = gtk_entry_new_with_max_length(30);
	gtk_table_attach(GTK_TABLE(table), title_entry, 1, 6, 0, 1,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Artist:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL,
			 GTK_FILL, 5, 5);

	artist_entry = gtk_entry_new_with_max_length(30);
	gtk_table_attach(GTK_TABLE(table), artist_entry, 1, 6, 1, 2,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Album:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL,
			 GTK_FILL, 5, 5);

	album_entry = gtk_entry_new_with_max_length(30);
	gtk_table_attach(GTK_TABLE(table), album_entry, 1, 6, 2, 3,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Comment:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL,
			 GTK_FILL, 5, 5);

	comment_entry = gtk_entry_new_with_max_length(30);
	gtk_table_attach(GTK_TABLE(table), comment_entry, 1, 6, 3, 4,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Year:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, GTK_FILL,
			 GTK_FILL, 5, 5);

	year_entry = gtk_entry_new_with_max_length(4);
	gtk_widget_set_usize(year_entry, 40, -1);
	gtk_table_attach(GTK_TABLE(table), year_entry, 1, 2, 4, 5,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Track number:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 2, 3, 4, 5, GTK_FILL,
			 GTK_FILL, 5, 5);

	tracknum_entry = gtk_entry_new_with_max_length(3);
	gtk_widget_set_usize(tracknum_entry, 40, -1);
	gtk_table_attach(GTK_TABLE(table), tracknum_entry, 3, 4, 4, 5,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	label = gtk_label_new(_("Genre:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_label_set_attributes(GTK_LABEL(label), attrs);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, GTK_FILL,
			 GTK_FILL, 5, 5);

	pango_attr_list_unref(attrs);

	genre_combo = gtk_combo_new();
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(genre_combo)->entry),
			       FALSE);
	if (!genre_list) {
	    for (i = 0; i < GENRE_MAX; i++)
		genre_list =
		    g_list_prepend(genre_list,
				   (char *) mpg123_id3_genres[i]);
	    genre_list = g_list_prepend(genre_list, "");
	    genre_list = g_list_sort(genre_list, genre_comp_func);
	}
	gtk_combo_set_popdown_strings(GTK_COMBO(genre_combo), genre_list);

	gtk_table_attach(GTK_TABLE(table), genre_combo, 1, 6, 5, 6,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);

	boxx = gtk_hbutton_box_new();
	gtk_hbutton_box_set_layout_default(GTK_BUTTONBOX_SPREAD);

	remove_id3 = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	gtk_container_add(GTK_CONTAINER(boxx), remove_id3);

//              revert = gtk_button_new_from_stock(GTK_STOCK_REVERT_TO_SAVED);
//              gtk_container_add(GTK_CONTAINER(boxx),revert);

	save = gtk_button_new_from_stock(GTK_STOCK_SAVE);
	gtk_container_add(GTK_CONTAINER(boxx), save);

	g_signal_connect(G_OBJECT(remove_id3), "clicked",
			 G_CALLBACK(remove_id3_cb), save);
	g_signal_connect(G_OBJECT(save), "clicked", G_CALLBACK(save_cb),
			 remove_id3);
//              g_signal_connect(G_OBJECT(revert), "clicked", G_CALLBACK(fill_entries), NULL);


	gtk_table_attach(GTK_TABLE(table), boxx, 0, 5, 6, 7, GTK_FILL, 0,
			 0, 8);

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);

	cancel = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	g_signal_connect_swapped(G_OBJECT(cancel), "clicked",
				 G_CALLBACK(gtk_widget_destroy),
				 G_OBJECT(window));
	GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), cancel, FALSE, FALSE, 0);
	gtk_widget_grab_default(cancel);

	gtk_table_attach(GTK_TABLE(left_vbox), bbox, 0, 4, 1, 2, GTK_FILL,
			 0, 0, 8);

	gtk_table_set_col_spacing(GTK_TABLE(left_vbox), 1, 10);


	g_signal_connect_swapped(G_OBJECT(title_entry), "changed",
				 G_CALLBACK(change_buttons), save);
	g_signal_connect_swapped(G_OBJECT(artist_entry), "changed",
				 G_CALLBACK(change_buttons), save);
	g_signal_connect_swapped(G_OBJECT(album_entry), "changed",
				 G_CALLBACK(change_buttons), save);
	g_signal_connect_swapped(G_OBJECT(year_entry), "changed",
				 G_CALLBACK(change_buttons), save);
	g_signal_connect_swapped(G_OBJECT(comment_entry), "changed",
				 G_CALLBACK(change_buttons), save);
	g_signal_connect_swapped(G_OBJECT(tracknum_entry), "changed",
				 G_CALLBACK(change_buttons), save);


	g_signal_connect(G_OBJECT(title_entry), "activate",
			 G_CALLBACK(press_save), NULL);
	g_signal_connect(G_OBJECT(artist_entry), "activate",
			 G_CALLBACK(press_save), NULL);
	g_signal_connect(G_OBJECT(album_entry), "activate",
			 G_CALLBACK(press_save), NULL);
	g_signal_connect(G_OBJECT(year_entry), "activate",
			 G_CALLBACK(press_save), NULL);
	g_signal_connect(G_OBJECT(comment_entry), "activate",
			 G_CALLBACK(press_save), NULL);
	g_signal_connect(G_OBJECT(tracknum_entry), "activate",
			 G_CALLBACK(press_save), NULL);

	g_signal_connect(G_OBJECT(window), "key_press_event",
			 G_CALLBACK(fileinfo_keypress_cb), NULL);


    }

    if (current_filename)
	g_free(current_filename);
    current_filename = g_strdup(filename);

    title = g_strdup_printf(_("File Info - %s"), g_basename(filename));
    gtk_window_set_title(GTK_WINDOW(window), title);
    g_free(title);

    gtk_entry_set_text(GTK_ENTRY(filename_entry),
		       g_locale_to_utf8(filename, -1, NULL, NULL, NULL));
    gtk_editable_set_position(GTK_EDITABLE(filename_entry), -1);

/*	title = g_strdup(g_basename(filename));
	if ((tmp = strrchr(title, '.')) != NULL)
		*tmp = '\0';
	gtk_entry_set_text(GTK_ENTRY(title_entry), title);
	g_free(title);
*/

    gtk_entry_set_text(GTK_ENTRY(artist_entry), "");
    gtk_entry_set_text(GTK_ENTRY(album_entry), "");
    gtk_entry_set_text(GTK_ENTRY(year_entry), "");
    gtk_entry_set_text(GTK_ENTRY(tracknum_entry), "");
    gtk_entry_set_text(GTK_ENTRY(comment_entry), "");
    gtk_list_select_item(GTK_LIST(GTK_COMBO(genre_combo)->list),
			 g_list_index(genre_list, ""));

    gtk_label_set_text(GTK_LABEL(mpeg_level), "MPEG Level:");
    gtk_label_set_text(GTK_LABEL(mpeg_level_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_bitrate), "Bitrate:");
    gtk_label_set_text(GTK_LABEL(mpeg_bitrate_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_samplerate), "Samplerate:");
    gtk_label_set_text(GTK_LABEL(mpeg_samplerate_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_error), "Error Protection:");
    gtk_label_set_text(GTK_LABEL(mpeg_error_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_copy), "Copyright:");
    gtk_label_set_text(GTK_LABEL(mpeg_copy_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_orig), "Original:");
    gtk_label_set_text(GTK_LABEL(mpeg_orig_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_emph), "Emphasis:");
    gtk_label_set_text(GTK_LABEL(mpeg_emph_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_frames), "Frames:");
    gtk_label_set_text(GTK_LABEL(mpeg_frames_val), "N/A");

    gtk_label_set_text(GTK_LABEL(mpeg_filesize), "Filesize:");
    gtk_label_set_text(GTK_LABEL(mpeg_filesize_val), "N/A");


    if (!strncasecmp(filename, "http://", 7)) {
	file_info_http(filename);
	return;
    }

    struct stat mstat;
    stat(filename, &mstat);
    gtk_widget_set_sensitive(id3_frame,
			     ((mstat.st_mode & S_IWRITE) == S_IWRITE));
    /* FIXME: Check for UID/GID too? */

    fill_entries(NULL, NULL);

    gtk_widget_set_sensitive(GTK_WIDGET(save), FALSE);
//      gtk_widget_set_sensitive(GTK_WIDGET(revert),FALSE);
    gtk_widget_show_all(window);
}

void fill_entries(GtkWidget * w, gpointer data)
{

    g_message("Refilling entries...");

    if ((fh = fopen(current_filename, "rb")) != NULL) {
	guint32 head;
	unsigned char tmp[4];
	struct frame frm;
	gboolean id3_found = FALSE;

	fseek(fh, -sizeof(tag), SEEK_END);
	if (fread(&tag, 1, sizeof(tag), fh) == sizeof(tag)) {
	    if (!strncmp(tag.tag, "TAG", 3)) {
		id3_found = TRUE;
		set_entry_tag(GTK_ENTRY(title_entry), tag.title, 30);
		set_entry_tag(GTK_ENTRY(artist_entry), tag.artist, 30);
		set_entry_tag(GTK_ENTRY(album_entry), tag.album, 30);
		set_entry_tag(GTK_ENTRY(year_entry), tag.year, 4);
		/* Check for v1.1 tags */
		if (tag.u.v1_1.__zero == 0) {
		    gchar *temp =
			g_strdup_printf("%d", tag.u.v1_1.track_number);
		    set_entry_tag(GTK_ENTRY(comment_entry),
				  tag.u.v1_1.comment, 28);
		    gtk_entry_set_text(GTK_ENTRY(tracknum_entry), temp);
		    g_free(temp);
		} else {
		    set_entry_tag(GTK_ENTRY(comment_entry),
				  tag.u.v1_0.comment, 30);
		    gtk_entry_set_text(GTK_ENTRY(tracknum_entry), "");
		}

		gtk_list_select_item(GTK_LIST
				     (GTK_COMBO(genre_combo)->list),
				     g_list_index(genre_list, (char *)
						  mpg123_id3_genres[tag.
								    genre]));
		gtk_widget_set_sensitive(GTK_WIDGET(remove_id3), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(save), FALSE);
//                              gtk_widget_set_sensitive(GTK_WIDGET(revert),FALSE);
	    } else {
		gtk_entry_set_text(GTK_ENTRY(title_entry), "");
		gtk_entry_set_text(GTK_ENTRY(artist_entry), "");
		gtk_entry_set_text(GTK_ENTRY(album_entry), "");
		gtk_entry_set_text(GTK_ENTRY(comment_entry), "");
		gtk_entry_set_text(GTK_ENTRY(year_entry), "");
		gtk_entry_set_text(GTK_ENTRY(album_entry), "");
		gtk_entry_set_text(GTK_ENTRY(tracknum_entry), "");
		gtk_widget_set_sensitive(GTK_WIDGET(remove_id3), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(save), FALSE);
//                              gtk_widget_set_sensitive(GTK_WIDGET(revert),FALSE);
	    }
	}
	rewind(fh);
	if (fread(tmp, 1, 4, fh) != 4) {
	    fclose(fh);
	    return;
	}
	head =
	    ((guint32) tmp[0] << 24) | ((guint32) tmp[1] << 16) |
	    ((guint32) tmp[2] << 8) | (guint32) tmp[3];
	while (!mpg123_head_check(head)) {
	    head <<= 8;
	    if (fread(tmp, 1, 1, fh) != 1) {
		fclose(fh);
		return;
	    }
	    head |= tmp[0];
	}
	if (mpg123_decode_header(&frm, head)) {
	    guchar *buf;
	    gdouble tpf;
	    gint pos;
	    xing_header_t xing_header;
	    guint32 num_frames;

	    buf = g_malloc(frm.framesize + 4);
	    fseek(fh, -4, SEEK_CUR);
	    fread(buf, 1, frm.framesize + 4, fh);
	    tpf = mpg123_compute_tpf(&frm);
	    set_mpeg_level_label(frm.mpeg25, frm.lsf, frm.lay);
	    pos = ftell(fh);
	    fseek(fh, 0, SEEK_END);
	    if (mpg123_get_xing_header(&xing_header, buf)) {
		num_frames = xing_header.frames;
		label_set_text(mpeg_bitrate_val,
			       _("Variable,\navg. bitrate: %d KBit/s"),
			       (gint) ((xing_header.bytes * 8) /
				       (tpf * xing_header.frames * 1000)));
	    } else {
		num_frames =
		    ((ftell(fh) - pos -
		      (id3_found ? 128 : 0)) / mpg123_compute_bpf(&frm)) +
		    1;
		label_set_text(mpeg_bitrate_val, _("%d KBit/s"),
			       tabsel_123[frm.lsf][frm.lay -
						   1][frm.bitrate_index]);
	    }
	    label_set_text(mpeg_samplerate_val, _("%ld Hz"),
			   mpg123_freqs[frm.sampling_frequency]);
	    label_set_text(mpeg_error_val, _("%s"),
			   bool_label[frm.error_protection]);
	    label_set_text(mpeg_copy_val, _("%s"),
			   bool_label[frm.copyright]);
	    label_set_text(mpeg_orig_val, _("%s"),
			   bool_label[frm.original]);
	    label_set_text(mpeg_emph_val, _("%s"), emphasis[frm.emphasis]);
	    label_set_text(mpeg_frames_val, _("%d"), num_frames);
	    label_set_text(mpeg_filesize_val, _("%lu Bytes"), ftell(fh));
	    g_free(buf);
	}
	fclose(fh);
    }
}
