/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "GenericRequest.h"

#include "ClientCache.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "WriteBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Constructors and destructors.
//

GenericRequestStore::GenericRequestStore(Compressor *compressor, Decompressor *decompressor)

  : MessageStore(compressor, decompressor)
{
  enableCache    = GENERICREQUEST_ENABLE_CACHE;
  enableData     = GENERICREQUEST_ENABLE_DATA;
  enableSplit    = GENERICREQUEST_ENABLE_SPLIT;
  enableCompress = GENERICREQUEST_ENABLE_COMPRESS;

  if (control -> isProtoStep3() == 1)
  {
    dataLimit  = GENERICREQUEST_DATA_LIMIT_PROTO_3;
    dataOffset = GENERICREQUEST_DATA_OFFSET_PROTO_3;
  }
  else
  {
    dataLimit  = GENERICREQUEST_DATA_LIMIT_PROTO_1;
    dataOffset = GENERICREQUEST_DATA_OFFSET_PROTO_1;
  }

  cacheSlots          = GENERICREQUEST_CACHE_SLOTS;
  cacheThreshold      = GENERICREQUEST_CACHE_THRESHOLD;
  cacheLowerThreshold = GENERICREQUEST_CACHE_LOWER_THRESHOLD;

  messages_ -> resize(cacheSlots);

  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    *i = NULL;
  }

  temporary_ = NULL;
}

GenericRequestStore::~GenericRequestStore()
{
  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    destroy(*i);
  }

  destroy(temporary_);
}

//
// Here are the methods to handle messages' content.
//

int GenericRequestStore::encodeIdentity(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                            const unsigned int size, int bigEndian,
                                                ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Encoding full message identity.\n" << logofs_flush;
  #endif

  encodeBuffer.encodeValue(size >> 2, 16, 10);

  encodeBuffer.encodeCachedValue(*(buffer + 1), 8,
                     clientCache -> genericRequestCharCache);

  for (unsigned int i = 0; i < 8 && (i * 2 + 4) < size; i++)
  {
    encodeBuffer.encodeCachedValue(GetUINT(buffer + (i * 2) + 4, bigEndian), 16,
                       *clientCache -> genericRequestIntCache[i]);
  }

  #ifdef DEBUG
  *logofs << name() << ": Encoded full message identity.\n" << logofs_flush;
  #endif

  return 1;
}

int GenericRequestStore::decodeIdentity(DecodeBuffer &decodeBuffer, unsigned char *&buffer,
                                            unsigned int &size, int bigEndian, WriteBuffer *writeBuffer,
                                                ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Decoding full message identity.\n" << logofs_flush;
  #endif

  decodeBuffer.decodeValue(size, 16, 10);

  size <<= 2;

  buffer = writeBuffer -> addMessage(size);

  decodeBuffer.decodeCachedValue(*(buffer + 1), 8,
                     clientCache -> genericRequestCharCache);

  unsigned int value;

  for (unsigned int i = 0; i < 8 && (i * 2 + 4) < size; i++)
  {
    decodeBuffer.decodeCachedValue(value, 16,
                       *clientCache -> genericRequestIntCache[i]);

    PutUINT(value, buffer + 4 + (i * 2), bigEndian);
  }

  #ifdef DEBUG
  *logofs << name() << ": Decoded full message identity.\n" << logofs_flush;
  #endif

  return 1;
}

int GenericRequestStore::parseIdentity(Message *message, const unsigned char *buffer,
                                           unsigned int size, int bigEndian) const
{
  GenericRequestMessage *genericRequest = (GenericRequestMessage *) message;

  genericRequest -> byte_data = *(buffer + 1);

  for (unsigned int i = 0; i < 8; i++)
  {
    if ((i * 2 + 4) < size)
    {
      genericRequest -> short_data[i] = GetUINT(buffer + i * 2 + 4, bigEndian); 

      //
      // Clear the field in incoming message
      // so it doesn't disturb the checksum.
      //

      if (control -> isProtoStep3() == 0)
      {
        PutUINT(0, ((unsigned char *) buffer) + i * 2 + 4, bigEndian);
      }

      #ifdef DEBUG
      *logofs << name() << ": Parsed short_data[" << i << "].\n"
              << logofs_flush;
      #endif
    }
    else
    {
      genericRequest -> short_data[i] = 0;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Parsed identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  return true;
}

int GenericRequestStore::parseEnd(Message *message, unsigned char *buffer,
                                      unsigned int size, int bigEndian) const
{
  if (control -> isProtoStep3() == 1)
  {
    return 1;
  }

  GenericRequestMessage *genericRequest = (GenericRequestMessage *) message;

  //
  // We replaced bytes in data with zeroes to help
  // checksum. Need to restore the original content
  // before going through differential encoding in
  // channel.
  //

  for (unsigned int i = 0; i < 8 && (i * 2 + 4) < size; i++)
  {
    PutUINT(genericRequest -> short_data[i], ((unsigned char *) buffer) + i * 2 + 4, bigEndian);
  }

  #ifdef DEBUG
  *logofs << name() << ": Ended parse of identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  return true;
}

int GenericRequestStore::unparseIdentity(const Message *message, unsigned char *buffer,
                                             unsigned int size, int bigEndian) const
{
  GenericRequestMessage *genericRequest = (GenericRequestMessage *) message;

  *(buffer + 1) = genericRequest -> byte_data;

  for (unsigned int i = 0; i < 8 && (i * 2 + 4) < size; i++)
  {
    PutUINT(genericRequest -> short_data[i], buffer + i * 2 + 4, bigEndian);
  }

  #ifdef DEBUG
  *logofs << name() << ": Unparsed identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  return true;
}

void GenericRequestStore::dumpIdentity(const Message *message) const
{
  #ifdef DUMP

  GenericRequestMessage *genericRequest = (GenericRequestMessage *) message;

  *logofs << name() << ": Identity byte_data " << (unsigned) genericRequest -> byte_data;

  for (int i = 0; i < 8; i++)
  {
    *logofs << ", short_data[" << i << "] " << genericRequest -> short_data[i];
  }

  *logofs << ", size " << genericRequest -> size_ << ".\n" << logofs_flush;

  #endif
}

void GenericRequestStore::identityChecksum(const Message *message, const unsigned char *buffer,
                                               unsigned int size, int bigEndian) const
{
  //
  // As data offset can be beyond the real end of
  // message, we need to include size or we will
  // match any message of size less or equal to
  // data offset.
  //

  if (control -> isProtoStep3() == 1)
  {
    md5_append(md5_state_, buffer + 2,  2);
  }
}

void GenericRequestStore::updateIdentity(EncodeBuffer &encodeBuffer, const Message *message,
                                             const Message *cachedMessage,
                                                 ChannelCache *channelCache) const
{
  //
  // Encode the variant part.
  //

  GenericRequestMessage *genericRequest       = (GenericRequestMessage *) message;
  GenericRequestMessage *cachedGenericRequest = (GenericRequestMessage *) cachedMessage;

  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef TEST
  *logofs << name() << ": Encoding value "
          << (unsigned) genericRequest -> byte_data
          << " as byte_data field.\n"
          << logofs_flush;
  #endif

  encodeBuffer.encodeCachedValue((unsigned int) genericRequest -> byte_data, 8,
                     clientCache -> genericRequestCharCache);

  cachedGenericRequest -> byte_data = genericRequest -> byte_data;

  for (int i = 0; i < 8 && (i * 2 + 4) < genericRequest -> size_; i++)
  {
    #ifdef TEST
    *logofs << name() << ": Encoding value "
            << (unsigned short ) genericRequest -> short_data[i]
            << " as short_data[" << i << "] field.\n"
            << logofs_flush;
    #endif

    encodeBuffer.encodeCachedValue((unsigned int) genericRequest -> short_data[i], 16,
                 *clientCache -> genericRequestIntCache[i]);

    cachedGenericRequest -> short_data[i] = genericRequest -> short_data[i];
  }
}

void GenericRequestStore::updateIdentity(DecodeBuffer &decodeBuffer, const Message *message,
                                             ChannelCache *channelCache) const
{
  GenericRequestMessage *genericRequest = (GenericRequestMessage *) message;

  ClientCache *clientCache = (ClientCache *) channelCache;

  decodeBuffer.decodeCachedValue(genericRequest -> byte_data, 8,
                     clientCache -> genericRequestCharCache);

  #ifdef TEST
  *logofs << name() << ": Decoded value "
          << (unsigned) genericRequest -> byte_data
          << " as byte_data field.\n"
          << logofs_flush;
  #endif

  unsigned int value;

  for (int i = 0; i < 8 && (i * 2 + 4) < genericRequest -> size_; i++)
  {
    decodeBuffer.decodeCachedValue(value, 16,
                 *clientCache -> genericRequestIntCache[i]);

    genericRequest -> short_data[i] = (unsigned short) value;

    #ifdef TEST
    *logofs << name() << ": Decoded value "
            << (unsigned short) genericRequest -> short_data[i]
            << " as short_data[" << i << "] field.\n"
            << logofs_flush;
    #endif
  }
}
