#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <sys/stat.h>
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <cerrno>
#include <string>

#include <glib.h>
#include <glib/gi18n.h>

#include "stardict.h"

#include "conf.h"

std::auto_ptr<AppConf> conf;
std::string gStarDictDataDir;

inline void free_list(GSList *l){
  if(l!=NULL){
    g_slist_foreach(l, (GFunc)g_free, NULL);
    g_slist_free(l);
	}
}

#if defined(_WIN32) || defined(WITHOUT_GNOME)

const char STRING_SEP = (const char) 0xff;

//---------------------------------------------------------------------------------
static GSList * Str2List(gchar *str)
{
  GSList *list = NULL;
  gchar *p;
  while ((p = strchr(str, STRING_SEP))!=NULL) {
    list = g_slist_append(list, g_strndup(str, p - str));
    str = p+1;
	}
  if (str[0])
    list = g_slist_append(list, g_strdup(str));
  return list;
}

//---------------------------------------------------------------------------------
ConfigLine *ConfigSection::CreateString(const gchar * key, const gchar * value)
{
  ConfigLine *line;
  line = (ConfigLine *)g_malloc0(sizeof(ConfigLine));
  line->key = g_strchug(g_strchomp(g_strdup (key)));
  line->value = g_strchug(g_strchomp(g_strdup (value)));
  this->lines = g_list_append(this->lines, line);

  return line;
}
//---------------------------------------------------------------------------------
ConfigLine *ConfigSection::FindString(const gchar * key)
{
  ConfigLine *line;
  GList *list;

  list = this->lines;
  while (list){
    line = (ConfigLine *) list->data;
    if (!strcasecmp (line->key, key))
      return line;
    list = g_list_next (list);
  }
  return NULL;
}
//---------------------------------------------------------------------------------
#endif

//
//TBaseConf methods
//
#if defined(_WIN32) || defined(WITHOUT_GNOME)

ConfigSection *TBaseConf::CreateSection(const gchar * name)
{
  ConfigSection *section;
  section = (ConfigSection *)g_malloc0(sizeof(ConfigSection));
  section->name = g_strdup(name);
  this->sections = g_list_append(this->sections, section);

  return section;
}
//---------------------------------------------------------------------------------
ConfigSection *TBaseConf::FindSection(const gchar * name)
{
  ConfigSection *section;
  GList *list;

  list = this->sections;
  while (list){
    section = (ConfigSection *) list->data;
    if (!strcasecmp (section->name, name))
      return section;
    list = g_list_next (list);
  }
  return NULL;
}
#endif

TBaseConf::TBaseConf(const gchar *conf_path)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  sections=NULL;
  cfgfilename=NULL;
  Open(conf_path);
#else
  cfgfilename=g_strdup(conf_path);
  if ((gconf_client = gconf_client_get_default())==NULL)
    g_warning(_("Cannot connect to gconf."));
  else
    gconf_client_add_dir(gconf_client, conf_path, GCONF_CLIENT_PRELOAD_RECURSIVE, NULL);
#endif
}
//---------------------------------------------------------------------------------
TBaseConf::~TBaseConf()
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  ConfigSection *section;
  ConfigLine *line;
  GList *section_list, *line_list;

  g_free(this->cfgfilename);
  section_list=this->sections;
  while (section_list) {
    section = (ConfigSection *) section_list->data;
    g_free (section->name);

    line_list = section->lines;
    while (line_list) {
      line = (ConfigLine *) line_list->data;
      g_free (line->key);
      g_free (line->value);
      g_free (line);
      line_list = g_list_next(line_list);
    }
    g_list_free (section->lines);
    g_free (section);
    
    section_list = g_list_next(section_list);
  }
  g_list_free(this->sections);
  this->sections = NULL;
  this->cfgfilename = NULL;
#else
	if (!gconf_client)
		return;
  gconf_client_remove_dir(gconf_client, cfgfilename, NULL);
  g_object_unref(gconf_client);
  g_free(cfgfilename);
#endif
}
//---------------------------------------------------------------------------------
#if defined(_WIN32) || defined(WITHOUT_GNOME)
bool TBaseConf::Open(const gchar * filename)
{	
  FILE *file;
  gchar *buffer, **lines, *tmp;
  gint i;
  struct stat stats;
  ConfigSection *section = NULL;
  if (NULL==filename)
    return false;
  g_free(this->cfgfilename);
  this->cfgfilename = g_strdup(filename);

  if (stat(filename, &stats) == -1)
    return false;
  
  if (!(file = fopen (filename, "rb")))
    return false;
  

  buffer = (gchar *)g_malloc (stats.st_size + 1);
  fread(buffer, 1, stats.st_size, file);
  fclose(file);
  buffer[stats.st_size] = '\0';

  lines = g_strsplit(buffer, "\n", 0);
  g_free(buffer);
  i = 0;
  while (lines[i]) {
    if (lines[i][0] == '[') {
      if ((tmp = strchr(lines[i], ']'))) {
	*tmp = '\0';
	section = CreateSection(&lines[i][1]);
      }
    } else if(lines[i][0] != '#' && section) {
      if ((tmp = strchr (lines[i], '='))) {
	*tmp = '\0';
	tmp++;
	section->CreateString(lines[i], tmp);
      }
    }
    i++;
  }
  g_strfreev(lines);

  return true;
}
//---------------------------------------------------------------------------------
bool TBaseConf::Write(const gchar * filename)
{
  FILE *file;
  GList *section_list, *line_list;
  ConfigSection *section;
  ConfigLine *line;
  if (NULL==filename)
    return false;
  if (!(file = fopen (filename, "wb"))) {
    g_warning(N_("Can not open: %s - %s\n"), filename, strerror(errno));
    return false;
  }
  section_list = this->sections;
  while (section_list) {
    section = (ConfigSection *) section_list->data;
    if (section->lines) {
      fprintf(file, "[%s]\n", section->name);
      line_list = section->lines;
      while (line_list) {
	line = (ConfigLine *)line_list->data;
	fprintf(file, "%s=%s\n", line->key, line->value);
	line_list = g_list_next(line_list);
      }
      fprintf(file, "\n");
    }
    section_list = g_list_next(section_list);
  }
  fclose(file);
  return true;
}
//---------------------------------------------------------------------------------
#endif
bool TBaseConf::ReadBool(const gchar *section, const gchar *key, gboolean *value, gboolean def)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  gchar *str;

  if (!ReadString(section, key, &str)) {
    *value = def;
    return false;
  }

  if (!strcmp (str, "0"))
    *value=FALSE;
	else
    *value=TRUE;

  g_free(str);
#else
  if (!gconf_client) {
    *value=def;
    return false;
  }
  
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  *value=gconf_client_get_bool(gconf_client, real_key, NULL);
  g_free(real_key);
#endif
  return true;
}
//---------------------------------------------------------------------------------
bool TBaseConf::ReadInt(const gchar *section, const gchar *key, gint *value, gint def)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  gchar *str;

  if (!ReadString(section, key, &str)) {
    *value = def;
    return false;
  }
  *value = atoi(str);
  g_free(str);
#else
  if (!gconf_client) {
    *value=def;
    return false;
  }
  
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  *value=gconf_client_get_int(gconf_client, real_key, NULL);
  g_free(real_key);
#endif
  return true;
}
//---------------------------------------------------------------------------------
bool TBaseConf::ReadString(const gchar *section, const gchar *key, gchar **value)
{
  *value=NULL;
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  ConfigSection *sect;
  ConfigLine *line;

  if (!(sect = FindSection(section)))
    return false;
  if (!(line = sect->FindString(key)))
    return false;
  *value = g_strdup(line->value);
#else
	if (!gconf_client)
    return false;

  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  *value = gconf_client_get_string(gconf_client, real_key, NULL);
  g_free(real_key);

#endif
  return true;
}
//---------------------------------------------------------------------------------
bool TBaseConf::ReadStrList(const gchar *section, const gchar *key, GSList **list)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  gchar *str;

  if (!ReadString(section, key, &str)) {
    *list = NULL;
    return false;
  }

  *list = Str2List(str);
  g_free (str);
#else
  if (!gconf_client) {
    *list = NULL;
    return false;
  }
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  *list = gconf_client_get_list(gconf_client, real_key, GCONF_VALUE_STRING, NULL);
  g_free(real_key);
  
#endif
  return true;
}
//---------------------------------------------------------------------------------
void TBaseConf::WriteBool(const gchar *section, const gchar *key, gboolean value)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  if (value)
    WriteString(section, key, "1");
	else
    WriteString(section, key, "0");
#else
  if (!gconf_client)
    return;
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  gconf_client_set_bool(gconf_client, real_key, value, NULL);
  g_free(real_key);
#endif
}
//---------------------------------------------------------------------------------
void TBaseConf::WriteInt(const gchar *section, const gchar *key,  gint value)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  gchar *strvalue = g_strdup_printf("%d", value);
  WriteString(section, key, strvalue);
  g_free(strvalue);
#else
	if (!gconf_client)
		return;
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  gconf_client_set_int(gconf_client, real_key, value, NULL);
  g_free(real_key);
#endif
}
//---------------------------------------------------------------------------------
void TBaseConf::WriteString(const gchar *section, const gchar *key, const gchar *value)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  ConfigSection *sect;
  ConfigLine *line;

  sect = FindSection(section);
  if(!sect)
    sect = CreateSection(section);
  if((line = sect->FindString(key))){
    g_free(line->value);
    line->value = g_strchug(g_strchomp(g_strdup (value)));
  }
	else
    sect->CreateString(key, value);
  Save();
#else
  if(!gconf_client)
    return;
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  gconf_client_set_string(gconf_client, real_key, value, NULL);
  g_free(real_key);
#endif
}
//---------------------------------------------------------------------------------
void TBaseConf::WriteStrList(const gchar *section, const gchar *key, GSList *list)
{
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  std::string string;
  if (list) {
    string+=(gchar *)(list->data);
    list=list->next;
  }
  while (list) {
    string += STRING_SEP;
    string += (gchar *)(list->data);
    list = list->next;
  }
  WriteString(section, key, string.c_str());
#else
	if (!gconf_client)
		return;
  gchar *real_key=g_strdup_printf("%s/%s", section, key);
  gconf_client_set_list(gconf_client, real_key, GCONF_VALUE_STRING, list, NULL);
  g_free(real_key);
#endif
}
//---------------------------------------------------------------------------------


AppConf::AppConf(const gchar *conf_path) : TBaseConf(conf_path)
{
#if !defined(_WIN32) && !defined(WITHOUT_GNOME)
  if (gconf_client!=NULL) {
    gconf_client_notify_add(gconf_client, "/apps/stardict/preferences/dictionary/scan_selection", 
			    dictionary_scan_selection_changed_cb, NULL, NULL, NULL);
    gconf_client_notify_add(gconf_client, "/apps/stardict/preferences/main_window/hide_list", 
			    main_window_hide_list_changed_cb, NULL, NULL, NULL);
    gconf_client_notify_add(gconf_client, "/apps/stardict/preferences/floating_window/lock", 
			    floatwin_lock_changed_cb, NULL, NULL, NULL);
    gconf_client_notify_add(gconf_client, "/apps/stardict/preferences/floating_window/lock_x", 
			    floatwin_lock_x_changed_cb, NULL, NULL, NULL);
    gconf_client_notify_add(gconf_client, "/apps/stardict/preferences/floating_window/lock_y", 
			    floatwin_lock_y_changed_cb, NULL, NULL, NULL);
  }
#endif
  Load();
}

AppConf::~AppConf()
{
  free_list(search_website_list);
  free_list(dict_order_list);
  free_list(dict_disable_list);
  free_list(treedict_order_list);
  free_list(treedict_disable_list);
}

#if !defined(_WIN32) && !defined(WITHOUT_GNOME)
void AppConf::dictionary_scan_selection_changed_cb(GConfClient *client, guint id, 
						   GConfEntry *entry, gpointer data)
{
	GConfValue *value = gconf_entry_get_value (entry);
    
	gboolean scan = gconf_value_get_bool (value);
  //	gpAppFrame->oAppCore.oSelection.bEnable = scan;	
	gtk_widget_set_sensitive(gpAppFrame->oAppCore.oFloatWin.StopButton, scan);
  if(scan != gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gpAppFrame->oAppCore.oBottomWin.ScanSelectionCheckButton))) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gpAppFrame->oAppCore.oBottomWin.ScanSelectionCheckButton),scan);
		//although this make check button's callback func write to Gconf again,this callback will not be called again.
	}		
	if (scan) {
		if (!GTK_WIDGET_VISIBLE(gpAppFrame->oAppCore.window))
			gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_SCAN_ICON);
    if (conf->get_lock() && (gpAppFrame->oAppCore.oFloatWin.QueryingWord.c_str())[0]!='\0')
			gpAppFrame->oAppCore.oFloatWin.Show();
	}
	else {
		if (!GTK_WIDGET_VISIBLE(gpAppFrame->oAppCore.window))
			gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_STOP_ICON);
		gpAppFrame->oAppCore.oFloatWin.Hide();
		gpAppFrame->oAppCore.oSelection.LastClipWord.clear();
	}
}


void AppConf::main_window_hide_list_changed_cb(GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value = gconf_entry_get_value (entry);
    
  gboolean hide = gconf_value_get_bool (value);
  if (hide) {
    gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oToolWin.HideListButton);
    gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oToolWin.ShowListButton);
    gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oIndexWin.vbox);
  }
  else {
    gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oToolWin.ShowListButton);
    gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oToolWin.HideListButton);
    gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oIndexWin.vbox);
  }
}


void AppConf::floatwin_lock_changed_cb(GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value = gconf_entry_get_value (entry);
    
  gboolean lock = gconf_value_get_bool (value);
  //  gpAppFrame->oAppCore.oFloatWin.bIsLocked = lock;
  if (lock)
    gtk_image_set_from_stock(GTK_IMAGE(gpAppFrame->oAppCore.oFloatWin.lock_image),GTK_STOCK_GOTO_LAST,GTK_ICON_SIZE_MENU);
  else
    gtk_image_set_from_stock(GTK_IMAGE(gpAppFrame->oAppCore.oFloatWin.lock_image),GTK_STOCK_GO_FORWARD,GTK_ICON_SIZE_MENU);
}

void AppConf::floatwin_lock_x_changed_cb(GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
  if (conf->get_lock()){
	GConfValue *value = gconf_entry_get_value (entry);
    gint lock_x = gconf_value_get_int (value);
    
    gint old_x,old_y;
    gtk_window_get_position(GTK_WINDOW(gpAppFrame->oAppCore.oFloatWin.FloatWindow),&old_x,&old_y);
    if (lock_x!=old_x)
      gtk_window_move(GTK_WINDOW(gpAppFrame->oAppCore.oFloatWin.FloatWindow),lock_x,old_y);
  }
}

void AppConf::floatwin_lock_y_changed_cb(GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
  if (conf->get_lock()){
	GConfValue *value = gconf_entry_get_value (entry);
    gint lock_y = gconf_value_get_int (value);
    
    gint old_x,old_y;
    gtk_window_get_position(GTK_WINDOW(gpAppFrame->oAppCore.oFloatWin.FloatWindow),&old_x,&old_y);
    if (lock_y!=old_y)
      gtk_window_move(GTK_WINDOW(gpAppFrame->oAppCore.oFloatWin.FloatWindow),old_x,lock_y);
  }
}
#else
void on_conf_dictionary_scan_selection_changed(gboolean scan)
{
  if (conf->get_scan_selection() == scan)
    return;

  gtk_widget_set_sensitive(gpAppFrame->oAppCore.oFloatWin.StopButton, scan);
  if (scan != gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gpAppFrame->oAppCore.oBottomWin.ScanSelectionCheckButton)))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gpAppFrame->oAppCore.oBottomWin.ScanSelectionCheckButton),scan);

  if (scan) {
    if (!GTK_WIDGET_VISIBLE(gpAppFrame->oAppCore.window))
      gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_SCAN_ICON);
    if (conf->get_lock() && (gpAppFrame->oAppCore.oFloatWin.QueryingWord.c_str())[0]!='\0')
      gpAppFrame->oAppCore.oFloatWin.Show();
  } else {
    if (!GTK_WIDGET_VISIBLE(gpAppFrame->oAppCore.window)) 
      gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_STOP_ICON);    
    gpAppFrame->oAppCore.oFloatWin.Hide();
    gpAppFrame->oAppCore.oSelection.LastClipWord.clear();
	}
}

    
void on_conf_main_window_hide_list_changed(gboolean hide)
{
	if (hide) {
		gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oToolWin.HideListButton);
		gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oToolWin.ShowListButton);
		gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oIndexWin.vbox);
	}
	else {
		gtk_widget_hide(gpAppFrame->oAppCore.oMidWin.oToolWin.ShowListButton);
		gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oToolWin.HideListButton);
		gtk_widget_show(gpAppFrame->oAppCore.oMidWin.oIndexWin.vbox);
	}
}



void on_conf_floatwin_lock_changed(gboolean lock)
{
  if (lock)
    gtk_image_set_from_stock(GTK_IMAGE(gpAppFrame->oAppCore.oFloatWin.lock_image),GTK_STOCK_GOTO_LAST,GTK_ICON_SIZE_MENU);
  else
    gtk_image_set_from_stock(GTK_IMAGE(gpAppFrame->oAppCore.oFloatWin.lock_image),GTK_STOCK_GO_FORWARD,GTK_ICON_SIZE_MENU);
}
    
#endif

//---------------------------------------------------------------------------------
//load preference
void AppConf::Load(void)
{
  ReadInt("/apps/stardict/preferences/main_window", "hpaned_pos", &hpaned_pos, DEFAULT_HPANED_POS);
  ReadInt("/apps/stardict/preferences/main_window", "window_width", &window_width, DEFAULT_WINDOW_WIDTH);
  ReadInt("/apps/stardict/preferences/main_window", "window_height", &window_height, DEFAULT_WINDOW_HEIGHT);
  ReadInt("/apps/stardict/preferences/floating_window", "lock_x", &lock_x, 0);
  ReadInt("/apps/stardict/preferences/floating_window", "lock_y", &lock_y, 0);
  ReadBool("/apps/stardict/preferences/main_window", "maximized", &maximized, FALSE);
  ReadBool("/apps/stardict/preferences/dictionary", "use_custom_font", &use_custom_font, FALSE);
  ReadString("/apps/stardict/preferences/dictionary", "custom_font", &custom_font);
  ReadBool("/apps/stardict/preferences/main_window", "hide_on_startup", &hide_on_startup, FALSE);
  ReadBool("/apps/stardict/preferences/dictionary", "enable_sound_event", &enable_sound_event, TRUE);
  ReadBool("/apps/stardict/preferences/main_window", "hide_list", &hide_list, FALSE);
  ReadStrList("/apps/stardict/preferences/main_window", "search_website_list", &search_website_list);
  ReadBool("/apps/stardict/preferences/dictionary", "scan_selection", &scan_selection, TRUE);
  ReadBool("/apps/stardict/preferences/notification_area_icon", "query_in_floatwin", &query_in_floatwin, TRUE);
  ReadBool("/apps/stardict/preferences/dictionary", "only_scan_while_modifier_key", &only_scan_while_modifier_key, FALSE);
  ReadBool("/apps/stardict/preferences/dictionary", "hide_floatwin_when_modifier_key_released", &hide_floatwin_when_modifier_key_released, TRUE);
  ReadInt("/apps/stardict/preferences/dictionary", "scan_modifier_key", &scan_modifier_key, 0);
  ReadBool("/apps/stardict/preferences/main_window", "hide_on_startup", &hide_on_startup, FALSE);
  ReadBool("/apps/stardict/preferences/floating_window", "pronounce_when_popup", &pronounce_when_popup, FALSE);
  ReadInt("/apps/stardict/preferences/floating_window", "max_window_width", &max_window_width, DEFAULT_MAX_FLOATWIN_WIDTH);
  ReadInt("/apps/stardict/preferences/floating_window", "max_window_height", &max_window_height, DEFAULT_MAX_FLOATWIN_HEIGHT);
  ReadStrList("/apps/stardict/manage_dictionaries", "treedict_order_list", &treedict_order_list);
  ReadStrList("/apps/stardict/manage_dictionaries", "treedict_disable_list", &treedict_disable_list);
  ReadStrList("/apps/stardict/manage_dictionaries", "dict_order_list", &dict_order_list);
  ReadStrList("/apps/stardict/manage_dictionaries", "dict_disable_list", &dict_disable_list);
  ReadBool("/apps/stardict/preferences/floating_window", "lock", &lock, FALSE);
}
//---------------------------------------------------------------------------------
void AppConf::set_hpaned_pos(gint value)
{
  if (hpaned_pos==value)
    return;

  WriteInt("/apps/stardict/preferences/main_window", "hpaned_pos", value);

  hpaned_pos=value;
}

void AppConf::set_window_width(gint value)
{
  if (window_width==value)
    return;
    
  WriteInt("/apps/stardict/preferences/main_window", "window_width", value);

  window_width=value;
}

void AppConf::set_window_height(gint value)
{
  if (window_height==value)
    return;
    
  WriteInt("/apps/stardict/preferences/main_window", "window_height", value);

  window_height=value;
}

void AppConf::set_lock_x(gint value)
{
  if (lock_x==value)
    return;
    
  WriteInt("/apps/stardict/preferences/floating_window", "lock_x", value);

  lock_x=value;
}

void AppConf::set_lock_y(gint value)
{
  if (lock_y==value)
    return;
    
  WriteInt("/apps/stardict/preferences/floating_window", "lock_y", value);

  lock_y=value;
}

void AppConf::set_maximized(gboolean value)
{
  if (maximized==value)
    return;

  WriteBool("/apps/stardict/preferences/main_window", "maximized", value);

  maximized=value;
}

void AppConf::set_hide_list(gboolean value)
{
  if (hide_list==value)
    return;

  WriteBool("/apps/stardict/preferences/main_window", "hide_list", value);
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  on_conf_main_window_hide_list_changed(value);
#endif

  hide_list=value;
}

void AppConf::set_search_website_list(GSList *value)
{
  if (search_website_list==value)
    return;
  g_slist_foreach(search_website_list, (GFunc)g_free, NULL);
  g_slist_free(search_website_list);
  search_website_list = NULL;

  WriteStrList("/apps/stardict/preferences/main_window", "search_website_list", value);

  search_website_list=value;
}

void AppConf::set_scan_selection(gboolean value)
{
  if (scan_selection==value)
    return;

  WriteBool("/apps/stardict/preferences/dictionary", "scan_selection", value);
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  on_conf_dictionary_scan_selection_changed(value);	
#endif

  scan_selection=value;
}

void AppConf::set_only_scan_while_modifier_key(gboolean value)
{
  if (only_scan_while_modifier_key==value)
    return;

  WriteBool("/apps/stardict/preferences/dictionary", "only_scan_while_modifier_key", value);

  only_scan_while_modifier_key=value;
}

void AppConf::set_hide_floatwin_when_modifier_key_released(gboolean value)
{
  if (hide_floatwin_when_modifier_key_released==value)
    return;

  WriteBool("/apps/stardict/preferences/dictionary", "hide_floatwin_when_modifier_key_released", value);

  hide_floatwin_when_modifier_key_released=value;
}

void AppConf::set_scan_modifier_key(gint value)
{
  if (scan_modifier_key==value)
    return;

  WriteInt("/apps/stardict/preferences/dictionary", "scan_modifier_key", value);

  scan_modifier_key=value;
}

void AppConf::set_custom_font(const gchar *value)
{
  if (custom_font==value || 
     (custom_font && value && strcmp(custom_font, value)==0))
    return;

  WriteString("/apps/stardict/preferences/dictionary", "custom_font", value);

  custom_font=g_strdup(value); 
}

void AppConf::set_enable_sound_event(gboolean value)
{
  if (enable_sound_event==value)
    return;

  WriteBool("/apps/stardict/preferences/dictionary", "enable_sound_event", value);

  enable_sound_event=value;
}

void AppConf::set_use_custom_font(gboolean value)
{
  if (use_custom_font==value)
    return;

  WriteBool("/apps/stardict/preferences/dictionary", "use_custom_font", value);

  use_custom_font=value;
}

void AppConf::set_hide_on_startup(gboolean value)
{
  if (hide_on_startup==value)
    return;

  WriteBool("/apps/stardict/preferences/main_window", "hide_on_startup", value);

  hide_on_startup=value;
}

void AppConf::set_query_in_floatwin(gboolean value)
{
  if (query_in_floatwin==value)
    return;

  WriteBool("/apps/stardict/preferences/notification_area_icon", "query_in_floatwin", value);

  query_in_floatwin=value;
}

void AppConf::set_pronounce_when_popup(gboolean value)
{
  if (pronounce_when_popup==value)
    return;

  WriteBool("/apps/stardict/preferences/floating_window", "pronounce_when_popup", value);

  pronounce_when_popup=value;
}

void AppConf::set_max_window_width(gint value)
{
  if (max_window_width==value)
    return;

  WriteInt("/apps/stardict/preferences/floating_window", "max_window_width", value);

  max_window_width=value;
}

void AppConf::set_max_window_height(gint value)
{
  if (max_window_height==value)
    return;

  WriteInt("/apps/stardict/preferences/floating_window", "max_window_height", value);

  max_window_height=value;
}

void AppConf::set_treedict_order_list(GSList *value)
{
  if (treedict_order_list==value)
    return;

  WriteStrList("/apps/stardict/manage_dictionaries", "treedict_order_list", value);

  g_slist_foreach(treedict_order_list, (GFunc)g_free, NULL);
  g_slist_free(treedict_order_list);
  treedict_order_list=value;
}

void AppConf::set_treedict_disable_list(GSList *value)
{
  if (treedict_disable_list==value)
    return;

  WriteStrList("/apps/stardict/manage_dictionaries", "treedict_disable_list", value);

  g_slist_foreach(treedict_disable_list, (GFunc)g_free, NULL);
  g_slist_free(treedict_disable_list);
  treedict_disable_list=value;
}

void AppConf::set_dict_order_list(GSList *value)
{
  if (dict_order_list==value)
    return;

  WriteStrList("/apps/stardict/manage_dictionaries", "dict_order_list", value);

  g_slist_foreach(dict_order_list, (GFunc)g_free, NULL);
  g_slist_free(dict_order_list);
  dict_order_list=value;
}

void AppConf::set_dict_disable_list(GSList *value)
{
  if (dict_disable_list==value)
    return;

  WriteStrList("/apps/stardict/manage_dictionaries", "dict_disable_list", value);

  g_slist_foreach(dict_disable_list, (GFunc)g_free, NULL);
  g_slist_free(dict_disable_list);
  
  dict_disable_list=value;
}

void AppConf::set_lock(gboolean value)
{
  if (lock==value)
    return;

  WriteBool("/apps/stardict/preferences/floating_window", "lock", value);
#if defined(_WIN32) || defined(WITHOUT_GNOME)
  on_conf_floatwin_lock_changed(value);
#endif

  lock=value;
}
