/*****************************************************************************
 * grab-v4l.c: interface to the v4l driver
 *****************************************************************************
 * $Id: grab-v4l.c,v 1.31 2004/09/12 12:55:13 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * Original code:
 *
 *   (c) 1997,98 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 *****************************************************************************/
#include "config.h"

#ifdef HAVE_V4L

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <X11/Intrinsic.h>

#include "strtab.h"
#include "grab.h"
#include "colorspace.h"
#include "blackborder.h"

static int bttvfd = -1;
extern int vbifd;

#include <asm/types.h>            /* XXX glibc */
#include "videodev.h"
#include "videodev2.h"
#include "mixer.h"
#include "bttv.h"
#include "x11.h"
#include "channel.h"
//#include "plugin.h"
#include "memcpy.h"

char *int_to_str (int n, struct STRTAB *tab);

extern int have_xv;
extern int have_video_xv;
extern int cur_capture;
//extern double vts;
/* ---------------------------------------------------------------------- */

extern struct GRABBER grab_v4l;
static inline void reset(void);
static int grab_queue (struct video_mmap *gb);
static int grab_wait (struct video_mmap *gb);
static int grab_audio (int mute, int volume, int *mode);
extern int owidth,oheight;

/* ---------------------------------------------------------------------- */
#define DEV_CAP_CAPTURE    (1<<0)
#define DEV_CAP_TUNER      (1<<1)
#define DEV_CAP_TELETEXT   (1<<2)
#define DEV_CAP_OVERLAY    (1<<3)
#define DEV_CAP_CHROMAKEY  (1<<4)
#define DEV_CAP_CLIPPING   (1<<5)
#define DEV_CAP_FRAMERAM   (1<<6)
#define DEV_CAP_SCALES     (1<<7)
#define DEV_CAP_MONOCHROME (1<<8)
#define DEV_CAP_SUBCAPTURE (1<<9)

static char *device_cap[] = {
  "capture", "tuner", "teletext", "overlay", "chromakey", "clipping",
  "frameram", "scales", "monochrome", "subcapture", NULL
};

static char *device_pal[] = {
  "-", "grey", "hi240", "rgb16(565)", "rgb24", "rgb32", "rgb15(555)",
  "yuv422", "yuyv", "uyvy", "yuv420", "yuv411", "raw", "yuv422p", "yuv411p",
  "yuv420p", "yuv410p"
};
static int device_pal_depth[] = {0,8,8,16,24,32,16,16,16,16,
				 16,16,8,16,12,12,9};
#define PALETTE(x) ((x < sizeof(device_pal)/sizeof(char*)) ? device_pal[x] : "UNKNOWN")

static struct STRTAB norms[] = {
  {VIDEO_MODE_PAL,   "PAL"},
  {VIDEO_MODE_NTSC,  "NTSC"},
  {VIDEO_MODE_SECAM, "SECAM"},
  {VIDEO_MODE_AUTO,  "AUTO"},
  {-1, NULL}
};

static struct STRTAB norms_bttv[] = {
  { VIDEO_MODE_PAL,  "PAL"   },
  { VIDEO_MODE_NTSC, "NTSC"  },
  { VIDEO_MODE_SECAM, "SECAM"  },
  { 3, "PAL-NC" },
  { 4, "PAL-M"  },
  { 5, "PAL-N"  },
  { 6,"NTSC-JP" },
  { -1, NULL }
};

static struct STRTAB *inputs;

static int fmts_xaw_to_v4l[MAX_VIDEO_FMT] = {
  [VIDEO_GRAY8]  = VIDEO_PALETTE_GREY,
  [VIDEO_HI240]  = VIDEO_PALETTE_HI240,
  [VIDEO_RGB15]  = VIDEO_PALETTE_RGB555,
  [VIDEO_RGB16]  = VIDEO_PALETTE_RGB565,
  [VIDEO_RGB24]  = VIDEO_PALETTE_RGB24,
  [VIDEO_RGB32]  = VIDEO_PALETTE_RGB32,
  [VIDEO_YUYV]   = VIDEO_PALETTE_YUV422,
  [VIDEO_YUV420] = VIDEO_PALETTE_YUV420P,
  [VIDEO_UYVY]   = VIDEO_PALETTE_UYVY,
  [VIDEO_YUV410] = VIDEO_PALETTE_YUV410P,
  [VIDEO_YUV411P] = VIDEO_PALETTE_YUV411P,
  [VIDEO_YUV422P] = VIDEO_PALETTE_YUV422P
};

/* generic informations */
static struct video_capability v4l_capability;
static struct video_channel *v4l_channels;
static struct video_audio *v4l_audios;
static struct video_tuner v4l_tuner;
static struct video_picture v4l_pict;
static int cur_input;
static int cur_norm;

/* overlay */
static struct video_window ov_win;
static struct video_clip ov_clips[32];
static struct video_buffer ov_fbuf;

/* screen grab */
struct video_mmap *gb;
static int gb_grab=0,gb_sync=0;
static int imgq=0,imgw=0;
struct video_mbuf gb_buffers = { 2 * 0x151000, 0, {0, 0x151000} };
//struct video_mbuf gb_buffers = { 2 * 0x1B0000, 0, {0, 0x1B0000} };
char *map = NULL;
static double *stamps;

/* state */
static int overlay;

/* pass 0/1 by reference */
static int one = 1, zero = 0;


struct driver_st {
  const char *name;
  const char* devname;
  int capabilities;
};
static struct driver_st driver;

/* ---------------------------------------------------------------------- */

static int grab_open(struct device_t *device)
{
  int i, ov_error=0;
  int rc;

  if (-1 != bttvfd)
    goto err;

  driver.name = "v4l";

  if (debug)
    fprintf (stderr, "v4l: open\n");

  if (-1 == (bttvfd = open (device->video ? device->video : "/dev/video", O_RDWR)))
    {
      fprintf (stderr, "open %s: %s\n",
               device->video ? device->video : "/dev/video", strerror (errno));
      goto err;
    }

  if (-1 == ioctl (bttvfd, VIDIOCGCAP, &v4l_capability))
    goto err;
  driver.devname = v4l_capability.name;
  driver.capabilities = v4l_capability.type;

  if(device->vbi)
    if (-1 == (vbifd = open (device->vbi, O_RDONLY)))
      {
	fprintf (stderr, "v4l: cant't open %s for reading, %s\n",
		 device->vbi, strerror (errno));
	//goto err;
      }

  if (-1 != (rc = ioctl(bttvfd, BTTV_VERSION, &i)))
    {
      if (debug || rc < 0x000700)
        fprintf(stderr, "v4l: bttv version %d.%d.%d\n\n",
                (rc >> 16) & 0xff,
                (rc >> 8)  & 0xff,
                rc        & 0xff);

      if (rc < 0x000700)
        fprintf(stderr,
                "v4l: prehistoric bttv version found, please try to\n"
                "     upgrade the driver (http://bytesex.org/bttv/)\n"
                "     before mailing bug reports\n");
    }

  if (debug)
    {
      fprintf (stderr, "%s:", v4l_capability.name);
      for (i = 0; device_cap[i] != NULL; i++)
        if (v4l_capability.type & (1 << i))
          fprintf (stderr, " %s", device_cap[i]);
      fprintf (stderr, "\n");
    }

  /* input sources */
  if (debug)
    fprintf (stderr, "  channels: %d\n", v4l_capability.channels);
  v4l_channels = malloc (sizeof (struct video_channel) * v4l_capability.channels);
  memset (v4l_channels, 0, sizeof (struct video_channel) * v4l_capability.channels);
  inputs = malloc (sizeof (struct STRTAB) * (v4l_capability.channels + 1));
  memset (inputs, 0, sizeof (struct STRTAB) * (v4l_capability.channels + 1));
  for (i = 0; i < v4l_capability.channels; i++)
    {
      v4l_channels[i].channel = i;
      if (-1 == ioctl (bttvfd, VIDIOCGCHAN, &v4l_channels[i]))
        perror ("ioctl VIDIOCGCHAN"), exit (0);
      inputs[i].nr = i;
      inputs[i].str = v4l_channels[i].name;
      if (debug)
        fprintf (stderr, "    %s: %d %s%s %s%s\n",
                 v4l_channels[i].name,
                 v4l_channels[i].tuners,
                 (v4l_channels[i].flags & VIDEO_VC_TUNER) ?    "tuner " :  "",
                 (v4l_channels[i].flags & VIDEO_VC_AUDIO) ?    "audio " :  "",
                 (v4l_channels[i].type  & VIDEO_TYPE_TV) ?     "tv " :     "",
                 (v4l_channels[i].type  & VIDEO_TYPE_CAMERA) ? "camera " : "");
    }
  inputs[i].nr = -1;
  inputs[i].str = NULL;
  grab_v4l.inputs = inputs;

  /* ioctl probe, switch to input 0 */
  if (-1 == ioctl (bttvfd, VIDIOCSCHAN, &v4l_channels[0]))
    {
      fprintf (stderr, "v4l: you need a newer bttv version (>= 0.5.14)\n");
      goto err;
    }

  /* audios */
  if (debug)
    fprintf (stderr, "  audios  : %d\n", v4l_capability.audios);
  v4l_audios = malloc (sizeof (struct video_audio) * v4l_capability.audios);
  memset (v4l_audios, 0, sizeof (struct video_audio) * v4l_capability.audios);
  for (i = 0; i < v4l_capability.audios; i++)
    {
      v4l_audios[i].audio = i;
      if (-1 == ioctl (bttvfd, VIDIOCGAUDIO, &v4l_audios[i]))
        perror ("ioctl VIDIOCGCAUDIO") /* , exit(0) */ ;
      if (debug)
        {
          fprintf (stderr, "    %d (%s): ", i, v4l_audios[i].name);
          if (v4l_audios[i].flags & VIDEO_AUDIO_MUTABLE)
            fprintf (stderr, "muted=%s ",
                     (v4l_audios[i].flags & VIDEO_AUDIO_MUTE) ? "yes" : "no");
          if (v4l_audios[i].flags & VIDEO_AUDIO_VOLUME)
            fprintf (stderr, "volume=%d ", v4l_audios[i].volume);
          if (v4l_audios[i].flags & VIDEO_AUDIO_BASS)
            fprintf (stderr, "bass=%d ", v4l_audios[i].bass);
          if (v4l_audios[i].flags & VIDEO_AUDIO_TREBLE)
            fprintf (stderr, "treble=%d ", v4l_audios[i].treble);
          fprintf (stderr, "\n");
        }
    }
  if (debug)
    fprintf (stderr, "  size    : %dx%d => %dx%d\n",
             v4l_capability.minwidth, v4l_capability.minheight,
             v4l_capability.maxwidth, v4l_capability.maxheight);

  /* tuner (more than one???) */
  if (v4l_capability.type & DEV_CAP_TUNER)
    {
      if (-1 == ioctl (bttvfd, VIDIOCGTUNER, &v4l_tuner))
        perror ("ioctl VIDIOCGTUNER");

      if (debug)
        fprintf (stderr, "  tuner   : %s %lu-%lu",
                 v4l_tuner.name, v4l_tuner.rangelow, v4l_tuner.rangehigh);

      for (i = 0; norms[i].str != NULL; i++)
        {
          if (v4l_tuner.flags & (1 << i))
            {
              if (debug)
                fprintf (stderr, " %s", norms[i].str);
            }
          else
            norms[i].nr = -1;
        }
      if (debug)
        fprintf (stderr, "\n");
    }

#if 1
    /* dirty hack time / v4l design flaw -- works with bttv only
     * this adds support for a few less common PAL versions */
  if( rc != -1 && rc >= 0x000700 && rc<0x000800)
    grab_v4l.norms = norms_bttv;
#endif

  /* frame buffer */
  if (-1 == ioctl (bttvfd, VIDIOCGFBUF, &ov_fbuf))
    perror ("ioctl VIDIOCGFBUF");

  if (debug)
    fprintf (stderr, "  fbuffer : base=0x%p size=%dx%d depth=%d bpl=%d\n",
             ov_fbuf.base, ov_fbuf.width, ov_fbuf.height,
             ov_fbuf.depth, ov_fbuf.bytesperline);

  /* picture parameters */
  if (-1 == ioctl (bttvfd, VIDIOCGPICT, &v4l_pict))
    perror ("ioctl VIDIOCGPICT");

  if (debug)
    {
      fprintf (stderr,
               "  picture : brightness=%d hue=%d colour=%d contrast=%d\n",
               v4l_pict.brightness, v4l_pict.hue, v4l_pict.colour, v4l_pict.contrast);
      fprintf (stderr,
               "  picture : whiteness=%d depth=%d palette=%s\n",
               v4l_pict.whiteness, v4l_pict.depth, PALETTE (v4l_pict.palette));
    }

  /* double-check settings */
  if (v4l_capability.type & DEV_CAP_OVERLAY)
    {
      fprintf (stderr, "v4l: %dx%d, %d bit/pixel, %d byte/scanline\n",
	       ov_fbuf.width, ov_fbuf.height,
	       ov_fbuf.depth, ov_fbuf.bytesperline);
      if(ov_fbuf.base==NULL) { 
	fprintf(stderr,
		"WARNING: video memory base unknown, may be caused by a problem\n"
		"  with xdtv_v4l-conf or a non-availability of DGA\n"
		"  and frame buffer devices: OVERLAY IS DISABLED !\n");
	ov_error=1;
      }
      
      if (debug)
	{
	  fprintf (stderr, "v4l: framebuffer v4l base=%p\n", ov_fbuf.base);
	  /*fprintf (stderr, "v4l: framebuffer dga base=%p\n", base);*/
	}
    }
  else
    {
      fprintf(stderr, "v4l: device %s does not support overlay!\n", device->video);
      ov_error = 1;
    }

  if (ov_error) {
    fprintf(stderr,"WARNING: overlay mode disabled\n");
    grab_v4l.grab_overlay = NULL; //to force capture mode...
  }

  /* map grab buffer */
  if (-1 == ioctl (bttvfd, VIDIOCGMBUF, &gb_buffers))
    {
      perror ("ioctl VIDIOCGMBUF");
    }
  map =
    mmap (0, gb_buffers.size, PROT_READ | PROT_WRITE, MAP_SHARED, bttvfd, 0);

  if ((char *) -1 == map)
    {
      perror ("mmap");
    }
  else
    {
      fprintf (stderr, "v4l: mmap()'ed buffer size = 0x%x\n",
	       gb_buffers.size);
    }

#if 0
  if (map == (char *) -1)
    {
      /* XXX no mmap grabbing, use read */
    }
#endif

  nbufs=nbufs_default;
  if(nbufs>gb_buffers.frames) nbufs=gb_buffers.frames;
  gb=(struct video_mmap *) malloc(nbufs*sizeof(struct video_mmap));
  stamps=(double *) malloc(nbufs*sizeof(double));
  
  if(debug)
    fprintf(stderr,"nbufs=%d (max=%d)\n",nbufs,gb_buffers.frames);
  
  // Check palette capabilities
  fprintf(stderr, "Checking device palette capabilities:\n");
  for (i=1;i<sizeof(device_pal)/sizeof(char*);i++)
    {
      v4l_pict.palette = i;
      v4l_pict.depth = device_pal_depth[i]; /* for recent drivers... */
      if (-1 != ioctl (bttvfd, VIDIOCSPICT, &v4l_pict))
        {
          fprintf(stderr, "    %s\n", device_pal[i]);
        }
    }
  fprintf(stderr, "\n");

  for(i=0;i<nbufs;i++)
    gb[i].frame=i;
  
  return 0;

 err:
  if (bttvfd != -1)
    {
      close (bttvfd);
      bttvfd = -1;
    }
  return -1;
}



static int
grab_close ()
{
  if (-1 == bttvfd)
    return 0;

  grab_audio (1 /* mute */ , -1 /* volume */ , NULL);

  if (gb_grab > gb_sync)
    {
      if (-1 == /*ioctl(bttvfd,VIDIOCSYNC,0) */ 0)
        perror ("ioctl VIDIOCSYNC");
      else
        gb_sync++;
    }

  if ((char *) -1 != map)
    munmap (map, gb_buffers.size);

  if (debug)
    fprintf (stderr, "v4l: close\n");

  close (bttvfd);
  bttvfd = -1;
  close (vbifd);
  vbifd = -1;
  free(gb);
  free(stamps);

  return 0;
}

/* ---------------------------------------------------------------------- */

static int
grab_overlay (int x, int y, int width, int height, int format,
              struct OVERLAY_CLIP *oc, int count)
{
  int i;

  if(bttvfd == -1) return -1;

  while(gb_sync<gb_grab) {
    if(grab_wait(&gb[imgw])==-1) reset();
    imgw++;
    if(imgw==nbufs) imgw=0;
  }
  
  if (width == 0 || height == 0)
    {
      if (debug)
        fprintf (stderr, "v4l: overlay off\n");
      ioctl (bttvfd, VIDIOCCAPTURE, &zero);
      overlay = 0;
      return 0;
    }

  ov_win.x = x;
  ov_win.y = y;
  ov_win.width = width;
  ov_win.height = height;
  ov_win.flags = 0;

  if (v4l_capability.type & VID_TYPE_CLIPPING)
    {
      ov_win.clips = ov_clips;
      ov_win.clipcount = count;

      for (i = 0; i < count; i++)
        {
          ov_clips[i].x = oc[i].x1;
          ov_clips[i].y = oc[i].y1;
          ov_clips[i].width = oc[i].x2 - oc[i].x1 /* XXX */ ;
          ov_clips[i].height = oc[i].y2 - oc[i].y1;
          if (debug)
            fprintf (stderr, "v4l: clip=%dx%d+%d+%d\n",
                     ov_clips[i].width, ov_clips[i].height,
                     ov_clips[i].x, ov_clips[i].y);
          if (ov_clips[i].x < 0 || ov_clips[i].y < 0 ||
              ov_clips[i].width < 0 || ov_clips[i].height < 0)
            {
              fprintf (stderr, "v4l: bug trap - overlay off\n");
              ioctl (bttvfd, VIDIOCCAPTURE, &zero);
              overlay = 0;
              return 0;
            }
        }
    }
  if (v4l_capability.type & VID_TYPE_CHROMAKEY)
    {
      ov_win.chromakey = 0;        /* XXX */
    }
  if (!overlay)
    {
      v4l_pict.palette = fmts_xaw_to_v4l[format];
      v4l_pict.depth = device_pal_depth[v4l_pict.palette];
      if (-1 == ioctl (bttvfd, VIDIOCSPICT, &v4l_pict))
        perror ("ioctl VIDIOCSPICT");
    }

  if (-1 == ioctl (bttvfd, VIDIOCSWIN, &ov_win))
    perror ("ioctl VIDIOCSWIN");

  if(!overlay) {
    if (-1 == ioctl (bttvfd, VIDIOCCAPTURE, &one))
      perror ("ioctl VIDIOCCAPTURE");
    overlay = 1;
  }

  if (debug)
    fprintf (stderr, "v4l: overlay win=%dx%d+%d+%d, %d clips\n",
             width, height, x, y, count);

  return 0;
}

static int
grab_queue (struct video_mmap *gb)
{
  if (debug > 1)
    fprintf (stderr, "g%d", gb->frame);

  if (-1 == ioctl (bttvfd, VIDIOCMCAPTURE, gb))
    {
      if (errno == EAGAIN)
        fprintf (stderr, "grabber chip can't sync (no station tuned in?)\n");
      else
        fprintf (stderr, "ioctl VIDIOCMCAPTURE(%d,%s,%dx%d): %s\n",
                 gb->frame, PALETTE (gb->format), gb->width, gb->height,
                 strerror (errno));
      return -1;
    }

  gb_grab++;
  if (debug > 1)
    fprintf (stderr, "* ");

  return 0;
}

static int
grab_wait (struct video_mmap *gb)
{
  if (debug)
    fprintf (stderr, "s%d", gb->frame);

  if (-1 == ioctl (bttvfd, VIDIOCSYNC, &(gb->frame)))
    {
      perror ("ioctl VIDIOCSYNC");
      return -1;
    }
  
  gb_sync++;
  if (debug)
    fprintf (stderr, "* ");
  
  return 0;
}


static inline void reset(void) {
  fprintf(stderr,"total reset\n");
  for (imgw=0;imgw<nbufs;imgw++) grab_wait(&gb[imgw]);
  gb_grab=gb_sync=imgq=imgw=0;
}


static void* get_img(video_fmt f, int width, int height) {
  int n=0,format;
  struct timeval tv;
  
  format = fmts_xaw_to_v4l[f];
  if(!format) return 0;
 
  if(imgq!=gb_grab%nbufs || imgw!=gb_sync%nbufs)
    fprintf(stderr,"PB imgq=%d gb_grab=%d imgw=%d gb_sync=%d nbufs=%d\n",
	    imgq,gb_grab,imgw,gb_sync,nbufs);

  // single est toujours egal a 1 !!!
  do {
    n++;
    if(n>30) {
      fprintf(stderr,"expected frame will never come ?\n");
      return NULL;
    }
    while ((gb_grab-gb_sync)<nbufs) {
      gb[imgq].width=width; gb[imgq].height=height;
      gb[imgq].format=format;
      if (-1 == grab_queue(&gb[imgq]))  {
	reset();
	return NULL;
      }
      gettimeofday (&tv, NULL);
      stamps[imgq]=tv.tv_usec/1000000.0+tv.tv_sec;
      imgq++;
      if (imgq>=nbufs) imgq=0;
    }
    
    if (-1 == grab_wait(&gb[imgw])) {
      reset();
      return NULL;
    }
    
    img=imgw;
    
    imgw++;
    if (imgw==nbufs) imgw=0;
  } while( (gb[img].width!=width) || (gb[img].height!=height) || 
	   (gb[img].format!=format));
  gettimeofday (&tv, NULL);
  videostampmax = tv.tv_usec/1000000.0+tv.tv_sec;
  //videostampmin = stamps[img];
  // perhaps the following will give more synchronization....
  videostampmin = videostampmax - (videostampmax-stamps[img])/nbufs;
  if(n>1 && debug) 
    fprintf(stderr,"%d wasted images\n",n-1);
  return map + gb_buffers.offsets[img];
}


static int fmt_available(video_fmt f) {
  int format = fmts_xaw_to_v4l[f];
  if(!format) return 0;
  v4l_pict.palette = format;
  v4l_pict.depth = device_pal_depth[format];
  return (-1 != ioctl (bttvfd, VIDIOCSPICT, &v4l_pict));
}

/* ---------------------------------------------------------------------- */

static int
grab_tune (unsigned long freq)
{
  if (debug)
    fprintf (stderr, "v4l: freq: %.3f\n", (float) freq / 16);

  if (! (driver.capabilities & DEV_CAP_TUNER))
    return 0;
  
  if (-1 == ioctl (bttvfd, VIDIOCSFREQ, &freq))
    perror ("ioctl VIDIOCSFREQ");
  
  return 0;
}

static int
grab_tuned ()
{
#if 0
  /* (quick & dirty -- grabbing works with hsync only) */
  return grab_one (-1, 64, 48) ? 1 : 0;
#else
  if (-1 == ioctl (bttvfd, VIDIOCGTUNER, &v4l_tuner))
    {
      perror ("ioctl VIDIOCGTUNER");
      return 0;
    }
  return v4l_tuner.signal ? 1 : 0;
#endif
}

static int
grab_input (int input, int norm)
{
  if (-1 != input)
    {
      if (debug)
        fprintf (stderr, "v4l: input: %d\n", input);
      cur_input = input;
    }
  if (-1 != norm)
    {
      if (debug)
        fprintf (stderr, "v4l: norm : %d\n", norm);
      cur_norm = norm;
    }

  v4l_channels[cur_input].norm = cur_norm;

  if (-1 == ioctl (bttvfd, VIDIOCSCHAN, &v4l_channels[cur_input]))
    perror ("ioctl VIDIOCSCHAN");
  /* bug of the compatibility V4L2 -> V4L1 (with bttv 0.9.11)*/
  {
    struct v4l2_capability cap2;
    int sid = 0;
    if (ioctl(bttvfd,VIDIOC_QUERYCAP,&cap2) != -1) {
      switch (v4l_channels[cur_input].norm) {
      case VIDEO_MODE_PAL:
	sid = V4L2_STD_PAL;
	break;
      case VIDEO_MODE_NTSC:
	sid = V4L2_STD_NTSC;
	break;
      case VIDEO_MODE_SECAM:
	sid = V4L2_STD_SECAM;
	break;
      }
      if(ioctl(bttvfd,VIDIOC_S_STD, &sid) == -1)
	perror("ioctl VIDIOC_S_STD");
    }
  }
  return 0;
}

static int
grab_picture (int color, int bright, int hue, int contrast)
{
  if (-1 == ioctl (bttvfd, VIDIOCGPICT, &v4l_pict))
    perror ("ioctl VIDIOCGPICT");
  
  if (color != -1)
    v4l_pict.colour = color;
  if (contrast != -1)
    v4l_pict.contrast = contrast;
  if (bright != -1)
    v4l_pict.brightness = bright;
  if (hue != -1)
    v4l_pict.hue = hue;

  if (-1 == ioctl (bttvfd, VIDIOCSPICT, &v4l_pict))
        perror ("ioctl VIDIOCSPICT");
  return 0;
}

static int
grab_audio (int mute, int volume, int *mode)
{
  if (mute != -1)
    {
      if (mute)
        v4l_audios[0].flags |= VIDEO_AUDIO_MUTE;
      else
        v4l_audios[0].flags &= ~VIDEO_AUDIO_MUTE;
    }
  if (volume != -1)
    v4l_audios[0].volume = volume;

  v4l_audios[0].audio = cur_input;
  v4l_audios[0].mode = mode ? *mode : 0;

  if (-1 == ioctl (bttvfd, VIDIOCSAUDIO, &v4l_audios[0]))
    perror ("ioctl VIDIOCSAUDIO");
  
  if (mode)
    {
      if (-1 == ioctl (bttvfd, VIDIOCGAUDIO, &v4l_audios[0]))
	perror ("ioctl VIDIOCGAUDIO");
      *mode = v4l_audios[0].mode & (VIDEO_SOUND_MONO|VIDEO_SOUND_STEREO);
    }
  
  return 0;
}

static int is525(int norm) {
  return norm == VIDEO_MODE_NTSC || norm==4 || norm==6;
}

static int issecam(int norm) {
  return norm == VIDEO_MODE_SECAM;
}

static void* get_buf(int i){
  return map + gb_buffers.offsets[i];
}

struct GRABBER grab_v4l = {
  "video4linux",
  norms, NULL,
  grab_open,
  grab_close,
  grab_overlay,
  fmt_available,
  get_img,
  grab_tune,
  grab_tuned,
  grab_input,
  grab_picture,
  grab_audio,
  is525,
  issecam,
  get_buf
};

#endif
