/*****************************************************************************
 * $Id: mixer-alsa.c,v 1.2 2004/09/19 15:55:49 alainjj Exp $
 * Program under GNU General Public License (see ../COPYING)
 *****************************************************************************/

#include "config.h"
#ifdef HAVE_ALSA
#include <stdio.h>
#ifdef HAVE_ALSA
#  if defined(HAVE_ALSA_ASOUNDLIB_H)
#    include <alsa/asoundlib.h>
#  elif defined (HAVE_SYS_ASOUNDLIB_H)
#    include <sys/asoundlib.h>
#  endif
#endif // HAVE_ALSA
#include "mixer.h"
extern int debug;

#define NB_DEVICES 6
static char *chans_name[]={"Line", "PCM", "Capture", "Wave Surround", 
			   "SB Live Analog/Digital Output Jack", "Master"};
static int sav_muted[NB_DEVICES];
static long sav_volume[NB_DEVICES];
static long volume[NB_DEVICES];
snd_mixer_elem_t *elem[NB_DEVICES];
static snd_mixer_t *handle;
static int oldcap;

static inline long percent_to_alsa(int val, long pmin, long pmax) {
  return pmin + ((pmax - pmin) * val + 50) / 100;  
}
static inline int alsa_to_percent(long val, long pmin, long pmax) {
  long diff = pmax - pmin;
  if(diff == 0) return 50;
  return ((val - pmin) * 100 + (diff / 2)) / diff;
}

static int alsa_open(void) {
  int j, err;
  static char *devname="default";
  if(mixer_tvchan) chans_name[MIXER_CHAN_TV]=mixer_tvchan;
  if(mixer_pcmchan) chans_name[MIXER_CHAN_PCM]=mixer_pcmchan;
  if(mixer_dev) devname=mixer_dev;
  if((err = snd_mixer_open( &handle, 0 )) < 0) {
    fprintf(stderr,"snd_mixer_open: %s\n", snd_strerror(err));
    return -1;
  }
  if((err = snd_mixer_attach( handle, devname )) < 0) {
    fprintf(stderr,"snd_mixer_open: %s\n", snd_strerror(err));
    snd_mixer_close(handle);
    return -1;
  }
  if((err = snd_mixer_selem_register( handle, 0, 0 )) < 0) {
    fprintf(stderr,"snd_mixer_open: %s\n", snd_strerror(err));
    snd_mixer_close(handle);
    return -1;
  }
  if((err = snd_mixer_load( handle )) < 0) {
    fprintf(stderr,"snd_mixer_open: %s\n", snd_strerror(err));
    snd_mixer_close(handle);
    return -1;
  }
  for(j = 0; j < NB_DEVICES; j++) {
    long v,pmin,pmax;
    snd_mixer_selem_id_t *sid;
    snd_mixer_selem_id_alloca(&sid);
    snd_mixer_selem_id_set_index(sid,0);
    snd_mixer_selem_id_set_name(sid, chans_name[j]);
    elem[j] = snd_mixer_find_selem(handle, sid);
    if(elem[j] == NULL) continue;
    if(j == MIXER_CHAN_CAPTURE) {
      snd_mixer_selem_get_capture_volume_range(elem[j],&pmin,&pmax);
      snd_mixer_selem_get_capture_volume(elem[j], 0, &v);
    } else {
      snd_mixer_selem_get_playback_switch(elem[j],0,&sav_muted[j]);
      snd_mixer_selem_set_playback_switch_all(elem[j], 1);
      snd_mixer_selem_get_playback_volume_range(elem[j],&pmin,&pmax);
      snd_mixer_selem_get_playback_volume(elem[j], 0, &v);
      sav_volume[j]=v;
    }
    volume[j] = alsa_to_percent(v, pmin, pmax);
    if(volume[j] == 0) {
      volume[j] = (j == MIXER_CHAN_CAPTURE) ? 10 : 68;
      v = percent_to_alsa(volume[j], pmin, pmax);
      if(j == MIXER_CHAN_CAPTURE)
        snd_mixer_selem_set_capture_volume_all(elem[j], v);
      else
        snd_mixer_selem_set_playback_volume_all(elem[j], v);
    }
    if(debug)
      fprintf(stderr,"ALSAMIXER: %-8s=%3ld (%3ld%%) min=%3ld max=%3ld\n",
              chans_name[j], v, volume[j], pmin, pmax);
  }
  snd_mixer_selem_get_capture_switch(elem[MIXER_CHAN_TV],0,&oldcap);
  return 0;
}

static void alsa_close(void) {
  if(restoresnd) {
    int j;
    for(j = 0; j < NB_DEVICES; j++) {
      if(elem[j] == NULL) continue;
      if(j != MIXER_CHAN_CAPTURE) {
	snd_mixer_selem_set_playback_switch_all(elem[j], sav_muted[j]);
	snd_mixer_selem_set_playback_volume_all(elem[j], sav_volume[j]);
	if(debug)
	  fprintf(stderr,"ALSAMIXER RESTORE: %-8s level:%3ld muted:%3d\n",
		  chans_name[j], sav_volume[j], sav_muted[j]);
      }
    }
  }
  snd_mixer_close(handle);
}

static void alsa_mute (int chan, int muted) { 
  if(NULL == elem[chan]) return;
  snd_mixer_selem_set_playback_switch_all(elem[chan], !muted);
}

static int alsa_get_volume (int chan) {
  if(NULL == elem[chan]) return 0;
  return volume[chan];
}

static void alsa_set_volume (int chan, int val) {
  long v, pmin, pmax;
  if(NULL == elem[chan]) return;
  snd_mixer_selem_get_playback_volume_range(elem[chan],&pmin,&pmax);
  v = percent_to_alsa(val, pmin, pmax);
  if(debug)
    fprintf(stderr,"chan=%d %ld %ld %ld\n",chan,v,pmin,pmax);
  snd_mixer_selem_set_playback_volume_all(elem[chan], v);
  volume[chan] = val;
}

static void alsa_capture(void) {
  if(NULL == elem[MIXER_CHAN_TV]) return;
  snd_mixer_selem_set_capture_switch_all(elem[MIXER_CHAN_TV], 1);
}

static void alsa_uncapture (void) {
  if(NULL == elem[MIXER_CHAN_TV]) return;
  snd_mixer_selem_set_capture_switch_all(elem[MIXER_CHAN_TV], oldcap);
}

struct MIXER mixer_alsa = {
  "alsa",
  alsa_open,
  alsa_close,
  alsa_mute,
  alsa_get_volume,
  alsa_set_volume,
  alsa_capture,
  alsa_uncapture
};
#endif
