/*****************************************************************************
 * xdtv_scantv.c: check available channels
 *****************************************************************************
 * $Id: xdtv_scantv.c,v 1.2 2004/09/24 21:08:29 pingus77 Exp $
 *****************************************************************************
 * Copyright (C) 2003 Alain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * Original code:
 *
 *  (c) Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "config.h"

int debug = 0;
/* used in colorspace.c */
void *fast_memcpy(void *dest, const void *src, size_t n){
  return memcpy(dest,src,n);
}

#ifdef HAVE_V4L
#include "grab.h"
#include "channels.h"
#include "strtab.h"

#ifdef HAVE_ZVBI
#include <libzvbi.h>
#endif


extern struct GRABBER grab_v4l,grab_v4l2;
static struct GRABBER *grab = NULL;
int nbufs_default = 3, owidth, oheight,img;
int x11_bigendian = 0, x11_byteswap = 0, have_xv, nbufs, vbifd;
double videostampmin,videostampmax;
video_fmt x11_pixmap_format = 0, grab_format;
int get_ybar(int h) {return 0;}

struct device_t  device =
#ifdef HAVE_DEVFS
{"/dev/v4l/video0", "/dev/v4l/vbi0"};
#else
{"/dev/video0", "/dev/vbi0"};
#endif
extern struct STRTAB chan_names[];

char *tvname=NULL;
char *confname="xdtvrc.new";

#ifdef HAVE_ZVBI
extern int vbifd;

static void
event(struct vbi_event *ev, void *user)
{
  switch (ev->type) {
  case VBI_EVENT_NETWORK:
    if (NULL != tvname)
      free(tvname);
    if (strlen(ev->ev.network.name) > 0)
      tvname = strdup(ev->ev.network.name);
    break;
  }
}
struct vbi_capture *vbi_cap;
struct vbi_decoder *vbi_dec;
void * vbi_data;
int vbi_lines;
vbi_sliced *vbi_sli;
#endif

void gettvname(char *channelname, int freq) {
  char tvname_unknown[100];
#ifdef HAVE_ZVBI
  double   vbi_timestamp;
  struct timeval vbi_timeout, initial_time, current_time;
#endif
  
  tvname=NULL;
#ifdef HAVE_ZVBI
  gettimeofday (&initial_time, NULL);
  for(;;) {
    int ret;
    long delay;
    gettimeofday (&current_time, NULL);
    delay=5000000-((1000000*current_time.tv_sec+current_time.tv_usec)-
		   (1000000*initial_time.tv_sec+initial_time.tv_usec));
    if(delay<0) break;
    vbi_timeout.tv_sec=delay/1000000;
    vbi_timeout.tv_usec=delay%1000000;
    fprintf(stderr,"channel=%s freq=%.3f timeout=%.1f  \r",channelname,
	    (float)freq/16,(float)delay/1000000);
    fflush(stderr);
    ret=vbi_capture_read(vbi_cap, vbi_data, vbi_sli,
			 &vbi_lines, &vbi_timestamp, &vbi_timeout);
    if(ret==0) break;
    if(ret<0) {
      if(errno==EIO) {usleep(100000);continue;}
      perror("\nvbi_capture_read");
      break;
    }
    vbi_decode(vbi_dec, vbi_sli, vbi_lines, vbi_timestamp);
    if(tvname!=NULL) break;
  }
#endif
  if(tvname==NULL) 
    {
      if(channelname!=NULL)
	sprintf(tvname_unknown,"unknown (%s)",channelname);
      else 
	sprintf(tvname_unknown,"unknown (%.3f MHz)",(float) freq / 16);
      tvname = strdup(tvname_unknown);
    }
}


/* thanks to xawtv ! */
static int
menu(char *intro, struct STRTAB *tab, char *opt)
{
    int i,ret;
    char line[80];

    if (NULL != opt) {
	/* cmd line option -- non-interactive mode */
	for (i = 0; tab[i].str != NULL; i++)
	    if (0 == strcasecmp(tab[i].str,opt))
		return tab[i].nr;
	fprintf(stderr,"%s: not found\n",opt);
	exit(1);
    }

    fprintf(stderr,"\n%s\n",intro);
    for (i = 0; tab[i].str != NULL; i++)
	fprintf(stderr,"  %2d: %s\n",tab[i].nr,tab[i].str);

    for (;;) {
	fprintf(stderr,"nr ? ");
	fgets(line,79,stdin);
	ret = atoi(line);
	for (i = 0; tab[i].str != NULL; i++)
	    if (ret == tab[i].nr)
		return ret;
	fprintf(stderr,"invalid choice\n");
    }
}

void
display_help_message (FILE *fp)
{
  fprintf(fp,
	"\nXdTV scantv options:\n\n"
	    "   -h           print this text.\n"
	    "   -o conffile  set output file.\n"
	    "   -n norm      set tv norm.\n"
	    "   -f table     set frequency table.\n"
	    "   -c videodev  set video device file.\n"
	    "   -C vbidev    set vbi device file.\n"
	    "   -v4l1        force v4l1 mode.\n"
	    "   -v4l2        force v4l2 mode.\n"
	    "   -v           debug mode.\n"
	    "   -a           full scan (all frequencies, not just\n"
	    "                the ones from the frequency table)\n\n");
}

#define INCRARG(i) if(++i>=argc) \
{display_help_message(stderr); exit(0);} arg=argv[i];

int main(int argc, char **argv)
{
  int i;
  unsigned long freq;
  char *tvnorm=NULL,*freqtab=NULL;
  int tvnorm_index,freqtab_index;
  FILE *conf;
  int fullscan=0;
#ifdef HAVE_ZVBI
  struct vbi_raw_decoder *vbi_par;
  int services;
  char *err;
#endif
  int forcev4l1=0,forcev4l2=0;

  for (i = 1; i < argc; i++) 
    {
      char *arg=argv[i];
      if (strcmp (arg, "-c") == 0)
	{
	  INCRARG(i);
	  device.video=arg;
	}
      else if(strcmp (arg, "-C") == 0)
	{
	  INCRARG(i);
	  device.vbi=arg;	
	}
      else if(strcmp (arg, "-a") == 0)
	  fullscan=1;
      else if(strcmp (arg, "-v") == 0)
	{
	  INCRARG(i);
	  debug=atoi(arg);
	}
      else if(strcmp (arg, "-o") == 0)
	{
	  INCRARG(i);
	  confname=arg;	
	}
      else if(strcmp (arg, "-v4l1") == 0)
	{
	  forcev4l1=1;
	}
      else if(strcmp (arg, "-v4l2") == 0)
	{
	  forcev4l2=1;
	}
      else if(strcmp (arg, "-n") == 0)
	{
	  INCRARG(i);
	  tvnorm=arg;
	}
      else if(strcmp (arg, "-f") == 0)
	{
	  INCRARG(i);
	  freqtab=arg;
	}
      else {
	display_help_message(stderr);
	exit(1);
      }
    }
  if(grab == NULL && !forcev4l1) {
    grab = &grab_v4l2;
    if(grab->grab_open(&device)==-1)
      grab = NULL;
  }
  if(grab == NULL && !forcev4l2) {
    grab = &grab_v4l;
    if(grab->grab_open(&device)==-1)
      grab = NULL;
  }
  if(grab==NULL) {
    fprintf(stderr,"cannot open video device\n");
    exit(1);
  }
      
  tvnorm_index = menu("please select your TV norm",grab->norms,tvnorm);
  freqtab_index = menu("please select a frequency table",
		       chan_names,freqtab);

  conf=fopen(confname,"w");
  if(conf==NULL) {perror("fopen:"); exit(1);}
  fprintf(conf,"freqtab = %s\n",chan_names[freqtab_index].str);
  fprintf(conf,"source = Television\n");
  fprintf(conf,"norm = %s\n\n",grab->norms[tvnorm_index].str);
  fflush(conf);

  grab->grab_input(str_to_int("Television",grab->inputs),
		   tvnorm_index);
  grab->grab_audio(0,65535,NULL);
#ifdef HAVE_ZVBI
  close(vbifd);vbifd=-1;
  services= VBI_SLICED_VBI_525 | VBI_SLICED_VBI_625
    | VBI_SLICED_TELETEXT_B | VBI_SLICED_CAPTION_525
    | VBI_SLICED_CAPTION_625 | VBI_SLICED_VPS
    | VBI_SLICED_WSS_625 | VBI_SLICED_WSS_CPR1204;
  vbi_cap=vbi_capture_v4l_new(device.vbi,16,&services,-1,&err,0);
  if(vbi_cap==NULL)
    {
      fprintf(stderr,"Unable to open vbi device\n");
      exit(1);
    }
  vbi_dec=vbi_decoder_new();
  vbi_event_handler_add(vbi_dec,~0,event,NULL);
  vbi_par=vbi_capture_parameters(vbi_cap);
  vbi_lines=(vbi_par->count[0] + vbi_par->count[1]);
  vbi_data=malloc(vbi_lines *vbi_par->bytes_per_line);
  vbi_sli = malloc(vbi_lines * sizeof(vbi_sliced));
#endif
  if(!fullscan)
    for(i=0;i<CHAN_ENTRIES;i++) {
      freq=tvtuner[i].freq[freqtab_index];
      if(freq) {
	fprintf(stderr,"%d/%d %s %.3lf   \r",i,CHAN_ENTRIES,tvtuner[i].name,(float)freq/1000);
	fflush(stderr);
	grab->grab_tune(freq * 16 / 1000);
	usleep(200000);
	if(grab->grab_tuned()) {
	  gettvname(tvtuner[i].name,freq*16/1000);
	  printf("channel=%s freq=%.3f tvname=%s        \n",
		 tvtuner[i].name,(float)freq/1000,tvname);
	  fprintf(conf,"[%s]\nchannel = %s\n\n",tvname,tvtuner[i].name);
	  fflush(conf);
	}
      }
    } 
  else
    {
      int f,min_freq=44,max_freq=958;
      char *channel;
      for (f = min_freq*16; f <= max_freq*16; f += 4) 
	{
	  grab->grab_tune(f);
	  channel=NULL;
	  for(i=0;i<CHAN_ENTRIES;i++)
	    if(f*1000==tvtuner[i].freq[freqtab_index]*16) 
	      {channel=tvtuner[i].name;break;}
	  fprintf(stderr,"freq=%.3f (min=%d max=%d) channel=%s     \r",
		  (float) f / 16,min_freq,max_freq,channel);
	  usleep(200000);
	  if(grab->grab_tuned()) {
	    gettvname(channel,f);
	    printf("freq=%.3f channel=%s tvname=%s        \n",
		   (float)f/16,channel,tvname);
	    if(channel==NULL)
	      fprintf(conf,"[%s]\nfreq = %.3f\n\n",tvname,(float) f / 16);
	    else
	      fprintf(conf,"[%s]\nchannel = %s\n\n",tvname,channel);
	    fflush(conf);
	  }
	}
    }
  
#ifdef HAVE_ZVBI
  vbi_capture_delete(vbi_cap);
  vbi_decoder_delete(vbi_dec);
  free(vbi_sli);
  free(vbi_data);
  if(tvname!=NULL) free(tvname);
#endif

  grab->grab_close();
  fclose(conf);
  return 0;
}

/* used in grab-v4l2.c..*/
void video_overlay
(int (*set_overlay)(int,int,int,int,int,struct OVERLAY_CLIP *,int)) {
}
#else
int main(int argc, char **argv)
{
  fprintf(stderr,"xdtv compiled without V4L !\n");
  exit(1);
}
#endif

